"""Data structures used to manipulate the model locally within a data source."""
from .util import normalize_mac
from collections import namedtuple
from ipaddress import ip_network
from preseem_protobuf.model import service_pb2


def _validate_ip(value, field_name='ip_prefix'):
    """Check an IP prefix string value and return a normalized IP string."""
    net = ip_network(value, strict=False)
    if net.prefixlen == 0:
        raise ValueError(f"The '{field_name}' network cannot have prefixlen of 0")
    if net.prefixlen == net.max_prefixlen:
        return str(net.network_address)
    return str(net)


def _validate_mac(value, field_name='mac_address'):
    """Check a MAC address string value and return a normalized MAC address."""
    if value is None:
        raise ValueError(f"The '{field_name}' parameter must be non-null.")
    if not isinstance(value, str):
        raise TypeError(f"The '{field_name}' parameter must be of type string.")
    if value == '':
        raise ValueError(f"The '{field_name}' parameter must be non-empty.")
    mac = normalize_mac(value)
    if mac == '00:00:00:00:00:00':
        raise ValueError(f"The '{field_name}' parameter must not be the zero mac.")
    return mac


class Errorable:
    """A class to handle errors in a flexible way.  If error_parent is assigned to
       another Errorable class, the error will be set on the parent."""
    __slots__ = ["_errors", "error_parent"]
    Error = namedtuple('Error', ('field', 'value', 'exc_type', 'message'))
    Error.__repr__ = lambda x: f"Error setting '{x.field}' to '{x.value}': {x.message}"

    def __init__(self):
        self._errors = []
        self.error_parent = None  # set this to have errors set on a parent object.

    @property
    def errors(self):
        return self._errors

    def err(self, field, value, exc=None):
        if self.error_parent:
            self.error_parent.err(field, value, exc)
        else:
            self._errors.append(
                self.Error(field, value,
                           type(exc).__name__ if exc else None, str(exc)))


class Repeated(Errorable):
    """Class to make a "repeated" protobuf attribute pythonic."""

    def __init__(self, name, pb_field, cleaner=None):
        """Initialize a repeated field named pf_field in the protobuf, with an optional
           cleaner function that checks/cleans the data and returns a clean copy."""
        super().__init__()
        self.name = name
        self.pb_field = pb_field
        self.cleaner = cleaner

    def _clean(self, value):
        if self.cleaner:
            try:
                value = self.cleaner(value)
            except (TypeError, ValueError) as err:
                self.err(self.name, value, err)
                return None
        return value

    def __iter__(self):
        return iter(self.pb_field)

    def __len__(self):
        return len(self.pb_field)

    def __getitem__(self, i):
        return self.pb_field[i]

    def __setitem__(self, i, value):
        value = self._clean(value)
        if value is not None:
            self.pb_field[i] = value

    def __delitem__(self, i):
        del self.pb_field[i]

    def __reversed__(self):
        return reversed(self.pb_field)

    def append(self, value):
        value = self._clean(value)
        if value is not None:
            self.pb_field.append(value)

    def insert(self, i, value):
        value = self._clean(value)
        if value is not None:
            self.pb_field.insert(i, value)

    def extend(self, values):
        if self.cleaner:
            values = [self._clean(value) for value in values]
        self.pb_field.extend([x for x in values if x is not None])

    def __repr__(self):
        return f'{self.pb_field}'


class Attachment(Errorable):
    """A Service Attachment object that can be easily manimulated in Python."""
    __slots__ = ["pb", "_mac_addresses", "_network_prefixes"]

    def __init__(self, pb=None, **kwargs):
        super().__init__()
        self.pb = pb if pb else service_pb2.Attachment()
        self._mac_addresses = Repeated('mac_addresses', self.pb.mac_addresses,
                                       _validate_mac)
        self._mac_addresses.error_parent = self
        self._network_prefixes = Repeated('network_prefixes', self.pb.network_prefixes,
                                          _validate_ip)
        self._network_prefixes.error_parent = self
        for key, value in kwargs.items():
            setattr(self, key, value)

    @property
    def cpe_mac_address(self):
        return self.pb.cpe_mac_address if self.pb.HasField('cpe_mac_address') else None

    @cpe_mac_address.setter
    def cpe_mac_address(self, value):
        if value is not None:
            try:
                self.pb.cpe_mac_address = _validate_mac(value, 'cpe_mac_address')
            except (TypeError, ValueError) as err:
                self.err('cpe_mac_address', value, err)
        else:
            self.pb.ClearField('cpe_mac_address')

    @property
    def imsi(self):
        return self.pb.imsi if self.pb.HasField('imsi') else None

    @imsi.setter
    def imsi(self, value):
        if value is not None:
            if not isinstance(value, str):
                self.err('imsi', value,
                         TypeError("The 'imsi' parameter must be of type string."))
                return
            if value == "":
                self.err('imsi', value,
                         TypeError("The 'imsi' parameter must be non-empty."))
                return
            self.pb.imsi = str(value)
        else:
            self.pb.ClearField('imsi')

    @property
    def cpe_serial_number(self):
        return self.pb.cpe_serial_number if self.pb.HasField(
            'cpe_serial_number') else None

    @cpe_serial_number.setter
    def cpe_serial_number(self, value):
        if value is not None:
            if not isinstance(value, str):
                self.err(
                    'cpe_serial_number', value,
                    TypeError(
                        "The 'cpe_serial_number' parameter must be of type string."))
                return
            if value == "":
                self.err(
                    'cpe_serial_number', value,
                    TypeError("The 'cpe_serial_number' parameter must be non-empty."))
                return
            self.pb.cpe_serial_number = str(value)
        else:
            self.pb.ClearField('cpe_serial_number')

    @property
    def cpe_name(self):
        return self.pb.cpe_name if self.pb.HasField('cpe_name') else None

    @cpe_name.setter
    def cpe_name(self, value):
        if value is not None:
            if not isinstance(value, str):
                self.err('cpe_name', value,
                         TypeError("The 'cpe_name' parameter must be of type string."))
                return
            if value == "":
                self.err('cpe_name', value,
                         TypeError("The 'cpe_name' parameter must be non-empty."))
                return
            self.pb.cpe_name = str(value)
        else:
            self.pb.ClearField('cpe_name')

    @property
    def device_mac_address(self):
        return self.pb.device_mac_address if self.pb.HasField(
            'device_mac_address') else None

    @device_mac_address.setter
    def device_mac_address(self, value):
        if value is not None:
            try:
                self.pb.device_mac_address = _validate_mac(value, 'device_mac_address')
            except (TypeError, ValueError) as err:
                self.err('device_mac_address', value, err)
        else:
            self.pb.ClearField('device_mac_address')

    @property
    def username(self):
        return self.pb.username if self.pb.HasField('username') else None

    @username.setter
    def username(self, value):
        if value is not None:
            if not isinstance(value, str):
                self.err('username', value,
                         TypeError("The 'username' parameter must be of type string."))
                return
            if value == "":
                self.err('username', value,
                         TypeError("The 'username' parameter must be non-empty."))
                return
            self.pb.username = value
        else:
            self.pb.ClearField('username')

    @property
    def mac_addresses(self):
        return self._mac_addresses

    @property
    def network_prefixes(self):
        return self._network_prefixes

    def __repr__(self):
        field_values = [
            f"{field.name}={getattr(self, field.name)!r}"
            for field in self.pb.DESCRIPTOR.fields if hasattr(self, field.name)
        ]
        return f"{type(self).__name__}({', '.join(field_values)})"

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            return self.pb == other.pb
        return False


class Service(Errorable):
    """Service class to use within a bss/oss data source."""
    __slots__ = [
        "_id", "_account_id", "_package_id", "_download_rate", "_upload_rate",
        "_attachment", "_subscriber_identifier", "_url", "_account_uuid",
        "_package_uuid"
    ]
    # this is just used to list fields we want to print with __repr__
    attrs = ('id', 'account_id', 'package_id', 'download_rate', 'upload_rate',
             'subscriber_identifier', 'url', 'attachment')

    def __init__(self, id, **kwargs):
        super().__init__()
        if id is None or str(id) == '':
            raise ValueError("The 'id' parameter must be non-None.")
        self._id = str(id)
        self._account_id = None
        self._package_id = None
        self._download_rate = None
        self._upload_rate = None
        self._attachment = None
        self._subscriber_identifier = None
        self._url = None
        self._account_uuid = None  # used by sync_preseem_model for resolution
        self._package_uuid = None  # used by sync_preseem_model for resolution
        if 'attachment' not in kwargs:
            self._attachment = Attachment()  # for convenience; we usually want one
            self._attachment.error_parent = self
        for key, value in kwargs.items():
            setattr(self, key, value)

    @property
    def id(self):
        return self._id

    @property
    def account_id(self):
        """Get the ID of the account.  The id is immutable for the Service object."""
        return self._account_id

    @account_id.setter
    def account_id(self, value):
        if value == "":
            self.err('account_id', value,
                     ValueError("The 'account_id' parameter must be non-empty."))
            return
        self._account_id = str(value) if value is not None else None

    @property
    def package_id(self):
        return self._package_id

    @package_id.setter
    def package_id(self, value):
        if value == "":
            self.err('package_id', value,
                     ValueError("The 'package_id' parameter must be non-empty."))
            return
        self._package_id = str(value) if value is not None else None

    @property
    def download_rate(self):
        return self._download_rate

    @download_rate.setter
    def download_rate(self, value):
        try:
            self._download_rate = int(value) if value is not None else None
        except ValueError as err:
            self.err('download_rate', value, err)

    @property
    def upload_rate(self):
        return self._upload_rate

    @upload_rate.setter
    def upload_rate(self, value):
        try:
            self._upload_rate = int(value) if value is not None else None
        except ValueError as err:
            self.err('upload_rate', value, err)

    @property
    def attachment(self):
        return self._attachment

    @attachment.setter
    def attachment(self, value):
        if value is not None and not isinstance(value, Attachment):
            raise TypeError("service attachment must be of type Attachment.")
        elif value is None:
            self._attachment = None
        else:
            self._attachment = Attachment()
            self._attachment.pb.CopyFrom(value.pb)
            self._attachment.error_parent = self

    @property
    def subscriber_identifier(self):
        return self._subscriber_identifier

    @subscriber_identifier.setter
    def subscriber_identifier(self, value):
        if value == "":
            self.err('subscriber_identifier', value,
                     "The 'subscriber_identifier' parameter must be non-empty.")
            return
        self._subscriber_identifier = str(value) if value is not None else None

    @property
    def url(self):
        return self._url

    @url.setter
    def url(self, value):
        if not isinstance(value, str):
            self.err('url', value, "The router 'url' parameter must be a string.")
            return
        if value == "":
            self.err('url', value, "The router 'url' parameter must be non-empty.")
            return
        self._url = value if value is not None else None

    def __repr__(self):
        attr_values = [(attr, getattr(self, attr)) for attr in self.attrs]
        attributes_str = ', '.join(f"{attr}={value!r}" for attr, value in attr_values)
        return f"{type(self).__name__}({attributes_str})"

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            for slot in self.__slots__:
                if getattr(self, slot) != getattr(other, slot):
                    return False
            return True
        return False

    def __hash__(self):
        return hash(self.id)
