"""Function to sync a Preseem network metadata store to the new Preseem model."""
import asyncio
from collections import namedtuple
import ipaddress
import logging
from uuid import UUID

from grpc import RpcError

from preseem import NetworkMetadataReference
from preseem.source_model import Attachment, Service
from preseem_protobuf.model.service_pb2 import SetAttachmentReq

# STM-6039 Only allow the sync to run once at a time, to prevent duplicate
# entities from being created.
sync_lock = None

# This object is used to pass element items to the sync_preseem_model
# algorithm for a given source and intent role.
ElementSyncItem = namedtuple('ElementSyncItem', ('id', 'name', 'site', 'host'))


# Roles are hard-coded here since these should never change.
class ElementRole:
    SWITCH = UUID('41f2903a-311a-448d-9d4d-86dc8ba3cb07')
    AP = UUID('ae3f1aa5-063a-4a24-9744-822a82d107d9')
    ROUTER = UUID('bf433e28-122b-41df-b49a-77ac6544deac')
    CPE_RADIO = UUID('c72115d3-daf9-43ae-9475-083d62eae74f')


_account_field_map = {
    'type': lambda x: x.type,
    'status': lambda x: x.status,
    'name': lambda x: x.name,
    'first_name': lambda x: x.first_name,
    'last_name': lambda x: x.last_name,
}

_package_field_map = {
    'name': lambda x: x.name,
    'download_rate': lambda x: x.download_rate or None,
    'upload_rate': lambda x: x.upload_rate or None,
}

_service_field_map = {
    'account_uuid': lambda x: x._account_uuid,
    'package_uuid': lambda x: x._package_uuid,
    'download_rate': lambda x: x._download_rate or None,
    'upload_rate': lambda x: x._upload_rate or None,
    'attachment': lambda x: x._attachment.pb if x._attachment else None,
    'subscriber_identifier': lambda x: x._subscriber_identifier,
}


async def call_and_log_error(co):
    """Call a GRPC coroutine and log any errors from it."""
    try:
        return await co
    except asyncio.CancelledError:
        raise
    except RpcError as err:
        logging.warning('sync_preseem_model: error calling service: %s (%s)',
                        err.code(), err.details())
    except Exception as err:
        logging.warning('sync_preseem_model: %s error calling service: %s',
                        type(err).__name__,
                        err,
                        exc_info=1)
    return None


async def _sync_objects(model, company_uuid, source, bss_objs, model_objs, createrpc,
                        updaterpc, field_map, noact):
    """
    Function to sync python bss-sourced objects to model objects.
    Pass in the model and the company_uuid and source we are working on.
    - bss_objs is a dict {source_id -> obj) where obj is a python bss object.
    - model_objs is a list of objects from the model.
    - createrpc and updaterpc are the grpc update methods for the object.
    - field_map is a dict of {model field name -> mapping func} where the mapping func
      returns the value to sync from the bss object.
    """
    new_ids = set(bss_objs)
    new_objs = []
    for mobj in model_objs:
        if mobj.source != source:
            continue  # ignore objects owned by other sources
        bobj = bss_objs.get(mobj.source_id)
        if bobj:
            changes = []
            mods = {}
            new_ids.discard(mobj.source_id)
            if mobj.inactive:
                changes.append('reactivated')
                mods['inactive'] = False
            for f, m in field_map.items():
                mval = getattr(mobj, f)
                bval = m(bobj)
                if mval != bval:
                    if f == 'attachment':
                        # find what attachment fields changed for logging purposes
                        achgs = []
                        if mval and bval:
                            for sf in bval.DESCRIPTOR.fields:
                                smval = getattr(mval, sf.name)
                                sbval = getattr(bval, sf.name)
                                if smval != sbval:
                                    achgs.append(f"{sf.name}[{smval}->{sbval}]")
                        changes.append(f'{f}: {",".join(achgs)}')
                    else:
                        changes.append(f'{f}: {mval} -> {bval}')
                    mods[f] = bval
            if mods:
                objdesc = mobj
                if isinstance(bobj, Service):
                    # more concise error message for Service
                    objdesc = f"Service {UUID(bytes=mobj.uuid)}"
                logging.info('sync_preseem_model(%s): modify %s (%s)', source, objdesc,
                             ', '.join(changes))
                if not noact:
                    if isinstance(bobj, Service) and 'attachment' in mods:
                        # special case for setting attachment
                        attachment = mods.pop('attachment')
                        set_att_req = SetAttachmentReq()
                        if attachment:
                            set_att_req.attachment.CopyFrom(attachment)
                        mods['attachment_req'] = set_att_req
                    await call_and_log_error(
                        updaterpc(company_uuid=company_uuid, uuid=mobj.uuid, **mods))
        else:
            if not mobj.inactive:
                logging.info("sync_preseem_model(%s): deactivate %s %s", source,
                             type(mobj).__name__.lower(), UUID(bytes=mobj.uuid))
                if not noact:
                    await call_and_log_error(
                        updaterpc(company_uuid=company_uuid,
                                  uuid=mobj.uuid,
                                  inactive=True))
    for obj_id in new_ids:
        bobj = bss_objs.get(obj_id)
        if bobj:
            logging.info('sync_preseem_model(%s): create %s', source, bobj)
            if not noact:
                mobj = await call_and_log_error(
                    createrpc(company_uuid=company_uuid,
                              source=source,
                              source_id=bobj.id,
                              **{
                                  f: m(bobj)
                                  for f, m in field_map.items()
                              }))
                if mobj:
                    uuid = UUID(bytes=mobj.uuid)
                    logging.info('sync_preseem_model(%s): created %s "%s"', source,
                                 type(mobj).__name__, uuid)
                    new_objs.append(mobj)
    return new_objs


async def sync_preseem_model(preseem_model,
                             company_uuid,
                             source,
                             role,
                             items,
                             packages=None,
                             accounts=None,
                             services=None,
                             nm_model=None,
                             noact=False,
                             verbose=0):
    """
    Update the Preseem model to keep its ElementDiscovery objects in sync with
    another system.  This method takes the full set of items for a specific
    company, source, and role.  It diffs this against the current model and
    applies any create, modify or delete (set inactive) operations.
    It returns a map of {id -> uuid} for the caller to use.
    """
    global sync_lock
    if sync_lock is None:
        sync_lock = asyncio.Lock()

    assert isinstance(company_uuid, bytes)
    assert isinstance(source, str)
    assert isinstance(role, UUID)

    if source == '':
        raise ValueError('source cannot be the empty string')

    async with sync_lock:
        model_accounts, model_discs, model_services, model_sites, model_packages = await asyncio.gather(
            preseem_model.Account.List(company_uuid=company_uuid),
            preseem_model.Element.ListElementDiscovery(company_uuid=company_uuid,
                                                       intent_role_uuid=role.bytes),
            preseem_model.Service.List(company_uuid=company_uuid,
                                       include_attachment=True),
            preseem_model.Site.List(company_uuid=company_uuid),
            preseem_model.Package.List(company_uuid=company_uuid))

        model_sites = {x.uuid: x for x in model_sites}
        model_discs = {x.uuid: x for x in model_discs}
        disc_ids = {(x.source, x.source_id): x for x in model_discs.values()}

        # Create a map of name -> site.  Handle multiple sites with the same
        # name by preferring active sites, then taking the first sorted uuid.
        site_names = {}
        for site in sorted(model_sites.values(), key=lambda x: x.uuid):
            cur_site = site_names.get(site.name)
            if not cur_site or (cur_site.inactive and not site.inactive):
                site_names[site.name] = site

        # Make sure all sites we reference exist and do some pre-validation.
        new_sites = set()
        item_ids = set()
        dup_ids = set()
        for item in items:
            if not isinstance(item.id, str) or item.id == '':
                raise ValueError('invalid string value for source_id')
            if not isinstance(item.name, str) or item.name == '':
                raise ValueError('invalid string value for name')
            if item.site is not None and (not isinstance(item.site, str)
                                          or item.site == ''):
                raise ValueError('invalid string value for site')
            if item.id in item_ids:
                dup_ids.add(item.id)
            else:
                item_ids.add(item.id)
            if item.site:
                site = site_names.get(item.site)
                if not site:
                    new_sites.add(item.site)
                elif site.inactive:
                    logging.info('sync_preseem_model(%s): reactivate site "%s"', source,
                                 item.site)
                    if not noact:
                        await call_and_log_error(
                            preseem_model.Site.Update(company_uuid=company_uuid,
                                                      uuid=site.uuid,
                                                      inactive=False))
        if dup_ids:
            logging.warning('sync_preseem_model(%s): duplicate IDs found: %s', source,
                            dup_ids)
            raise ValueError('duplicate IDs found')

        for site_name in new_sites:
            logging.info('sync_preseem_model(%s): create site "%s"', source, site_name)
            if not noact:
                site = site_names[site_name] = await call_and_log_error(
                    preseem_model.Site.Create(company_uuid=company_uuid,
                                              name=site_name))
                if site:
                    model_sites[site.uuid] = site_names[site_name] = site

        unref_discs = set([x[0] for x in model_discs.items() if x[1].source == source])
        uuid_map = {}
        for item in items:
            host = str(ipaddress.ip_address(item.host))  # validate ip address
            key = (source, item.id)
            disc = disc_ids.get(key)
            if disc:
                uuid_map[item.id] = UUID(bytes=disc.uuid)
                unref_discs.remove(disc.uuid)
                site = model_sites.get(disc.site_uuid) if disc.site_uuid else None
                changes = []
                mods = {}
                if disc.inactive:
                    changes.append('reactivated')
                    mods['inactive'] = False
                    mods['element_uuid'] = None  # STM-9823
                if disc.name != item.name:
                    changes.append(f'name: {disc.name} -> {item.name}')
                    mods['name'] = item.name
                if disc.management_ip != host:
                    changes.append(f'management_ip: {disc.management_ip} -> {host}')
                    mods['management_ip'] = host
                if site.name != item.site:
                    changes.append(f'site: {site.name} -> {item.site}')
                    mods['site_uuid'] = site_names[item.site].uuid
                if mods:
                    logging.info('sync_preseem_model(%s): modify %s (%s)', source, item,
                                 ','.join(changes))
                    if not noact:
                        if nm_model and 'reactivated' in changes and disc.element_uuid:
                            # STM-9823 as part of clearing the element_uuid,
                            # we also need to delete the elem_discovery ref.
                            ref = NetworkMetadataReference('elem_discovery',
                                                           str(UUID(bytes=disc.uuid)),
                                                           {})
                            logging.info(' delete reference: %s', ref)
                            await nm_model.del_ref(ref)
                        await call_and_log_error(
                            preseem_model.Element.UpdateElementDiscovery(
                                company_uuid=company_uuid, uuid=disc.uuid, **mods))

            else:
                logging.info('sync_preseem_model(%s): create %s', source, item)
                if not noact:
                    site_uuid = site_names[item.site].uuid
                    disc = await call_and_log_error(
                        preseem_model.Element.CreateElementDiscovery(
                            company_uuid=company_uuid,
                            name=item.name,
                            management_ip=item.host,
                            site_uuid=site_uuid,
                            source=source,
                            source_id=item.id,
                            intent_role_uuid=role.bytes))
                    if disc:
                        uuid = UUID(bytes=disc.uuid)
                        logging.info(
                            'sync_preseem_model(%s): created ElementDiscovery "%s"',
                            source, uuid)
                        uuid_map[item.id] = uuid
        for disc_uuid in unref_discs:
            disc = model_discs.get(disc_uuid)
            if not disc.inactive:
                logging.info("sync_preseem_model(%s): deactivate %s", source,
                             UUID(bytes=disc_uuid))
                if not noact:
                    await call_and_log_error(
                        preseem_model.Element.UpdateElementDiscovery(
                            company_uuid=company_uuid, uuid=disc_uuid, inactive=True))
        new_packages = []
        if packages is not None:
            assert isinstance(packages, dict)
            for pkg in packages.values():
                if not isinstance(pkg.id, str) or pkg.id == '':
                    raise ValueError('invalid string value for package source_id')
                if not isinstance(pkg.name, str) or pkg.name == '':
                    raise ValueError('invalid string value for package name')
                if pkg.download_rate and (not isinstance(pkg.download_rate, int)
                                          or pkg.download_rate < 0):
                    raise ValueError('invalid value for package down rate')
                if pkg.upload_rate and (not isinstance(pkg.upload_rate, int)
                                        or pkg.upload_rate < 0):
                    raise ValueError('invalid value for package up rate')
            new_packages = await _sync_objects(preseem_model,
                                               company_uuid,
                                               source,
                                               packages,
                                               model_packages,
                                               preseem_model.Package.Create,
                                               preseem_model.Package.Update,
                                               _package_field_map,
                                               noact=noact)

        new_accounts = []
        if accounts is not None:
            assert isinstance(accounts, dict)
            for account in accounts.values():
                if not isinstance(account.id, str) or account.id == '':
                    raise ValueError('invalid string value for account source_id')
                if account.type is not None and (not isinstance(account.type, int)
                                                 or account.type <= 0):
                    raise ValueError('invalid value for account type')
                if account.status is None or not isinstance(account.status,
                                                            int) or account.status <= 0:
                    raise ValueError('invalid value for account status')
                if account.name is not None and (not isinstance(account.name, str)
                                                 or account.name == ''):
                    raise ValueError('invalid string value for account name')
                if account.first_name is not None and (not isinstance(
                        account.first_name, str) or account.first_name == ''):
                    raise ValueError('invalid string value for account first_name')
                if account.last_name is not None and (not isinstance(
                        account.last_name, str) or account.last_name == ''):
                    raise ValueError('invalid string value for account last_name')
            new_accounts = await _sync_objects(preseem_model,
                                               company_uuid,
                                               source,
                                               accounts,
                                               model_accounts,
                                               preseem_model.Account.Create,
                                               preseem_model.Account.Update,
                                               _account_field_map,
                                               noact=noact)

        if services is not None:
            assert isinstance(services, dict)
            # Setup maps to lookup account and package UUIDs from the source IDs.
            # We have to handle cases where there are multiple matching objects.
            accounts_by_id = {}
            for account in model_accounts + new_accounts:
                al = accounts_by_id.get(account.source_id)
                if al is None:
                    accounts_by_id[account.source_id] = account
                elif isinstance(al, list):
                    al.append(account)
                else:
                    accounts_by_id[account.source_id] = [al, account]
            packages_by_id = {}
            for package in model_packages + new_packages:
                pl = packages_by_id.get(package.source_id)
                if pl is None:
                    packages_by_id[package.source_id] = package
                elif isinstance(pl, list):
                    pl.append(package)
                else:
                    packages_by_id[package.source_id] = [pl, package]

            # type/value checks are done in the source_model class
            for service in services.values():
                assert isinstance(service, Service)
                account = accounts_by_id.get(
                    service.account_id) if service.account_id else None
                if account:
                    if isinstance(account, list):  # multiple matching accounts
                        acc = next((x for x in account if x.source == source), None)
                        if acc:
                            account = acc  # use account with matching source
                        else:
                            account = None  # multiple non-source matches; raise error
                            service.err(
                                'account_id', service.account_id,
                                ValueError("Multiple accounts match the account ID"))
                    if account:
                        service._account_uuid = account.uuid
                package = packages_by_id.get(
                    service.package_id) if service.package_id else None
                if package:
                    if isinstance(package, list):  # multiple matching packages
                        pkg = next((x for x in package if x.source == source), None)
                        if pkg:
                            package = pkg  # use package with matching source
                        else:
                            package = None  # multiple non-source matches; raise error
                            service.err(
                                'package_id', service.package_id,
                                ValueError("Multiple packages match the package ID"))
                    if package:
                        service._package_uuid = package.uuid
            await _sync_objects(preseem_model,
                                company_uuid,
                                source,
                                services,
                                model_services,
                                preseem_model.Service.Create,
                                preseem_model.Service.Update,
                                _service_field_map,
                                noact=noact)

        return uuid_map
