"""Test NetworkElementUpdate framework"""
import unittest
from hashlib import sha256
from uuid import uuid4

from preseem_protobuf.network_poller import network_poller_pb2
from preseem import network_element_update

class TestNetworkElementUpdate(unittest.TestCase):
    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_01(self):
        """Try setting and getting the fields."""
        neu = network_element_update.NetworkElementUpdate()

        ts = 1612927766
        neu.time = ts
        self.assertEqual(ts, neu.time)

        company_uuid = uuid4().bytes
        neu.company_uuid = company_uuid
        self.assertEqual(company_uuid, neu.company_uuid)

        element_uuid = uuid4().bytes
        neu.element_uuid = element_uuid
        self.assertEqual(element_uuid, neu.element_uuid)

        instance = 'MyInstance'
        neu.instance = instance
        self.assertEqual(instance, neu.instance)

        neu.active = True
        self.assertEqual(neu.active, True)

        neu.pingable = True
        self.assertEqual(neu.pingable, True)

        self.assertEqual(neu.ping_time, None)
        neu.ping_time = 120
        self.assertEqual(neu.ping_time, 120)
        neu.ping_time = None
        self.assertEqual(neu.ping_time, None)

        neu.snmp_version = network_poller_pb2.SnmpVersion.V2C
        self.assertEqual(neu.snmp_version, network_poller_pb2.SnmpVersion.V2C)

        sysobjid = '1.3.6.1.4.1.43356.1.1.3'
        self.assertEqual(neu.sysobjid, None)
        neu.sysobjid = sysobjid
        self.assertEqual(neu.sysobjid, sysobjid)
        neu.sysobjid = None
        self.assertEqual(neu.sysobjid, None)

        poller_version = '11.22.33'
        neu.poller_version = poller_version
        self.assertEqual(poller_version, neu.poller_version)

        # Check functionality of a simple repeated type
        self.assertEqual(list(neu.errors), [])
        neu.errors.append(network_poller_pb2.PollError.ERROR_SNMP_OFFLINE)
        neu.errors.append(network_poller_pb2.PollError.ERROR_SNMP_BAD_MIB)
        self.assertEqual(len(neu.errors), 2)
        self.assertEqual(neu.errors[0], network_poller_pb2.PollError.ERROR_SNMP_OFFLINE)
        self.assertEqual(neu.errors[1], network_poller_pb2.PollError.ERROR_SNMP_BAD_MIB)
        del neu.errors[0]
        self.assertEqual(len(neu.errors), 1)
        self.assertEqual(neu.errors[0], network_poller_pb2.PollError.ERROR_SNMP_BAD_MIB)

        # Check functionality of a message repeated type
        self.assertEqual(list(neu.peers), [])
        peer = neu.peers.add()
        self.assertEqual(len(neu.peers), 1)
        del neu.peers[0]
        self.assertEqual(len(neu.peers), 0)

        # Check copy behavior of sub-message assignment
        ne = network_element_update.NetworkElement()
        self.assertEqual(ne.management_ip, None)
        ne.management_ip = '192.0.2.2'
        neu.data = ne  # this is a copy, not a ref
        self.assertEqual(neu.data.management_ip, '192.0.2.2')
        ne.management_ip = None
        self.assertEqual(ne.management_ip, None)
        self.assertEqual(neu.data.management_ip, '192.0.2.2')

        self.assertEqual(ne.name, None)
        ne.name = 'my-name'
        self.assertEqual(ne.name, 'my-name')
        ne.name = None
        self.assertEqual(ne.name, None)

        self.assertEqual(ne.manufacturer, None)
        ne.manufacturer = 'my-mfg'
        self.assertEqual(ne.manufacturer , 'my-mfg')
        ne.manufacturer = None
        self.assertEqual(ne.manufacturer, None)

        self.assertEqual(ne.model, None)
        ne.model = 'my-mod'
        self.assertEqual(ne.model, 'my-mod')
        ne.model = None
        self.assertEqual(ne.model, None)

        self.assertEqual(ne.serial_number, None)
        ne.serial_number = 'my-ser'
        self.assertEqual(ne.serial_number, 'my-ser')
        ne.serial_number = None
        self.assertEqual(ne.serial_number, None)

        self.assertEqual(ne.system_mac_address, None)
        ne.system_mac_address = '0a:00:00:01:02:03'
        self.assertEqual(ne.system_mac_address, '0a:00:00:01:02:03')
        with self.assertRaises(ValueError):
            ne.system_mac_address = 'asdf'
        with self.assertRaises(ValueError):
            ne.system_mac_address = '11:22:33:44:55:66:77'

        ne.system_mac_address = None
        self.assertEqual(ne.system_mac_address, None)

        self.assertEqual(ne.sw_version, None)
        ne.sw_version = '1.2.3'
        self.assertEqual(ne.sw_version, '1.2.3')
        ne.sw_version = None
        self.assertEqual(ne.sw_version, None)

        ne.errctx = network_element_update.ErrorContext()
        self.assertEqual(ne.uptime, None)
        ne.uptime = 12345
        self.assertEqual(ne.uptime, 12345)
        ne.uptime = -1
        self.assertEqual(ne.uptime, 12345)
        ne.uptime = pow(2, 63) + 1
        self.assertEqual(ne.uptime, 12345)
        ne.uptime = None
        self.assertEqual(len(ne.errctx.errors), 2)
        ne.errctx = None

        ne.errctx = network_element_update.ErrorContext()
        self.assertEqual(ne.cpu, None)
        ne.cpu= 5.5
        self.assertEqual(round(ne.cpu, 4), 5.5)
        ne.cpu = -1
        self.assertEqual(round(ne.cpu, 4), 5.5)
        ne.cpu = 100.1
        self.assertEqual(round(ne.cpu, 4), 5.5)
        ne.cpu = None
        self.assertEqual(ne.cpu, None)
        self.assertEqual(len(ne.errctx.errors), 2)
        ne.errctx = None

        ne.errctx = network_element_update.ErrorContext()
        self.assertEqual(ne.mem, None)
        ne.mem = 5.6
        self.assertEqual(round(ne.mem, 4), 5.6)
        ne.mem = -1
        self.assertEqual(round(ne.mem, 4), 5.6)
        ne.mem = 100.1
        self.assertEqual(round(ne.mem, 4), 5.6)
        ne.mem = None
        self.assertEqual(ne.mem, None)
        self.assertEqual(len(ne.errctx.errors), 2)
        ne.errctx = None

        self.assertEqual(ne.temperature, None)
        ne.temperature = 31.02
        self.assertEqual(round(ne.temperature, 4), 31.02)
        ne.temperature = -10
        self.assertEqual(round(ne.temperature, 4), -10)
        ne.temperature = None
        self.assertEqual(ne.temperature, None)

        poller_hash = sha256().digest()
        ne.poller_hash = poller_hash
        self.assertEqual(poller_hash, ne.poller_hash)

        ne.errctx = network_element_update.ErrorContext()
        self.assertEqual(ne.gps_sats_tracked, None)
        ne.gps_sats_tracked = 12
        self.assertEqual(ne.gps_sats_tracked, 12)
        ne.gps_sats_tracked = -1
        self.assertEqual(ne.gps_sats_tracked, 12)
        ne.gps_sats_tracked = pow(2, 31) + 1
        self.assertEqual(ne.gps_sats_tracked, 12)
        ne.gps_sats_tracked = None
        self.assertEqual(len(ne.errctx.errors), 2)
        ne.errctx = None

        self.assertEqual(ne.latitude, None)
        ne.latitude = 43.4772382
        self.assertEqual(ne.latitude, 43.4772382)
        ne.latitude = None
        self.assertEqual(ne.latitude, None)

        self.assertEqual(ne.longitude, None)
        ne.longitude = -80.5516853
        self.assertEqual(ne.longitude, -80.5516853)
        ne.longitude = None
        self.assertEqual(ne.longitude, None)

        self.assertEqual(ne.elevation, None)
        ne.elevation = 45.67
        self.assertEqual(round(ne.elevation, 4), 45.67)
        ne.elevation = None
        self.assertEqual(ne.elevation, None)

        self.assertEqual(ne.height, None)
        ne.height = 78.90
        self.assertEqual(round(ne.height, 4), 78.90)
        ne.height = None
        self.assertEqual(ne.height, None)

        # Interface
        intf = ne.interfaces.add()
        intf.id = '1'
        self.assertEqual(intf.id, '1')

        self.assertEqual(intf.name, None)
        intf.name = 'interface 1'
        self.assertEqual(intf.name, 'interface 1')
        intf.name = None
        self.assertEqual(intf.name, None)

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.type, None)
        intf.type = 6
        self.assertEqual(intf.type, 6)
        intf.type = -1
        self.assertEqual(intf.type, 6)
        intf.type = pow(2, 31) + 1
        self.assertEqual(intf.type, 6)
        intf.type = None
        self.assertEqual(intf.type, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.mtu, None)
        intf.mtu = 1500
        self.assertEqual(intf.mtu, 1500)
        intf.mtu = -1
        self.assertEqual(intf.mtu, 1500)
        intf.mtu = pow(2, 31) + 1
        self.assertEqual(intf.mtu, 1500)
        intf.mtu = None
        self.assertEqual(intf.mtu, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.speed, None)
        intf.speed = 10000000000
        self.assertEqual(intf.speed, 10000000000)
        intf.speed = -1
        self.assertEqual(intf.speed, 10000000000)
        intf.speed = pow(2, 63) + 1
        self.assertEqual(intf.speed, 10000000000)
        intf.speed = None
        self.assertEqual(intf.speed, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        self.assertEqual(intf.mac_address, None)
        intf.mac_address = '0a:00:00:01:02:03'
        self.assertEqual(intf.mac_address, '0a:00:00:01:02:03')
        intf.mac_address = None
        self.assertEqual(intf.mac_address, None)

        self.assertEqual(intf.admin_up, None)
        intf.admin_up = True
        self.assertEqual(intf.admin_up, True)
        intf.admin_up = None
        self.assertEqual(intf.admin_up, None)

        self.assertEqual(intf.oper_up, None)
        intf.oper_up = False
        self.assertEqual(intf.oper_up, False)
        intf.oper_up = None
        self.assertEqual(intf.oper_up, None)

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.in_octets, None)
        intf.in_octets = pow(2,31) + 1
        self.assertEqual(intf.in_octets, pow(2, 31) + 1)
        intf.in_octets = -1
        self.assertEqual(intf.in_octets, pow(2, 31) + 1)
        intf.in_octets = pow(2, 63) + 1
        self.assertEqual(intf.in_octets, pow(2, 31) + 1)
        intf.in_octets = None
        self.assertEqual(intf.in_octets, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.in_ucast_pkts, None)
        intf.in_ucast_pkts = pow(2,32) + 1
        self.assertEqual(intf.in_ucast_pkts, pow(2, 32) + 1)
        intf.in_ucast_pkts = -1
        self.assertEqual(intf.in_ucast_pkts, pow(2, 32) + 1)
        intf.in_ucast_pkts = pow(2, 63) + 1
        self.assertEqual(intf.in_ucast_pkts, pow(2, 32) + 1)
        intf.in_ucast_pkts = None
        self.assertEqual(intf.in_ucast_pkts, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.in_nucast_pkts, None)
        intf.in_nucast_pkts = pow(2,33) + 1
        self.assertEqual(intf.in_nucast_pkts, pow(2, 33) + 1)
        intf.in_nucast_pkts = -1
        self.assertEqual(intf.in_nucast_pkts, pow(2, 33) + 1)
        intf.in_nucast_pkts = pow(2, 63) + 1
        self.assertEqual(intf.in_nucast_pkts, pow(2, 33) + 1)
        intf.in_nucast_pkts = None
        self.assertEqual(intf.in_nucast_pkts, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.in_discards, None)
        intf.in_discards = pow(2,34) + 1
        self.assertEqual(intf.in_discards, pow(2, 34) + 1)
        intf.in_discards= -1
        self.assertEqual(intf.in_discards, pow(2, 34) + 1)
        intf.in_discards = pow(2, 63) + 1
        self.assertEqual(intf.in_discards, pow(2, 34) + 1)
        intf.in_discards = None
        self.assertEqual(intf.in_discards, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.in_errors, None)
        intf.in_errors = pow(2,35) + 1
        self.assertEqual(intf.in_errors, pow(2, 35) + 1)
        intf.in_errors = -1
        self.assertEqual(intf.in_errors, pow(2, 35) + 1)
        intf.in_errors = pow(2, 63) + 1
        self.assertEqual(intf.in_errors, pow(2, 35) + 1)
        intf.in_errors = None
        self.assertEqual(intf.in_errors, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.in_unknown_protos, None)
        intf.in_unknown_protos = pow(2,36) + 1
        self.assertEqual(intf.in_unknown_protos, pow(2, 36) + 1)
        intf.in_unknown_protos = -1
        self.assertEqual(intf.in_unknown_protos, pow(2, 36) + 1)
        intf.in_unknown_protos = pow(2, 63) + 1
        self.assertEqual(intf.in_unknown_protos, pow(2, 36) + 1)
        intf.in_unknown_protos = None
        self.assertEqual(intf.in_unknown_protos, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.out_octets, None)
        intf.out_octets = pow(2,37) + 1
        self.assertEqual(intf.out_octets, pow(2, 37) + 1)
        intf.out_octets = -1
        self.assertEqual(intf.out_octets, pow(2, 37) + 1)
        intf.out_octets = pow(2, 63) + 1
        self.assertEqual(intf.out_octets, pow(2, 37) + 1)
        intf.out_octets = None
        self.assertEqual(intf.out_octets, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.out_ucast_pkts, None)
        intf.out_ucast_pkts = pow(2,38) + 1
        self.assertEqual(intf.out_ucast_pkts, pow(2, 38) + 1)
        intf.out_ucast_pkts = -1
        self.assertEqual(intf.out_ucast_pkts, pow(2, 38) + 1)
        intf.out_ucast_pkts = pow(2, 63) + 1
        self.assertEqual(intf.out_ucast_pkts, pow(2, 38) + 1)
        intf.out_ucast_pkts = None
        self.assertEqual(intf.out_octets, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.out_nucast_pkts, None)
        intf.out_nucast_pkts = pow(2,39) + 1
        self.assertEqual(intf.out_nucast_pkts, pow(2, 39) + 1)
        intf.out_nucast_pkts = -1
        self.assertEqual(intf.out_nucast_pkts, pow(2, 39) + 1)
        intf.out_nucast_pkts = pow(2, 63) + 1
        self.assertEqual(intf.out_nucast_pkts, pow(2, 39) + 1)
        intf.out_nucast_pkts = None
        self.assertEqual(intf.out_octets, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.out_discards, None)
        intf.out_discards = pow(2,40) + 1
        self.assertEqual(intf.out_discards, pow(2, 40) + 1)
        intf.out_discards = -1
        self.assertEqual(intf.out_discards, pow(2, 40) + 1)
        intf.out_discards = pow(2, 63) + 1
        self.assertEqual(intf.out_discards, pow(2, 40) + 1)
        intf.out_discards = None
        self.assertEqual(intf.out_discards, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        intf.errctx = network_element_update.ErrorContext()
        self.assertEqual(intf.out_errors, None)
        intf.out_errors = pow(2,41) + 1
        self.assertEqual(intf.out_errors, pow(2, 41) + 1)
        intf.out_errors = -1
        self.assertEqual(intf.out_errors, pow(2, 41) + 1)
        intf.out_errors = pow(2, 63) + 1
        self.assertEqual(intf.out_errors, pow(2, 41) + 1)
        intf.out_errors = None
        self.assertEqual(intf.out_discards, None)
        self.assertEqual(len(intf.errctx.errors), 2)
        intf.errctx = None

        self.assertEqual(list(intf.ip_addresses), [])
        intf.ip_addresses.append('192.0.2.2')
        self.assertEqual(len(intf.ip_addresses), 1)
        self.assertEqual(intf.ip_addresses[0], '192.0.2.2')
        del intf.ip_addresses[0]
        self.assertEqual(len(intf.ip_addresses), 0)

        self.assertEqual(list(intf.bridged_mac_addresses), [])
        intf.bridged_mac_addresses.append('0a:00:00:0a:0b:0c')
        self.assertEqual(len(intf.bridged_mac_addresses), 1)
        self.assertEqual(intf.bridged_mac_addresses[0], '0a:00:00:0a:0b:0c')
        self.assertEqual([x for x in intf.bridged_mac_addresses], ['0a:00:00:0a:0b:0c'])
        del intf.bridged_mac_addresses[0]
        self.assertEqual(len(intf.bridged_mac_addresses), 0)

        poller_hash = sha256().digest()
        intf.poller_hash = poller_hash
        self.assertEqual(poller_hash, intf.poller_hash)

        # Link
        link = intf.links.add()
        link.mac_address = '0a:00:00:0d:0e:0f'
        self.assertEqual(link.mac_address, '0a:00:00:0d:0e:0f')

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.in_octets, None)
        link.in_octets = pow(2,31) + 1
        self.assertEqual(link.in_octets, pow(2, 31) + 1)
        link.in_octets = -1
        self.assertEqual(link.in_octets, pow(2, 31) + 1)
        link.in_octets = pow(2, 63) + 1
        self.assertEqual(link.in_octets, pow(2, 31) + 1)
        link.in_octets = None
        self.assertEqual(link.in_octets, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.in_ucast_pkts, None)
        link.in_ucast_pkts = pow(2,32) + 1
        self.assertEqual(link.in_ucast_pkts, pow(2, 32) + 1)
        link.in_ucast_pkts = -1
        self.assertEqual(link.in_ucast_pkts, pow(2, 32) + 1)
        link.in_ucast_pkts = pow(2, 63) + 1
        self.assertEqual(link.in_ucast_pkts, pow(2, 32) + 1)
        link.in_ucast_pkts = None
        self.assertEqual(link.in_ucast_pkts, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.in_nucast_pkts, None)
        link.in_nucast_pkts = pow(2,33) + 1
        self.assertEqual(link.in_nucast_pkts, pow(2, 33) + 1)
        link.in_nucast_pkts = -1
        self.assertEqual(link.in_nucast_pkts, pow(2, 33) + 1)
        link.in_nucast_pkts = pow(2, 63) + 1
        self.assertEqual(link.in_nucast_pkts, pow(2, 33) + 1)
        link.in_nucast_pkts = None
        self.assertEqual(link.in_nucast_pkts, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.in_discards, None)
        link.in_discards = pow(2,34) + 1
        self.assertEqual(link.in_discards, pow(2, 34) + 1)
        link.in_discards= -1
        self.assertEqual(link.in_discards, pow(2, 34) + 1)
        link.in_discards = pow(2, 63) + 1
        self.assertEqual(link.in_discards, pow(2, 34) + 1)
        link.in_discards = None
        self.assertEqual(link.in_discards, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.in_errors, None)
        link.in_errors = pow(2,35) + 1
        self.assertEqual(link.in_errors, pow(2, 35) + 1)
        link.in_errors = -1
        self.assertEqual(link.in_errors, pow(2, 35) + 1)
        link.in_errors = pow(2, 63) + 1
        self.assertEqual(link.in_errors, pow(2, 35) + 1)
        link.in_errors = None
        self.assertEqual(link.in_errors, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.in_unknown_protos, None)
        link.in_unknown_protos = pow(2,36) + 1
        self.assertEqual(link.in_unknown_protos, pow(2, 36) + 1)
        link.in_unknown_protos = -1
        self.assertEqual(link.in_unknown_protos, pow(2, 36) + 1)
        link.in_unknown_protos = pow(2, 63) + 1
        self.assertEqual(link.in_unknown_protos, pow(2, 36) + 1)
        link.in_unknown_protos = None
        self.assertEqual(link.in_unknown_protos, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.out_octets, None)
        link.out_octets = pow(2,37) + 1
        self.assertEqual(link.out_octets, pow(2, 37) + 1)
        link.out_octets = -1
        self.assertEqual(link.out_octets, pow(2, 37) + 1)
        link.out_octets = pow(2, 63) + 1
        self.assertEqual(link.out_octets, pow(2, 37) + 1)
        link.out_octets = None
        self.assertEqual(link.out_octets, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.out_ucast_pkts, None)
        link.out_ucast_pkts = pow(2,38) + 1
        self.assertEqual(link.out_ucast_pkts, pow(2, 38) + 1)
        link.out_ucast_pkts = -1
        self.assertEqual(link.out_ucast_pkts, pow(2, 38) + 1)
        link.out_ucast_pkts = pow(2, 63) + 1
        self.assertEqual(link.out_ucast_pkts, pow(2, 38) + 1)
        link.out_ucast_pkts = None
        self.assertEqual(link.out_octets, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.out_nucast_pkts, None)
        link.out_nucast_pkts = pow(2,39) + 1
        self.assertEqual(link.out_nucast_pkts, pow(2, 39) + 1)
        link.out_nucast_pkts = -1
        self.assertEqual(link.out_nucast_pkts, pow(2, 39) + 1)
        link.out_nucast_pkts = pow(2, 63) + 1
        self.assertEqual(link.out_nucast_pkts, pow(2, 39) + 1)
        link.out_nucast_pkts = None
        self.assertEqual(link.out_octets, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.out_discards, None)
        link.out_discards = pow(2,40) + 1
        self.assertEqual(link.out_discards, pow(2, 40) + 1)
        link.out_discards = -1
        self.assertEqual(link.out_discards, pow(2, 40) + 1)
        link.out_discards = pow(2, 63) + 1
        self.assertEqual(link.out_discards, pow(2, 40) + 1)
        link.out_discards = None
        self.assertEqual(link.out_discards, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.out_errors, None)
        link.out_errors = pow(2,41) + 1
        self.assertEqual(link.out_errors, pow(2, 41) + 1)
        link.out_errors = -1
        self.assertEqual(link.out_errors, pow(2, 41) + 1)
        link.out_errors = pow(2, 63) + 1
        self.assertEqual(link.out_errors, pow(2, 41) + 1)
        link.out_errors = None
        self.assertEqual(link.out_discards, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        link.errctx = network_element_update.ErrorContext()
        self.assertEqual(link.out_retransmits, None)
        link.out_retransmits = pow(2,42) + 1
        self.assertEqual(link.out_retransmits, pow(2, 42) + 1)
        link.out_retransmits = -1
        self.assertEqual(link.out_retransmits, pow(2, 42) + 1)
        link.out_retransmits = pow(2, 63) + 1
        self.assertEqual(link.out_retransmits, pow(2, 42) + 1)
        link.out_retransmits = None
        self.assertEqual(link.out_discards, None)
        self.assertEqual(len(link.errctx.errors), 2)
        link.errctx = None

        self.assertEqual(list(link.bridged_mac_addresses), [])
        link.bridged_mac_addresses.append('1a:00:00:0a:0b:0c')
        self.assertEqual(len(link.bridged_mac_addresses), 1)
        self.assertEqual(link.bridged_mac_addresses[0], '1a:00:00:0a:0b:0c')
        self.assertEqual([x for x in link.bridged_mac_addresses], ['1a:00:00:0a:0b:0c'])
        del link.bridged_mac_addresses[0]
        self.assertEqual(len(link.bridged_mac_addresses), 0)

        # Radio
        radio = intf.radios.add()
        radio.id = 'radio1'
        self.assertEqual(radio.id, 'radio1')

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.frequency, None)
        radio.frequency = 5540
        self.assertEqual(radio.frequency, 5540)
        radio.frequency = -1
        self.assertEqual(radio.frequency, 5540)
        radio.frequency = pow(2, 31) + 1
        self.assertEqual(radio.frequency, 5540)
        radio.frequency = None
        self.assertEqual(radio.frequency, None)
        self.assertEqual(len(radio.errctx.errors), 2)
        radio.errctx = None

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.chan_width, None)
        radio.chan_width = 40
        self.assertEqual(radio.chan_width, 40)
        radio.chan_width = -1
        self.assertEqual(radio.chan_width, 40)
        radio.chan_width = pow(2, 31) + 1
        self.assertEqual(radio.chan_width, 40)
        radio.chan_width = None
        self.assertEqual(radio.chan_width, None)
        self.assertEqual(len(radio.errctx.errors), 2)
        radio.errctx = None

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.tx_power, None)
        radio.tx_power = 15
        self.assertEqual(radio.tx_power, 15)
        radio.tx_power = -3
        self.assertEqual(radio.tx_power, -3)
        radio.tx_power = pow(2, 31) + 1
        self.assertEqual(radio.tx_power, -3)
        radio.tx_power = None
        self.assertEqual(radio.tx_power, None)
        self.assertEqual(len(radio.errctx.errors), 1)
        radio.errctx = None

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.dl_percent, None)
        radio.dl_percent = 75
        self.assertEqual(radio.dl_percent, 75)
        radio.dl_percent = -1
        self.assertEqual(radio.dl_percent, 75)
        radio.dl_percent = pow(2, 31) + 1
        self.assertEqual(radio.dl_percent, 75)
        radio.dl_percent = None
        self.assertEqual(radio.dl_percent, None)
        self.assertEqual(len(radio.errctx.errors), 2)
        radio.errctx = None

        radio.framing = network_poller_pb2.FramingMode.FIXED
        self.assertEqual(radio.framing, network_poller_pb2.FramingMode.FIXED)

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.rx_airtime, None)
        radio.rx_airtime = 19.2
        self.assertEqual(round(radio.rx_airtime, 4), 19.2)
        radio.rx_airtime = -1
        self.assertEqual(round(radio.rx_airtime, 4), 19.2)
        radio.rx_airtime = 100.1
        self.assertEqual(round(radio.rx_airtime, 4), 100.1)
        radio.rx_airtime = None
        self.assertEqual(radio.rx_airtime, None)
        self.assertEqual(len(radio.errctx.errors), 1)
        radio.errctx = None

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.tx_airtime, None)
        radio.tx_airtime = 6.8
        self.assertEqual(round(radio.tx_airtime, 4), 6.8)
        radio.tx_airtime = -1
        self.assertEqual(round(radio.tx_airtime, 4), 6.8)
        radio.tx_airtime = 100.1
        self.assertEqual(round(radio.tx_airtime, 4), 100.1)
        radio.tx_airtime = None
        self.assertEqual(radio.tx_airtime, None)
        self.assertEqual(len(radio.errctx.errors), 1)
        radio.errctx = None

        self.assertEqual(radio.gps_sync, None)
        radio.gps_sync = True
        self.assertEqual(radio.gps_sync, True)
        radio.gps_sync = None
        self.assertEqual(radio.gps_sync, None)

        self.assertEqual(radio.chain_balance, None)
        radio.chain_balance = 0.56
        self.assertEqual(round(radio.chain_balance, 4), 0.56)
        radio.chain_balance = None
        self.assertEqual(radio.chain_balance, None)

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.rx_link_rate, None)
        radio.rx_link_rate = 7449000000
        self.assertEqual(radio.rx_link_rate, 7449000000)
        radio.rx_link_rate = -1
        self.assertEqual(radio.rx_link_rate, 7449000000)
        radio.rx_link_rate = pow(2, 63) + 1
        self.assertEqual(radio.rx_link_rate, 7449000000)
        radio.rx_link_rate = None
        self.assertEqual(radio.rx_link_rate, None)
        self.assertEqual(len(radio.errctx.errors), 2)
        radio.errctx = None

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.tx_link_rate, None)
        radio.tx_link_rate = 17449000000
        self.assertEqual(radio.tx_link_rate, 17449000000)
        radio.tx_link_rate = -1
        self.assertEqual(radio.tx_link_rate, 17449000000)
        radio.tx_link_rate = pow(2, 63) + 1
        self.assertEqual(radio.tx_link_rate, 17449000000)
        radio.tx_link_rate = None
        self.assertEqual(radio.tx_link_rate, None)
        self.assertEqual(len(radio.errctx.errors), 2)
        radio.errctx = None

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.frame_duration, None)
        radio.frame_duration = 5000
        self.assertEqual(radio.frame_duration, 5000)
        radio.frame_duration = -1
        self.assertEqual(radio.frame_duration, 5000)
        radio.frame_duration = pow(2, 63) + 1
        self.assertEqual(radio.frame_duration, 5000)
        radio.frame_duration = None
        self.assertEqual(radio.frame_duration, None)
        self.assertEqual(len(radio.errctx.errors), 2)
        radio.errctx = None

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.rx_multiplexing_gain, None)
        radio.rx_multiplexing_gain = 1.1
        self.assertEqual(round(radio.rx_multiplexing_gain, 4), 1.1)
        radio.rx_multiplexing_gain = None
        self.assertEqual(radio.rx_multiplexing_gain, None)

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.tx_multiplexing_gain, None)
        radio.tx_multiplexing_gain = 1.1
        self.assertEqual(round(radio.tx_multiplexing_gain, 4), 1.1)
        radio.tx_multiplexing_gain = None
        self.assertEqual(radio.tx_multiplexing_gain, None)

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.rx_average_group_size, None)
        radio.rx_average_group_size = 2.1
        self.assertEqual(round(radio.rx_average_group_size, 4), 2.1)
        radio.rx_average_group_size = None
        self.assertEqual(radio.rx_average_group_size, None)

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.tx_average_group_size, None)
        radio.tx_average_group_size = 2.1
        self.assertEqual(round(radio.tx_average_group_size, 4), 2.1)
        radio.tx_average_group_size = None
        self.assertEqual(radio.tx_average_group_size, None)

        radio.errctx = network_element_update.ErrorContext()
        self.assertEqual(radio.wifi_mode, None)
        radio.wifi_mode = network_poller_pb2.WifiStandard.WIFI_STANDARD_80211B
        self.assertEqual(radio.wifi_mode, network_poller_pb2.WifiStandard.WIFI_STANDARD_80211B)
        radio.wifi_mode = network_poller_pb2.WifiStandard.WIFI_STANDARD_INVALID
        self.assertEqual(len(radio.errctx.errors), 1)
        self.assertEqual(radio.wifi_mode, None)
        radio.wifi_mode = network_poller_pb2.WifiStandard.WIFI_STANDARD_80211AY
        self.assertEqual(radio.wifi_mode, network_poller_pb2.WifiStandard.WIFI_STANDARD_80211AY)
        radio.wifi_mode = None
        self.assertEqual(radio.wifi_mode, None)

        # RadioStream
        stream = radio.streams.add()
        stream.id = '1'
        self.assertEqual(stream.id, '1')

        self.assertEqual(stream.polarization, None)
        stream.polarization = 'HORIZONTAL'
        self.assertEqual(stream.polarization, 'HORIZONTAL')
        stream.polarization = None
        self.assertEqual(stream.polarization, None)

        self.assertEqual(stream.noise_floor, None)
        stream.noise_floor = -79
        self.assertEqual(round(stream.noise_floor, 4), -79)
        stream.noise_floor = None
        self.assertEqual(stream.noise_floor, None)

        # WirelessLink
        wlink = stream.links.add()
        wlink.mac_address = '0a:00:00:0d:0e:0f'
        self.assertEqual(wlink.mac_address, '0a:00:00:0d:0e:0f')

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.distance, None)
        wlink.distance = 600
        self.assertEqual(wlink.distance, 600)
        wlink.distance = -1
        self.assertEqual(wlink.distance, 600)
        wlink.distance = pow(2, 31) + 1
        self.assertEqual(wlink.distance, 600)
        wlink.distance = None
        self.assertEqual(wlink.distance, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.connected_time, None)
        wlink.connected_time = pow(2,31) + 100
        self.assertEqual(wlink.connected_time, pow(2, 31) + 100)
        wlink.connected_time = -1
        self.assertEqual(wlink.connected_time, pow(2, 31) + 100)
        wlink.connected_time = pow(2, 63) + 1
        self.assertEqual(wlink.connected_time, pow(2, 31) + 100)
        wlink.connected_time = None
        self.assertEqual(wlink.connected_time, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        self.assertEqual(wlink.noise_floor, None)
        wlink.noise_floor = -89
        self.assertEqual(round(wlink.noise_floor, 4), -89)
        wlink.noise_floor = None
        self.assertEqual(wlink.noise_floor, None)

        self.assertEqual(wlink.rssi, None)
        wlink.rssi = -51
        self.assertEqual(round(wlink.rssi, 4), -51)
        wlink.rssi = None
        self.assertEqual(wlink.rssi, None)

        self.assertEqual(wlink.snr, None)
        wlink.snr = 44
        self.assertEqual(round(wlink.snr, 4), 44)
        wlink.snr = None
        self.assertEqual(wlink.snr, None)

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.rx_link_rate, None)
        wlink.rx_link_rate = 17449000000
        self.assertEqual(wlink.rx_link_rate, 17449000000)
        wlink.rx_link_rate = -1
        self.assertEqual(wlink.rx_link_rate, 17449000000)
        wlink.rx_link_rate = pow(2, 63) + 1
        self.assertEqual(wlink.rx_link_rate, 17449000000)
        wlink.rx_link_rate = None
        self.assertEqual(wlink.rx_link_rate, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.tx_link_rate, None)
        wlink.tx_link_rate = 6410000000
        self.assertEqual(wlink.tx_link_rate, 6410000000)
        wlink.tx_link_rate = -1
        self.assertEqual(wlink.tx_link_rate, 6410000000)
        wlink.tx_link_rate = pow(2, 63) + 1
        self.assertEqual(wlink.tx_link_rate, 6410000000)
        wlink.tx_link_rate = None
        self.assertEqual(wlink.tx_link_rate, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.rx_mcs, None)
        wlink.rx_mcs = 4
        self.assertEqual(wlink.rx_mcs, 4)
        wlink.rx_mcs = -1
        self.assertEqual(wlink.rx_mcs, 4)
        wlink.rx_mcs = pow(2, 31) + 1
        self.assertEqual(wlink.rx_mcs, 4)
        wlink.rx_mcs = None
        self.assertEqual(wlink.rx_mcs, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.tx_mcs, None)
        wlink.tx_mcs = 4
        self.assertEqual(wlink.tx_mcs, 4)
        wlink.tx_mcs = -1
        self.assertEqual(wlink.tx_mcs, 4)
        wlink.tx_mcs = pow(2, 31) + 1
        self.assertEqual(wlink.tx_mcs, 4)
        wlink.tx_mcs = None
        self.assertEqual(wlink.tx_mcs, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        self.assertEqual(wlink.rx_modulation_rate, None)
        wlink.rx_modulation_rate = 10.8
        self.assertEqual(round(wlink.rx_modulation_rate, 4), 10.8)
        wlink.rx_modulation_rate = None
        self.assertEqual(wlink.rx_modulation_rate, None)

        self.assertEqual(wlink.tx_modulation_rate, None)
        wlink.tx_modulation_rate = 7.7
        self.assertEqual(round(wlink.tx_modulation_rate, 4), 7.7)
        wlink.tx_modulation_rate = None
        self.assertEqual(wlink.tx_modulation_rate, None)

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.rx_airtime, None)
        wlink.rx_airtime = 60.4
        self.assertEqual(round(wlink.rx_airtime, 4), 60.4)
        wlink.rx_airtime = -1
        self.assertEqual(round(wlink.rx_airtime, 4), 60.4)
        wlink.rx_airtime = 100.1
        self.assertEqual(round(wlink.rx_airtime, 4), 100.1)
        wlink.rx_airtime = None
        self.assertEqual(wlink.rx_airtime, None)
        self.assertEqual(len(wlink.errctx.errors), 1)
        wlink.errctx = None

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.tx_airtime, None)
        wlink.tx_airtime = 4.5
        self.assertEqual(round(wlink.tx_airtime, 4), 4.5)
        wlink.tx_airtime = -1
        self.assertEqual(round(wlink.tx_airtime, 4), 4.5)
        wlink.tx_airtime = 100.1
        self.assertEqual(round(wlink.tx_airtime, 4), 100.1)
        wlink.tx_airtime = None
        self.assertEqual(wlink.tx_airtime, None)
        self.assertEqual(len(wlink.errctx.errors), 1)
        wlink.errctx = None

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.amc, None)
        wlink.amc = 4.5
        self.assertEqual(round(wlink.amc, 4), 4.5)
        wlink.amc = -1
        self.assertEqual(round(wlink.amc, 4), 4.5)
        wlink.amc = 100.1
        self.assertEqual(round(wlink.amc, 4), 4.5)
        wlink.amc = None
        self.assertEqual(wlink.amc, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.amq, None)
        wlink.amq = 4.5
        self.assertEqual(round(wlink.amq, 4), 4.5)
        wlink.amq = -1
        self.assertEqual(round(wlink.amq, 4), 4.5)
        wlink.amq = 100.1
        self.assertEqual(round(wlink.amq, 4), 4.5)
        wlink.amq = None
        self.assertEqual(wlink.amq, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        wlink.errctx = network_element_update.ErrorContext()
        self.assertEqual(wlink.ccq, None)
        wlink.ccq = 4.5
        self.assertEqual(round(wlink.ccq, 4), 4.5)
        wlink.ccq = -1
        self.assertEqual(round(wlink.ccq, 4), 4.5)
        wlink.ccq = 100.1
        self.assertEqual(round(wlink.ccq, 4), 4.5)
        wlink.ccq = None
        self.assertEqual(wlink.ccq, None)
        self.assertEqual(len(wlink.errctx.errors), 2)
        wlink.errctx = None

        # Sector
        sector = intf.sectors.add()

        sector.ssid = 'MySSID'
        self.assertEqual(sector.ssid, 'MySSID')

        sector.mode = network_poller_pb2.SectorMode.AP
        self.assertEqual(sector.mode, network_poller_pb2.SectorMode.AP)
