"""Function to sync a Preseem network metadata store to the new Preseem model."""
import asyncio
from collections import namedtuple
import ipaddress
import logging
from uuid import UUID

from grpc import RpcError
from tabulate import tabulate

from preseem import NetworkMetadataReference
from preseem_protobuf.model.element_pb2 import ElementDiscovery, ElementStatus

# STM-6039 Only allow the sync to run once at a time, to prevent duplicate
# entities from being created.
sync_lock = None

# This object is used to pass element items to the sync_preseem_model
# algorithm for a given source and intent role.
ElementSyncItem = namedtuple('ElementSyncItem', ('id', 'name', 'site', 'host'))


# Roles are hard-coded here since these should never change.
class ElementRole:
    SWITCH = UUID('41f2903a-311a-448d-9d4d-86dc8ba3cb07')
    AP = UUID('ae3f1aa5-063a-4a24-9744-822a82d107d9')
    ROUTER = UUID('bf433e28-122b-41df-b49a-77ac6544deac')
    CPE_RADIO = UUID('c72115d3-daf9-43ae-9475-083d62eae74f')


async def call_and_log_error(co):
    """Call a GRPC coroutine and log any errors from it."""
    try:
        return await co
    except asyncio.CancelledError:
        raise
    except RpcError as err:
        logging.warning('sync_preseem_model: error calling service: %s (%s)',
                        err.code(), err.details())
    except Exception as err:
        logging.warning('sync_preseem_model: %s error calling service: %s',
                        type(err).__name__,
                        err,
                        exc_info=1)
    return None


async def sync_preseem_model(preseem_model,
                             company_uuid,
                             source,
                             role,
                             items,
                             packages=None,
                             noact=False,
                             verbose=0):
    """
    Update the Preseem model to keep its ElementDiscovery objects in sync with
    another system.  This method takes the full set of items for a specific
    company, source, and role.  It diffs this against the current model and
    applies any create, modify or delete (set inactive) operations.
    It returns a map of {id -> uuid} for the caller to use.
    """
    global sync_lock
    if sync_lock is None:
        sync_lock = asyncio.Lock()

    assert isinstance(company_uuid, bytes)
    assert isinstance(source, str)
    assert isinstance(role, UUID)

    if source == '':
        raise ValueError('source cannot be the empty string')

    async with sync_lock:
        model_discs, model_sites, model_packages = await asyncio.gather(
            preseem_model.Element.ListElementDiscovery(
                company_uuid=company_uuid, intent_role_uuid=role.bytes),
            preseem_model.Site.List(company_uuid=company_uuid),
            preseem_model.Package.List(company_uuid=company_uuid))

        model_sites = {x.uuid: x for x in model_sites}
        model_discs = {x.uuid: x for x in model_discs}
        disc_ids = {(x.source, x.source_id): x for x in model_discs.values()}

        # Create a map of name -> site.  Handle multiple sites with the same
        # name by preferring active sites, then taking the first sorted uuid.
        site_names = {}
        for site in sorted(model_sites.values(), key=lambda x: x.uuid):
            cur_site = site_names.get(site.name)
            if not cur_site or (cur_site.inactive and not site.inactive):
                site_names[site.name] = site

        # Make sure all sites we reference exist and do some pre-validation.
        new_sites = set()
        item_ids = set()
        dup_ids = set()
        for item in items:
            if not isinstance(item.id, str) or item.id == '':
                raise ValueError('invalid string value for source_id')
            if not isinstance(item.name, str) or item.name == '':
                raise ValueError('invalid string value for name')
            if item.site is not None and (not isinstance(item.site, str)
                                          or item.site == ''):
                raise ValueError('invalid string value for site')
            if item.id in item_ids:
                dup_ids.add(item.id)
            else:
                item_ids.add(item.id)
            if item.site:
                site = site_names.get(item.site)
                if not site:
                    new_sites.add(item.site)
                elif site.inactive:
                    logging.info(
                        'sync_preseem_model(%s): reactivate site "%s"',
                        source, item.site)
                    if not noact:
                        await call_and_log_error(
                            preseem_model.Site.Update(
                                company_uuid=company_uuid,
                                uuid=site.uuid,
                                inactive=False))
        if dup_ids:
            logging.warning('sync_preseem_model(%s): duplicate IDs found: %s',
                            source, dup_ids)
            raise ValueError(f'duplicate IDs found')

        for site_name in new_sites:
            logging.info('sync_preseem_model(%s): create site "%s"', source,
                         site_name)
            if not noact:
                site = site_names[site_name] = await call_and_log_error(
                    preseem_model.Site.Create(company_uuid=company_uuid,
                                              name=site_name))
                if site:
                    model_sites[site.uuid] = site_names[site_name] = site

        unref_discs = set(
            [x[0] for x in model_discs.items() if x[1].source == source])
        uuid_map = {}
        for item in items:
            host = str(ipaddress.ip_address(item.host))  # validate ip address
            key = (source, item.id)
            disc = disc_ids.get(key)
            if disc:
                uuid_map[item.id] = UUID(bytes=disc.uuid)
                unref_discs.remove(disc.uuid)
                site = model_sites.get(
                    disc.site_uuid) if disc.site_uuid else None
                changes = []
                mods = {}
                if disc.inactive:
                    changes.append('reactivated')
                    mods['inactive'] = False
                if disc.name != item.name:
                    changes.append(f'name: {disc.name} -> {item.name}')
                    mods['name'] = item.name
                if disc.management_ip != host:
                    changes.append(
                        f'management_ip: {disc.management_ip} -> {host}')
                    mods['management_ip'] = host
                if site.name != item.site:
                    changes.append(f'site: {site.name} -> {item.site}')
                    mods['site_uuid'] = site_names[item.site].uuid
                if mods:
                    logging.info('sync_preseem_model(%s): modify %s (%s)',
                                 source, item, ','.join(changes))
                    if not noact:
                        await call_and_log_error(
                            preseem_model.Element.UpdateElementDiscovery(
                                company_uuid=company_uuid,
                                uuid=disc.uuid,
                                **mods))

            else:
                logging.info('sync_preseem_model(%s): create %s', source,
                             item)
                if not noact:
                    site_uuid = site_names[item.site].uuid
                    disc = await call_and_log_error(
                        preseem_model.Element.CreateElementDiscovery(
                            company_uuid=company_uuid,
                            name=item.name,
                            management_ip=item.host,
                            site_uuid=site_uuid,
                            source=source,
                            source_id=item.id,
                            intent_role_uuid=role.bytes))
                    if disc:
                        uuid = UUID(bytes=disc.uuid)
                        logging.info(
                            'sync_preseem_model(%s): created ElementDiscovery "%s"',
                            source, uuid)
                        uuid_map[item.id] = uuid
        for disc_uuid in unref_discs:
            disc = model_discs.get(disc_uuid)
            if not disc.inactive:
                logging.info("sync_preseem_model(%s): deactivate %s", source,
                             UUID(bytes=disc_uuid))
                if not noact:
                    await call_and_log_error(
                        preseem_model.Element.UpdateElementDiscovery(
                            company_uuid=company_uuid,
                            uuid=disc_uuid,
                            inactive=True))

        if packages is not None:
            assert isinstance(packages, dict)
            new_package_ids = set(packages)
            for mpkg in model_packages:
                if mpkg.source != source:
                    continue
                bpkg = packages.get(mpkg.source_id)
                if bpkg:
                    if not isinstance(bpkg.name, str) or bpkg.name == '':
                        raise ValueError('invalid string value for package name')
                    if bpkg.download_rate and (not isinstance(bpkg.download_rate, int) or bpkg.download_rate < 0):
                        raise ValueError('invalid value for package down rate')
                    if bpkg.upload_rate and (not isinstance(bpkg.upload_rate, int) or bpkg.upload_rate < 0):
                        raise ValueError('invalid value for package up rate')
                    changes = []
                    mods = {}
                    new_package_ids.discard(mpkg.source_id)
                    if mpkg.inactive:
                        changes.append('reactivated')
                        mods['inactive'] = False
                    if mpkg.name != bpkg.name:
                        changes.append(f'name: {mpkg.name} -> {bpkg.name}')
                        mods['name'] = bpkg.name
                    if mpkg.download_rate != (bpkg.download_rate or None):
                        changes.append(f'download_rate: {mpkg.download_rate} -> {bpkg.download_rate}')
                        mods['download_rate'] = bpkg.download_rate or None
                    if mpkg.upload_rate != (bpkg.upload_rate or None):
                        changes.append(f'upload_rate: {mpkg.upload_rate} -> {bpkg.upload_rate}')
                        mods['upload_rate'] = bpkg.upload_rate or None
                    if mods:
                        logging.info('sync_packages(%s): modify %s (%s)',
                                     source, mpkg, ', '.join(changes))
                        await preseem_model.Package.Update(
                            company_uuid=company_uuid,
                            uuid=mpkg.uuid,
                            **mods)
                else:
                    if not mpkg.inactive:
                        logging.info("sync_preseem_model(%s): deactivate package %s", source,
                                     UUID(bytes=mpkg.uuid))
                        if not noact:
                            await call_and_log_error(
                                preseem_model.Package.Update(
                                    company_uuid=company_uuid,
                                    uuid=mpkg.uuid,
                                    inactive=True))
            for pkg_id in new_package_ids:
                bpkg = packages.get(pkg_id)
                if bpkg:
                    if not isinstance(bpkg.id, str) or bpkg.id == '':
                        raise ValueError('invalid string value for package source_id')
                    if not isinstance(bpkg.name, str) or bpkg.name == '':
                        raise ValueError('invalid string value for package name')
                    if bpkg.download_rate and (not isinstance(bpkg.download_rate, int) or bpkg.download_rate < 0):
                        raise ValueError('invalid value for package down rate')
                    if bpkg.upload_rate and (not isinstance(bpkg.upload_rate, int) or bpkg.upload_rate < 0):
                        raise ValueError('invalid value for package up rate')
                    logging.info('sync_preseem_model(%s): create %s', source, bpkg)
                    if not noact:
                        mpkg = await call_and_log_error(
                            preseem_model.Package.Create(
                                company_uuid=company_uuid,
                                name=bpkg.name,
                                source=source,
                                source_id=bpkg.id,
                                download_rate=bpkg.download_rate or None,
                                upload_rate=bpkg.upload_rate or None))
                        if mpkg:
                            uuid = UUID(bytes=mpkg.uuid)
                            logging.info(
                                'sync_preseem_model(%s): created Package "%s"',
                                source, uuid)

        return uuid_map
