"""Fake Preseem Model implementation."""
import copy
import time
from uuid import uuid4, UUID

import grpc
from preseem.preseem_model import BaseModelObject

class FakePreseemModel:
    _companies = {} # company_uuid to Company object
    _elements = {}  # company_uuid to {element_uuid to Element object}
    _sites = {}     # company_uuid to {site_uuid to Site object}
    _elem_disc = {} # company_uuid to {uuid to ElementDiscovery object}

    def _new_attrs():
        """Generate attributes for a new preseem-model object."""
        now = time.time()
        uuid = uuid4()
        return {
            'id': uuid.int & (1<<64)-1,
            'uuid': uuid.bytes,
            'created_at': now,
            'updated_at': now,
            'inactive': False,
            'inactive_time': None
        }

    def _set_inactive(obj):
        obj.inactive = True
        obj.updated_at = obj.inactive_time = time.time()

    async def init(self):
        pass

    async def copy_from(self, other, company_uuid):
        """Insert exact copies of entities from another model into this one."""
        try:
            for obj in await other.Company.List():
                FakePreseemModel._companies[obj.uuid] = obj
        except grpc.RpcError as err:  # likely permissions, fake this
            FakePreseemModel._companies[company_uuid] = other.Company(uuid=company_uuid)
        for obj in await other.Element.List(company_uuid=company_uuid):
            elems = FakePreseemModel._elements.get(company_uuid)
            if elems is None:
                elems = FakePreseemModel._elements[company_uuid] = {}
            elems[obj.uuid] = obj
        for obj in await other.Element.ListElementDiscovery(company_uuid=company_uuid):
            discs = FakePreseemModel._elem_disc.get(company_uuid)
            if discs is None:
                discs = FakePreseemModel._elem_disc[company_uuid] = {}
            discs[obj.uuid] = obj
        for obj in await other.Site.List(company_uuid=company_uuid):
            sites = FakePreseemModel._sites.get(company_uuid)
            if sites is None:
                sites = FakePreseemModel._sites[company_uuid] = {}
            sites[obj.uuid] = obj


    """Fake preseem model that implements the python API."""
    class Company(BaseModelObject):
        @classmethod
        async def Create(cls, **kwargs):
            obj = cls(**FakePreseemModel._new_attrs(),
                    name=kwargs.get('name') or '',
                    parent_uuid=kwargs.get('parent_uuid'),
                    old_id=kwargs.get('old_id'))
            FakePreseemModel._companies[obj.uuid] = obj
            return copy.deepcopy(obj)

        @classmethod
        async def Get(cls, **kwargs):
            uuid = kwargs.get('uuid')
            assert uuid
            return copy.deepcopy(FakePreseemModel._companies[uuid])

        @classmethod
        async def GetByOldId(cls, **kwargs):
            old_id = kwargs.get('old_id')
            assert old_id
            companies = [x for x in FakePreseemModel._companies.values() if x.old_id == old_id]
            assert len(companies) < 2
            if companies:
                return copy.deepcopy(companies[0])
            # TODO error for empty?

        @classmethod
        async def List(cls, **kwargs):
            return [copy.deepcopy(x) for x in FakePreseemModel._companies.values()]


    class ElementDiscovery(BaseModelObject):
        pass

    class Element(BaseModelObject):
        @classmethod
        async def Create(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            company = FakePreseemModel._companies[company_uuid]
            obj = cls(**FakePreseemModel._new_attrs(),
                    company_uuid=company.uuid,
                    name=kwargs.get('name'),
                    serial_number=kwargs.get('serial_number'),
                    system_mac_address=kwargs.get('system_mac_address'),
                    imsi=kwargs.get('imsi'),
                    management_ip=kwargs.get('management_ip'),
                    site_uuid=kwargs.get('site_uuid'),
                    status=kwargs.get('status') or 0,
                    poller_hash=kwargs.get('poller_hash'))
            elems = FakePreseemModel._elements.get(company.uuid)
            if elems is None:
                elems = FakePreseemModel._elements[company_uuid] = {}
            elems[obj.uuid] = obj
            return copy.deepcopy(obj)

        @classmethod
        async def List(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            # the real impl returns empty list if the company doesn't exist
            return [copy.deepcopy(x) for x in (FakePreseemModel._elements.get(company_uuid) or {}).values()]

        @classmethod
        async def Delete(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            elems = FakePreseemModel._elements[company_uuid]
            if elems:
                elem = elems.get(uuid)
                if elem:
                    del elems[uuid]

        @classmethod
        async def Update(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            elems = FakePreseemModel._elements[company_uuid]
            if elems:
                elem = elems.get(uuid)
                for attr, val in kwargs.items():
                    if attr == 'company_uuid' or attr == 'uuid':
                        continue
                    elif attr == 'inactive' and val is True:
                        FakePreseemModel._set_inactive(elem)
                    else:
                        setattr(elem, attr, val)

        @classmethod
        async def CreateElementDiscovery(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            company = FakePreseemModel._companies[company_uuid]
            obj = FakePreseemModel.ElementDiscovery(**FakePreseemModel._new_attrs(),
                    company_uuid=company.uuid,
                    name=kwargs.get('name'),
                    management_ip=kwargs.get('management_ip'),
                    site_uuid=kwargs.get('site_uuid'),
                    source=kwargs.get('source'),
                    source_id=kwargs.get('source_id'),
                    intent_role_uuid=kwargs.get('intent_role_uuid'),
                    element_uuid=kwargs.get('element_uuid'),
                    status=kwargs.get('status'))
            discs = FakePreseemModel._elem_disc.get(company.uuid)
            if discs is None:
                discs = FakePreseemModel._elem_disc[company_uuid] = {}
            discs[obj.uuid] = obj
            return copy.deepcopy(obj)

        @classmethod
        async def ListElementDiscovery(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            intent_role_uuid = kwargs.get('intent_role_uuid')
            assert company_uuid
            # the real impl returns empty list if the company doesn't exist
            return [copy.deepcopy(x) for x in (FakePreseemModel._elem_disc.get(company_uuid) or {}).values() if not intent_role_uuid or intent_role_uuid == x.intent_role_uuid]

        @classmethod
        async def DeleteElementDiscovery(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            discs = FakePreseemModel._elem_disc[company_uuid]
            if discs:
                disc = discs.get(uuid)
                if disc:
                    del discs[uuid]

        @classmethod
        async def UpdateElementDiscovery(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            discs= FakePreseemModel._elem_disc[company_uuid]
            if discs:
                disc = discs.get(uuid)
                for attr, val in kwargs.items():
                    if attr == 'company_uuid' or attr == 'uuid':
                        continue
                    elif attr == 'inactive' and val is True:
                        FakePreseemModel._set_inactive(disc)
                    else:
                        if getattr(disc, attr) != val:
                            disc.updated_at = time.time()
                        setattr(disc, attr, val)


    class Site(BaseModelObject):
        @classmethod
        async def Create(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            company = FakePreseemModel._companies[company_uuid]
            obj = cls(**FakePreseemModel._new_attrs(),
                    company_uuid=company.uuid,
                    name=kwargs.get('name') or '')
            sites = FakePreseemModel._sites.get(company.uuid)
            if sites is None:
                sites = FakePreseemModel._sites[company_uuid] = {}
            sites[obj.uuid] = obj
            return copy.deepcopy(obj)

        @classmethod
        async def List(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            # the real impl returns empty list if the company doesn't exist
            return [copy.deepcopy(x) for x in (FakePreseemModel._sites.get(company_uuid) or {}).values()]

        @classmethod
        async def Delete(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            sites = FakePreseemModel._sites[company_uuid]
            if sites:
                site = sites.get(uuid)
                if site:
                    del sites[uuid]

        @classmethod
        async def Update(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            sites = FakePreseemModel._sites[company_uuid]
            if sites:
                site = sites.get(uuid)
                for attr, val in kwargs.items():
                    if attr == 'company_uuid' or attr == 'uuid':
                        continue
                    elif attr == 'inactive' and val is True:
                        FakePreseemModel._set_inactive(site)
                    else:
                        if getattr(site, attr) != val:
                            site.updated_at = time.time()
                        setattr(site, attr, val)
