"""Function to sync a Preseem network metadata store to the new Preseem model."""
import asyncio
import logging
from uuid import UUID

from grpc import RpcError
from tabulate import tabulate

from preseem import NetworkMetadataReference
from preseem_protobuf.model.element_pb2 import ElementStatus

# STM-6039 Only allow the sync to run once at a time, to prevent duplicate
# entities from being created.
sync_lock = None

async def call_and_log_error(co):
    """Call a GRPC coroutine and log any errors from it."""
    try:
        return await co
    except asyncio.CancelledError:
        raise
    except RpcError as err:
        logging.warning('sync_preseem_model: error calling service: %s (%s)', err.code(), err.details())
    except Exception as err:
        logging.warning('sync_preseem_model: %s error calling service: %s', type(err).__name__, err)
    return None

async def sync_preseem_model(preseem_model,
                             nm_model,
                             company_uuid,
                             system=None,
                             delete=False,
                             noact=False,
                             throttle=None,
                             verbose=0):
    """
    Update the Preseem model to keep it in sync with metadata references.
    Right now this is done using the name for Site and the site+ap name for AP.
    In the future we will want to use system and system_id for this.
    The "delete" flag is deprecated.
    """
    global sync_lock
    if sync_lock is None:
        sync_lock = asyncio.Lock()

    async with sync_lock:
        aps = {
            ap.value: ap
            for ap in (nm_model.refs.get('ap') or {}).values()
            if system == ap.attributes.get('system')
        }
        if not aps:
            return  # nothing to do if no APs are defined for this system

        model_elems, model_sites = await asyncio.gather(
            preseem_model.Element.List(company_uuid=company_uuid),
            preseem_model.Site.List(company_uuid=company_uuid))

        # Organize sites and elements.  We never reactivate inactive objects.
        model_sites = {x.uuid: x for x in model_sites if not x.inactive}
        model_elems = {x.uuid: x for x in model_elems if not x.inactive}
        site_names = {x.name: x for x in model_sites.values()}
        linked_elems = set()

        ip_elems = {}
        for elem in model_elems.values():
            if elem.management_ip:
                el = ip_elems.get(elem.management_ip)
                if el is None:
                    el = ip_elems[elem.management_ip] = []
                el.append(elem)

        # We start by finding an Element for each AP reference, only looking
        # at network information (Management IP for now)
        rows = []
        ap_elem = {}
        new_sites = set()
        for ap_ref in aps.values():
            ip = ap_ref.attributes.get('ip_address')
            site_name = ap_ref.attributes.get('topology2')
            if site_name and site_name not in site_names:
                new_sites.add(site_name)
            if ip:
                el = ip_elems.get(ip)
                elem = None
                if el:
                    if len(el) == 1:
                        elem = el[0]
                    else:
                        # Multiple elements have this IP; find the best one.
                        # Prefer a polled one; if there are multiple polled
                        # make a deterministic choice by status, then by
                        # order of poller hash.
                        polled_el = [x for x in el if x.poller_hash]
                        if polled_el:
                            if len(polled_el) == 1:
                                elem = polled_el[0]
                            else:
                                polled_online = [
                                    x for x in polled_el if x.status in (
                                        ElementStatus.ELEMENT_STATUS_ONLINE,
                                        ElementStatus.ELEMENT_STATUS_WARNING,
                                        ElementStatus.ELEMENT_STATUS_ERROR)
                                ]
                                if polled_online:
                                    elem = sorted(
                                        polled_online,
                                        key=lambda x: x.poller_hash)[-1]
                                else:
                                    elem = sorted(
                                        polled_el,
                                        key=lambda x: x.poller_hash)[-1]
                        else:
                            elem = sorted(el, key=lambda x: x.uuid)[0]
                    if elem:
                        linked_elems.add(elem.uuid)
                ap_elem[ap_ref.value] = elem
                rows.append(
                    (ip, (site_name, ap_ref.attributes.get('topology1')),
                     ap_ref.value, [str(UUID(bytes=x.uuid))
                                    for x in el] if el else "",
                     UUID(bytes=elem.uuid) if elem else ''))

        logging.info('sync_preseem_model: loaded %s ap refs and %s elements',
                     len(aps), len(model_elems))
        if verbose:
            print(
                tabulate(sorted(rows, key=lambda x: (x[1], x[2], x[0])),
                         ('IP', 'BSS ID', 'Ref ID', 'IP Elems', 'Element')))

        # Make sure we have all sites we need.
        for site_name in new_sites:
            logging.info('sync_preseem_model: create site "%s"', site_name)
            if not noact:
                site = site_names[site_name] = await call_and_log_error(preseem_model.Site.Create(
                    company_uuid=company_uuid, name=site_name))
                if site:
                    model_sites[site.uuid] = site

        # Create or update elements
        ap_model_refs = nm_model.refs.get('ap_model') or {}
        num_set = 0
        for ap_ref in aps.values():
            elem = ap_elem.get(ap_ref.value)
            ip = ap_ref.attributes.get('ip_address')
            name = ap_ref.attributes.get('topology1')
            site_name = ap_ref.attributes.get('topology2')
            site_uuid = None
            if site_name:
                site = site_names.get(site_name)
                if site:
                    site_uuid = site.uuid
            if elem:
                updates = {}
                if elem.site_uuid != site_uuid:
                    updates['site_uuid'] = site_uuid
                if elem.name != name:
                    updates['name'] = name
                if updates:
                    logging.info('sync_preseem_model: update element "%s" %s',
                                 UUID(bytes=elem.uuid), updates)
                    if not noact:
                        await call_and_log_error(preseem_model.Element.Update(
                            company_uuid=company_uuid, uuid=elem.uuid, **updates))
            else:
                if ip:
                    logging.info(
                        'sync_preseem_model: create element "%s" in site "%s" with management IP %s',
                        name, site_name, ip)
                    if not noact:
                        elem = await call_and_log_error(
                            preseem_model.Element.Create(
                                company_uuid=company_uuid,
                                site_uuid=site_uuid,
                                name=name,
                                management_ip=ip,
                                status=ElementStatus.ELEMENT_STATUS_UNKNOWN)
                        )
                        if elem:
                            logging.info('sync_preseem_model: created element "%s"',
                                         UUID(bytes=elem.uuid))
                            linked_elems.add(elem.uuid)
            if elem:
                model_ref = ap_model_refs.get(ap_ref.value)
                elem_uuid = str(UUID(bytes=elem.uuid))
                if not model_ref or model_ref.attributes.get(
                        'uuid') != elem_uuid:
                    if not throttle or num_set == 0:
                        await nm_model.set_ref(
                            NetworkMetadataReference('ap_model', ap_ref.value,
                                                     {'uuid': elem_uuid}))
                        num_set += 1

        # Cleanup elements that were referencing a billing entity but are no longer.
        for elem in model_elems.values():
            if elem.uuid in linked_elems or not (elem.site_uuid and elem.name):
                continue
            if elem.poller_hash:
                # Just remove the billing ID from polled elements
                logging.info(
                    "sync_preseem_model: remove billing ID from element %s",
                    UUID(bytes=elem.uuid))
                if not noact:
                    await call_and_log_error(preseem_model.Element.Update(company_uuid=company_uuid,
                                                       uuid=elem.uuid,
                                                       site_uuid=None,
                                                       name=None))
            else:
                # Deactivate elements that haven't been polled
                logging.info("sync_preseem_model: deactivate element %s",
                             UUID(bytes=elem.uuid))
                if not noact:
                    await call_and_log_error(preseem_model.Element.Update(company_uuid=company_uuid,
                                                       uuid=elem.uuid,
                                                       inactive=True))
