"""
API to the Preseem Data Model.
The API is dynamically built from the GRPC model, by creating a class under the
PreseemModel object for each GRPC module, with static methods in that class.

This model maintains a map of all objects by uuid to pb and py objs.
The python object is kept in sync with changes on the server.
Local changes to the python objects are ignored; update methods work as
per the model API, where fields are passed directly to the method which
enact changes on the server.  This implementation anticipates being able
to subscribe to changes in the future to keep the local cache up to date;
in the meantime, users can do Get or List operations to update the local
objects.

Python objects can only be created by the client as a result of a create,
list or get operation.
"""
from functools import partial
from importlib import import_module
import logging
import os.path
import pkgutil
from uuid import UUID

import google.protobuf.timestamp_pb2

import preseem_protobuf.model

async def _op(model, op, reqtype, delete=False, **kwargs):
    """Wrapper function to perform a GRPC operation on the model."""
    def pb2py(pb, cls):
        """convert a protobuf to a python object."""
        fields = {}
        for f in pb.DESCRIPTOR.fields_by_name:
            try:
                v = getattr(pb, f, None) if pb.HasField(f) else None
            except ValueError:  # HasField only works for optional and submessage
                v = getattr(pb, f, None)
            if isinstance(v, google.protobuf.timestamp_pb2.Timestamp):
                v = v.ToNanoseconds() / 1000000000
            fields[f] = v
        uuid = fields.get('uuid')
        if uuid:
            cur = model._objs.get(uuid)
            if cur:
                curpb, curpy = cur
                for f, v in fields.items():
                    try:
                        pbv = getattr(curpb, f, None) if curpb.HasField(f) else None
                    except ValueError:
                        pbv = getattr(curpb, f, None)
                    if isinstance(pbv,  google.protobuf.timestamp_pb2.Timestamp):
                        pbv = pbv.ToNanoseconds() / 1000000000
                    if v != pbv:
                        pyv = getattr(curpy, f, None)
                        if pyv and pbv != pyv:
                            # This field is dirty.  We may want different
                            # behavior here someday?  For now overwrite it.
                            setattr(curpy, f, v)
                        else:
                            setattr(curpy, f, v)
                curpb.CopyFrom(pb)
            else:
                curpy = cls(**fields)
                model._objs[uuid] = (pb, curpy)
            return curpy
        raise RuntimeError('No UUID found on received object')

    req = reqtype()
    for k, v in kwargs.items():
        f = reqtype.DESCRIPTOR.fields_by_name[k]
        if v is None and not (f.message_type and f.message_type.name.startswith('Nullable')):
            continue  # allow args to be passed a None to avoid setting them.
        if f.message_type:  # this is a nested message, not a simple type
            if f.message_type.name.startswith('Nullable'):
                # special case where we have a OneOf for optional values
                if v is None:
                    setattr(getattr(req, k), 'null', 0) # NullValue.NULL_VALUE
                else:
                    setattr(getattr(req, k), 'data', v)
            else:
                if isinstance(v, list):
                    for x in v:
                        lv = getattr(req, k).add()
                        for nk in f.message_type.fields_by_name:
                            nv = getattr(x, nk)
                            if nv is not None:
                                setattr(lv, nk, getattr(x, nk))
                assert False  # TODO nested types, we don't need this yet
        else:
            setattr(req, k, v)

    r = await model._client._grpc_op(op, req, collect_results=True)
    if isinstance(r, list):
        if r:
            cls = model._types.get(r[0].DESCRIPTOR.name)
            return [pb2py(x, cls) for x in r]
        return []
    else:
        cls = model._types.get(r.DESCRIPTOR.name)
        if cls:
            return pb2py(r, cls)
        if delete:
            # this was a delete operation; delete our cached object data
            uuid = kwargs.get('uuid')
            if uuid:
                cur = model._objs.pop(uuid, None)
                return cur[1] if cur else None
        if len(r.DESCRIPTOR.fields) == 0:
            # This is just an empty response, we simply return None.
            return
        return r


# XXX once we require Python 3.7, this should be a dataclass.
class BaseModelObject:
    """Base class for all pythonized model objects."""
    def __init__(self, **kwargs):
        self.__dict__.update(kwargs)

    def __repr__(self):
        attrs = {x: str(UUID(bytes=y)) if y and x.endswith('uuid') else y for x, y in self.__dict__.items()}
        return f'{type(self).__name__}({attrs})'

    def __eq__(self, other):
        if other.__class__ is not self.__class__:
            return NotImplemented
        return self.__dict__ == other.__dict__

    def __hash__(self):
        return hash((x[1] for x in sorted(self.__dict__.items())))

    def replace(self, **kwargs):
        """Return a copy of this object with some fields updated."""
        args = self.__dict__.copy()
        args.update(kwargs)
        return self.__class__(**args)


class PreseemModel:
    """API to the Preseem Data Model."""
    def __init__(self, client):
        """Initialize the model with a GRPC client."""
        self._client = client
        self._objs = {}
        self._types = {}

    async def init(self):
        """Dynamically load model protobuf modules."""
        await self._client.connect()
        pkgpath = os.path.dirname(preseem_protobuf.model.__file__)
        for x in pkgutil.iter_modules([pkgpath]):
            if x.name.endswith('_pb2_grpc'):
                name = x.name[:-9]  # trim "_pb2_grpc" from the name
                if name == 'common':
                    continue

                # Load the python modules for this model object type
                pmod = import_module(f'preseem_protobuf.model.{name}_pb2')
                gmod = import_module(f'preseem_protobuf.model.{name}_pb2_grpc')
                stubcls = [x for x in dir(gmod) if x.endswith('Stub')][0]
                svcstub = getattr(gmod, stubcls)(self._client._channel)
                svcname = stubcls[:-11]  # trim "ServiceStub" from the name

                # Create a new python class for this model object type
                cls = type(svcname, (BaseModelObject,), {})
                setattr(self, svcname, cls)
                self._types[svcname] = cls

                # Create python methods for the services
                for sname, s in pmod.DESCRIPTOR.services_by_name.items():
                    for method in s.methods:
                        reqtype = getattr(pmod, method.input_type.name)
                        reqns = type(reqtype.DESCRIPTOR.name, (BaseModelObject,), {})
                        setattr(cls, reqtype.DESCRIPTOR.name, reqns)
                        op=getattr(svcstub, method.name)
                        setattr(cls, method.name, partial(_op, model=self, op=op, delete=method.name=='Delete', reqtype=reqtype))
                        for nt in reqtype.DESCRIPTOR.nested_types:
                            # Create a nested object class
                            nc = type(nt.name, (BaseModelObject,), {f: None for f in nt.fields_by_name})
                            setattr(reqns, nt.name, nc)
