"""
This is a helper class to provide some common functionality around HTTP
client operations.
"""
import asyncio
import aiohttp
import logging
import ssl
import vcr

# This library used by aiohttp 3.8 needs its logging turned down
logging.getLogger("charset_normalizer").setLevel(logging.ERROR)

class HttpClient(object):
    """Helper class to simplify HTTP operations on an open session."""
    def __init__(self, url, verify_ssl=True, username=None, password=None,
                 auth=None, cert=None, key=None, user_agent=None,
                 connection_limit=None, conn_timeout=None, read_timeout=None,
                 record=None):
        """Initialize the session pool and connector objects.
           We found enable_cleanup_closed reduced memory leaks (STM-3012)."""
        work_around_aiohttp_3535()
        timeout = aiohttp.ClientTimeout(connect=conn_timeout or 30,
                                        sock_read=read_timeout or 300)
        if verify_ssl:
            conn = aiohttp.TCPConnector(limit_per_host=connection_limit,
                                        enable_cleanup_closed=True)
            self._session = aiohttp.ClientSession(connector=conn, timeout=timeout)
        else:
            ssl_ctx = ssl._create_unverified_context()
            try:
                ssl_ctx.set_ciphers('DEFAULT@SECLEVEL=1')  # STM-4531
            except ssl.SSLError:
                pass  # some platforms don't allow this, ok
            if cert and key:
                ssl_ctx.load_cert_chain(cert, key)
                conn = aiohttp.TCPConnector(ssl_context=ssl_ctx,
                                            limit_per_host=connection_limit,
                                            enable_cleanup_closed=True)
            else:
                conn = aiohttp.TCPConnector(ssl_context=ssl_ctx,
                                            limit_per_host=connection_limit,
                                            enable_cleanup_closed=True)
            self._session = aiohttp.ClientSession(connector=conn, timeout=timeout,
                                                  cookie_jar=aiohttp.CookieJar(unsafe=True))
        self._url = url
        self.auth = auth
        if username:
            if password:
                self.auth = aiohttp.BasicAuth(username, password)
            else:
                self.auth = aiohttp.BasicAuth(username)
        self.user_agent = user_agent or 'preseem-netmeta-agent'

        self.vcr_cassette = None
        if record:  # record all http transactions to the provided name
            self.record(record)

    def record(self, name):
        if not self.vcr_cassette:
            vcr_log = logging.getLogger("vcr")
            vcr_log.setLevel(logging.WARNING)
            self.vcr = vcr.VCR(record_mode='all', cassette_library_dir='/tmp')
            self.vcr_cm = self.vcr.use_cassette(name)
            self.vcr_cassette = self.vcr_cm.__enter__()

    async def close(self):
        if self.vcr_cassette:
            self.vcr_cm.__exit__()
            self.vcr_cassette = None
        await self._session.close()

    async def request(self, path, num_retries=5, handler=None, response_encoding=None, **kwargs):
        """Post a request to the session."""
        hdrs = {
            'user-agent': self.user_agent
        }
        hdrs.update(kwargs.get('headers') or {})
        if 'headers' in kwargs:
            del kwargs['headers']
        if self.auth and 'auth' not in kwargs:
            kwargs['auth'] = self.auth
        url = '{}{}'.format(self._url, path)
        err = None
        for retry_num in range(num_retries + 1):
            if retry_num > 0:
                await asyncio.sleep(2**retry_num)
                if retry_num > 1:
                    # don't log on the first retry, let it fail once silently
                    logging.warning('Retry request to {}, last error {}'.format(url, err))
            try:
                method = 'POST' if kwargs.get('data') or kwargs.get('json') else 'GET'
                if 'method' in kwargs:
                    method = kwargs['method']
                    del kwargs['method']
                async with self._session.request(method, url, headers=hdrs, **kwargs) as r:
                    if handler:
                        # Allow a custom response handler to be passed in
                        return await handler(r)
                    if 200 <= r.status < 300:
                        if r.content_type == 'application/json':
                            if response_encoding:
                                return await r.json(encoding=response_encoding)
                            else:
                                return await r.json()
                        else:
                            if response_encoding:
                                return await r.text(encoding=response_encoding)
                            else:
                                return await r.text()
                    msg = await r.text()
                    err = 'Status code={} text={}'.format(r.status, msg)
                    if r.status >= 500:
                        err = r.text
                        continue
                    r.raise_for_status()
            except asyncio.TimeoutError:
                err = 'TimeoutError'
            except aiohttp.ServerDisconnectedError:
                # This is just a regular occurrence.
                # We only log an error on the second retry to avoid this.
                err = 'ServerDisconnectedError'
            except aiohttp.ClientConnectionError as e:
                err = 'ClientConnectionError'
            except aiohttp.ServerDisconnectedError:
                err = 'ServerDisconnectedError'
        raise RuntimeError('Error fetching {}: {}'.format(url, err))


def ignore_aiohttp_ssl_eror(loop, aiohttpversion='3.7.4.post0'):
    """Ignore aiohttp #3535 issue with SSL data after close
    There appears to be an issue on Python 3.7 and aiohttp SSL that throws a
    ssl.SSLError fatal error (ssl.SSLError: [SSL: KRB5_S_INIT] application data
    after close notify (_ssl.c:2609)) after we are already done with the
    connection. See GitHub issue aio-libs/aiohttp#3535

    Given a loop, this sets up a exception handler that ignores this specific
    exception, but passes everything else on to the previous exception handler
    this one replaces.

    If the current aiohttp version is not exactly equal to aiohttpversion
    nothing is done, assuming that the next version will have this bug fixed.
    This can be disabled by setting this parameter to None
    """
    if aiohttpversion is not None and aiohttp.__version__ != aiohttpversion:
        return

    orig_handler = loop.get_exception_handler() or loop.default_exception_handler

    def ignore_ssl_error(loop, context):
        if context.get('message') == 'SSL error in data received':
            # validate we have the right exception, transport and protocol
            exception = context.get('exception')
            protocol = context.get('protocol')
            if (
                isinstance(exception, ssl.SSLError) and exception.reason == 'KRB5_S_INIT' and
                isinstance(protocol, asyncio.sslproto.SSLProtocol) and
                isinstance(protocol._app_protocol, aiohttp.client_proto.ResponseHandler)
            ):  
                if loop.get_debug():
                    asyncio.log.logger.debug('Ignoring aiohttp SSL KRB5_S_INIT error')
                return
        try:
            orig_handler(loop, context)
        except TypeError:
            orig_handler(context)

    loop.set_exception_handler(ignore_ssl_error)

_worked_around_aiohttp_3535 = False
def work_around_aiohttp_3535():
    global _worked_around_aiohttp_3535
    if _worked_around_aiohttp_3535:
        return
    try:
        ignore_aiohttp_ssl_eror(asyncio.get_running_loop())
    except AttributeError:  # predates python 3.7
        ignore_aiohttp_ssl_eror(asyncio.get_event_loop())
    finally:
        _worked_around_aiohttp_3535 = True
