"""Function to sync a Preseem network metadata store to the new Preseem model."""
import asyncio
import logging

# STM-6039 Only allow the sync to run once at a time, to prevent duplicate
# entities from being created.
sync_lock = asyncio.Lock()

async def sync_preseem_model(preseem_model, nm_model, company_uuid, system=None, delete=False, verbose=0):
    """
    Update the Preseem model to keep it in sync with metadata references.
    Right now this is done using the name for Site and the site+ap name for AP.
    In the future we will want to use system and system_id for this.
    Pass in "delete=True" if you want it to assume it owns all APs and delete
    any it doesn't own.
    """
    async with sync_lock:
        model_elems = {(x.site_uuid, x.name): x for x in await preseem_model.Element.List(company_uuid=company_uuid) if x.site_uuid}
        model_sites = {x.name: x for x in await preseem_model.Site.List(company_uuid=company_uuid)}
        logging.info('sync_preseem_model: current model has %s elements assigned to %s sites', len(model_elems), len(model_sites))

        ap_refs = nm_model.refs.get('ap') or {}
        site_names = {}
        sites = {}
        aps = {}
        for ap_ref in ap_refs.values():
            if system == ap_ref.attributes.get('system'):
                elem_name = ap_ref.attributes.get('topology1')
                site_name = ap_ref.attributes.get('topology2')
                if ap_ref.value.startswith('site.'):
                    sites[ap_ref.value[5:]] = site_name
                    site_names[site_name] = ap_ref.value[5:]
                else:
                    aps[ap_ref.value] = ap_ref
                    if site_name not in site_names:
                        site_names[site_name] = None

        if verbose > 1:
            logging.info('Elements:')
            for elem in model_elems.values():
                logging.info(f' {elem}')

        bss_aps = set()
        bss_sites = set()
        if aps:
            for site_name, site_id in site_names.items():
                if site_id is None:
                    sites[site_name] = site_name
            # List aps and sites
            for site in sites.values():
                # Create site if it doesn't exist (by name or source_id)
                if site not in model_sites:
                    if verbose:
                        logging.info('Create site: %s', site)
                    model_sites[site] = await preseem_model.Site.Create(company_uuid=company_uuid, name=site)
                    if verbose:
                        logging.info('Created site: %s', model_sites[site])
                bss_sites.add(site)
            for ap in aps.values():
                # Create ap if it doesn't exist (by site+name or ideally source_id)
                model_site = model_sites.get(ap.attributes.get('topology2'))
                if not model_site:
                    continue  # shouldn't be possible
                elem = model_elems.get((model_site.uuid, ap.attributes['topology1']))
                if not elem:
                    if verbose:
                        logging.info('Create element: %s', ap)
                    r = await preseem_model.Element.Create(company_uuid=company_uuid, site_uuid=model_site.uuid, name=ap.attributes['topology1'], management_ip=ap.attributes.get('ip_address'))
                    if verbose:
                        logging.info('Created element: %s', r)
                else:
                    # check for changes
                    updates = {}
                    if elem.inactive:
                        updates['inactive'] = False
                    if elem.management_ip != ap.attributes.get('ip_address'):
                        updates['management_ip'] = ap.attributes.get('ip_address')
                    if updates:
                        if verbose:
                            logging.info('Update element %s: %s', ap, updates)
                        await preseem_model.Element.Update(company_uuid=company_uuid, uuid=elem.uuid, **updates)
                bss_aps.add((model_site.uuid, ap.attributes['topology1']))

        if aps and delete:  # only do this if we have APs for multi-system deployments
            for elem_id, elem in model_elems.items():
                if elem_id not in bss_aps:
                    if verbose:
                        logging.info('Delete element: %s', elem)
                    await preseem_model.Element.Update(uuid=elem.uuid, company_uuid=elem.company_uuid, inactive=True)

            for site_id, site in model_sites.items():
                if site_id not in bss_sites:
                    if verbose:
                        logging.info('Delete site: %s', site)
                    await preseem_model.Site.Update(uuid=site.uuid, company_uuid=site.company_uuid, inactive=True)
