"""Fake Preseem Model implementation."""
import copy
import time
from uuid import uuid4

from preseem.preseem_model import BaseModelObject

class FakePreseemModel:
    _companies = {} # company_uuid to Company object
    _elements = {}  # company_uuid to {element_uuid to Element object}
    _sites = {}     # company_uuid to {site_uuid to Site object}

    def _new_attrs():
        """Generate attributes for a new preseem-model object."""
        now = time.time()
        uuid = uuid4()
        return {
            'id': uuid.int & (1<<64)-1,
            'uuid': uuid.bytes,
            'created_at': now,
            'updated_at': now,
            'inactive': False,
            'inactive_time': None
        }

    def _set_inactive(obj):
        obj.inactive = True
        obj.updated_at = obj.inactive_time = time.time()

    async def init(self):
        pass

    async def copy_from(self, other):
        """Insert exact copies of entities from another model into this one."""
        for obj in await other.Company.List():
            FakePreseemModel._companies[obj.uuid] = obj
        for obj in await other.Element.List():
            FakePreseemModel._companies[obj.uuid] = obj
        for obj in await other.Site.List():
            FakePreseemModel._companies[obj.uuid] = obj


    """Fake preseem model that implements the python API."""
    class Company(BaseModelObject):
        @classmethod
        async def Create(cls, **kwargs):
            obj = cls(**FakePreseemModel._new_attrs(),
                    name=kwargs.get('name') or '',
                    parent_uuid=kwargs.get('parent_uuid'),
                    old_id=kwargs.get('old_id'))
            FakePreseemModel._companies[obj.uuid] = obj
            return copy.deepcopy(obj)

        @classmethod
        async def Get(cls, **kwargs):
            uuid = kwargs.get('uuid')
            assert uuid
            return copy.deepcopy(FakePreseemModel._companies[uuid])

        @classmethod
        async def GetByOldId(cls, **kwargs):
            old_id = kwargs.get('old_id')
            assert old_id
            companies = [x for x in FakePreseemModel._companies.values() if x.old_id == old_id]
            assert len(companies) < 2
            if companies:
                return copy.deepcopy(companies[0])
            # TODO error for empty?

        @classmethod
        async def List(cls, **kwargs):
            return [copy.deepcopy(x) for x in FakePreseemModel._companies.values()]


    class Element(BaseModelObject):
        @classmethod
        async def Create(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            company = FakePreseemModel._companies[company_uuid]
            obj = cls(**FakePreseemModel._new_attrs(),
                    company_uuid=company.uuid,
                    name=kwargs.get('name'),
                    serial_number=kwargs.get('serial_number'),
                    system_mac_address=kwargs.get('system_mac_address'),
                    imsi=kwargs.get('imsi'),
                    management_ip=kwargs.get('management_ip'),
                    site_uuid=kwargs.get('site_uuid'),
                    status=kwargs.get('status') or 0)
            elems = FakePreseemModel._elements.get(company.uuid)
            if elems is None:
                elems = FakePreseemModel._elements[company_uuid] = {}
            elems[obj.uuid] = obj
            return copy.deepcopy(obj)

        @classmethod
        async def List(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            # the real impl returns empty list if the company doesn't exist
            return [copy.deepcopy(x) for x in (FakePreseemModel._elements.get(company_uuid) or {}).values()]

        @classmethod
        async def Delete(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            elems = FakePreseemModel._elements[company_uuid]
            if elems:
                elem = elems.get(uuid)
                if elem:
                    del elems[uuid]

        @classmethod
        async def Update(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            elems = FakePreseemModel._elements[company_uuid]
            if elems:
                elem = elems.get(uuid)
                for attr, val in kwargs.items():
                    if attr == 'company_uuid' or attr == 'uuid':
                        continue
                    elif attr == 'inactive' and val is True:
                        FakePreseemModel._set_inactive(elem)
                    else:
                        setattr(elem, attr, val)


    class Site(BaseModelObject):
        @classmethod
        async def Create(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            company = FakePreseemModel._companies[company_uuid]
            obj = cls(**FakePreseemModel._new_attrs(),
                    company_uuid=company.uuid,
                    name=kwargs.get('name') or '')
            sites = FakePreseemModel._sites.get(company.uuid)
            if sites is None:
                sites = FakePreseemModel._sites[company_uuid] = {}
            sites[obj.uuid] = obj
            return copy.deepcopy(obj)

        @classmethod
        async def List(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            assert company_uuid
            # the real impl returns empty list if the company doesn't exist
            return [copy.deepcopy(x) for x in (FakePreseemModel._sites.get(company_uuid) or {}).values()]

        @classmethod
        async def Delete(cls, **kwargs):
            company_uuid = kwargs.get('company_uuid')
            uuid = kwargs.get('uuid')
            sites = FakePreseemModel._sites[company_uuid]
            if sites:
                site = sites.get(uuid)
                if site:
                    del elems[uuid]
