"""Test Switch Polling"""
import asyncio
from collections import namedtuple
from ipaddress import IPv4Network, IPv4Address
import os
import sys
import unittest

sys.path.append(os.path.dirname(__file__))  # let code under test load stubs
from fake_context import FakeContext
from fake_snmp import FakeSnmpClient
from ne import NetworkElementRegistry
from router import RouterPoller, Route
from preseem import NetworkMetadataReference
from preseem.network_element_update import Interface
import switch

class Switch(switch.Switch):
    """Extend the Switch object to add some test synchronization helpers"""
    async def poll(self):
        FakeContext.ap_event.clear()
        await super().poll()
        FakeContext.ap_event.set()

class FakeRouterPoller:
    """Fake class to provide minimum of what these tests need for RouterPoller."""
    def __init__(self, ctx, id, host):
        pass
    async def poll_standard(self, device):
        self.ifs = {
            1: Interface(id='1', mac_address='01:02:03:01:01:01'),
            2: Interface(id='2', mac_address='01:02:03:01:01:02'),
            3: Interface(id='3', mac_address='01:02:03:01:01:03'),
        }
        self.routes = {
            IPv4Network('0.0.0.0/0'): Route(net=IPv4Network('0.0.0.0/0'), if_index=2, next_hop=IPv4Address('192.0.2.1'), mac_address='01:02:03:05:00:01')
        }

class FakeRouterPollerNoRoutes(FakeRouterPoller):
    async def poll_standard(self, device):
        await super().poll_standard(device)
        self.routes = {}

class TestSwitch(unittest.TestCase):
    def setUp(self):
        self.ctx = FakeContext()
        self.loop = asyncio.new_event_loop() # needed to initialize asyncio
        self.reg = NetworkElementRegistry(self.ctx, {}, Switch)
        self.reg.ne_type = 'switch'
        asyncio.set_event_loop(self.loop)
        self._await(self.ctx.start())

    def tearDown(self):
        self._await(self.reg.close())
        self._await(self.ctx.close())
        self.loop.close()

    def _await(self, co):
        return self.loop.run_until_complete(asyncio.wait_for(co, timeout=1.0))

    def wait_for_poll(self):
        """Wait for a switch poll to complete."""
        self._await(self.ctx.ap_event.wait())

    #TEST:
        # old and new mikrotik examples (real) for one time brtbl read, check refs
        # then make a fake switch to test expiry and add/delete cases. 
        # expired mac gets added to a different switch 

    def test_switch_poll_old_mikrotik(self):
        """Poll a Mikrotik before bridge port translation firmware."""
        ne = self._await(self.reg.set('Switch1', {'name': 'Switch 1', 'host': 'TEST', 'site': '', 'path': 'test/data/mikrotik.mikrotik-switch.RB941-2nD.6.46.7.01.snmp'}))
        self.wait_for_poll()
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '10:0c:6b:65:fc:14': NetworkMetadataReference(type='net_mac', value='10:0c:6b:65:fc:14', attributes={'switch': 'Switch1'})
        })

    def test_switch_poll_new_mikrotik(self):
        """Poll a Mikrotik after bridge port translation firmware."""
        ne = self._await(self.reg.set('Switch1', {'name': 'Switch 1', 'host': 'TEST', 'site': '', 'path': 'test/data/mikrotik.mikrotik-switch.RBD52G-5HacD2HnD.6.48.6.01.snmp'}))
        self.wait_for_poll()
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '3c:84:6a:ee:d0:05': NetworkMetadataReference(type='net_mac', value='3c:84:6a:ee:d0:05', attributes={'switch': 'Switch1'}),
            '00:0b:82:f3:f3:f9': NetworkMetadataReference(type='net_mac', value='00:0b:82:f3:f3:f9', attributes={'switch': 'Switch1'})
        })

    def test_mac_expiry(self):
        """A MAC is added to the bridge table.  It gets mapped, then deleted."""
        fake_bridge_table = {}
        async def read_bridge_table(sysobjid=None):
            return fake_bridge_table
        dev = FakeSnmpClient({}, None)
        dev.read_bridge_table = read_bridge_table
        dev.connection_lost = False
        switch = Switch(self.ctx, 'test-sw-1-id', 'test-sw-1', 'site-1', dev, None, 'switch', 0)
        switch.router_poller_cls = FakeRouterPoller

        # A new bridge entry gets mapped.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01', '01:02:03:04:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

        # Delete the bridge entry.  It is still mapped.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

        # The bridge entry is removed after expiry
        switch.TIMEOUT = 0
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {})

    def test_mac_changes_port(self):
        """A MAC changes from one port to another on the switch."""
        fake_bridge_table = {}
        async def read_bridge_table(sysobjid=None):
            return fake_bridge_table
        dev = FakeSnmpClient({}, None)
        dev.read_bridge_table = read_bridge_table
        dev.connection_lost = False
        switch = Switch(self.ctx, 'test-sw-1-id', 'test-sw-1', 'site-1', dev, None, 'switch', 0)
        switch.router_poller_cls = FakeRouterPoller

        # A new bridge entry gets mapped.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01', '01:02:03:04:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

        # The port changes.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01'},
            2: {'01:02:03:05:00:01'},
            3: {'01:02:03:04:01:01'},
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

        # The bridge entry is still mapped after expiry
        switch.TIMEOUT = 0
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

    def test_net_mac_ref_deleted(self):
        """The net_mac reference is deleted.  It is replaced."""
        fake_bridge_table = {}
        async def read_bridge_table(sysobjid=None):
            return fake_bridge_table
        dev = FakeSnmpClient({}, None)
        dev.read_bridge_table = read_bridge_table
        dev.connection_lost = False
        switch = Switch(self.ctx, 'test-sw-1-id', 'test-sw-1', 'site-1', dev, None, 'switch', 0)
        switch.router_poller_cls = FakeRouterPoller

        # A new bridge entry gets mapped.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01', '01:02:03:04:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

        # The reference gets deleted.
        self._await(self.ctx.netmeta_model.del_ref(NetworkMetadataReference('net_mac', '01:02:03:04:01:01', {})))
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {})

        # The reference is recreated
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

    def test_net_mac_ref_changed(self):
        """A MAC changes from one switch to another.  It is left mapped to the active switch."""
        fake_bridge_table = {}
        async def read_bridge_table(sysobjid=None):
            return fake_bridge_table
        dev = FakeSnmpClient({}, None)
        dev.read_bridge_table = read_bridge_table
        dev.connection_lost = False
        switch = Switch(self.ctx, 'test-sw-1-id', 'test-sw-1', 'site-1', dev, None, 'switch', 0)
        switch.router_poller_cls = FakeRouterPoller

        # A new bridge entry gets mapped.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01', '01:02:03:04:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

        # The MAC moves to another switch which maps it.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('net_mac', '01:02:03:04:01:01', {'switch': 'test-sw-2-id'})))
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference('net_mac', '01:02:03:04:01:01', {'switch': 'test-sw-2-id'})
        })

        # The reference is left alone
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference('net_mac', '01:02:03:04:01:01', {'switch': 'test-sw-2-id'})
        })

    def test_no_mapping_if_no_upstream_port(self):
        """A switch where the upstream port is not known is not mapped."""
        fake_bridge_table = {}
        async def read_bridge_table(sysobjid=None):
            return fake_bridge_table
        dev = FakeSnmpClient({}, None)
        dev.read_bridge_table = read_bridge_table
        dev.connection_lost = False
        switch = Switch(self.ctx, 'test-sw-1-id', 'test-sw-1', 'site-1', dev, None, 'switch', 0)
        switch.router_poller_cls = FakeRouterPollerNoRoutes

        # A new bridge entry does not get mapped.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01', '01:02:03:04:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {})
        return

    def test_mappings_removed_from_offline_switch(self):
        """Bridge mappings are removed if the switch goes offline."""
        fake_bridge_table = {}
        async def read_bridge_table(sysobjid=None):
            return fake_bridge_table
        dev = FakeSnmpClient({}, None)
        dev.read_bridge_table = read_bridge_table
        dev.connection_lost = False
        switch = Switch(self.ctx, 'test-sw-1-id', 'test-sw-1', 'site-1', dev, None, 'switch', 0)
        switch.router_poller_cls = FakeRouterPoller

        # A new bridge entry gets mapped.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01', '01:02:03:04:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

        # Switch goes offline, all MACs are removed.
        fake_bridge_table = {}
        switch.offline = True
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {})

    def test_old_mappings_removed(self):
        """Initial net_mac references for the switch are removed (after expiry)."""
        fake_bridge_table = {}
        async def read_bridge_table(sysobjid=None):
            return fake_bridge_table
        dev = FakeSnmpClient({}, None)
        dev.read_bridge_table = read_bridge_table
        dev.connection_lost = False
        switch = Switch(self.ctx, 'test-sw-1-id', 'test-sw-1', 'site-1', dev, None, 'switch', 0)
        switch.router_poller_cls = FakeRouterPoller

        # A bridge entry is read from metadata
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('net_mac', '01:02:03:04:01:01', {'switch': 'test-sw-1-id'})))
        fake_bridge_table = {
            1: {'01:02:03:01:01:01'},
            2: {'01:02:03:05:00:01'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })

        # The bridge entry is removed after expiry
        switch.TIMEOUT = 0
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {})

    def test_upstream_port_by_cpe_mac(self):
        """PPA-299 upstream port fallback by cpe_mac"""
        fake_bridge_table = {}
        async def read_bridge_table(sysobjid=None):
            return fake_bridge_table
        dev = FakeSnmpClient({}, None)
        dev.read_bridge_table = read_bridge_table
        dev.connection_lost = False
        switch = Switch(self.ctx, 'test-sw-1-id', 'test-sw-1', 'site-1', dev, None, 'switch', 0)
        switch.router_poller_cls = FakeRouterPollerNoRoutes
        # map our upstream mac as a bridged mac of some upstream access device.
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('cpe_mac', '01:02:03:01:01:02', {'ap': 'ap01', 'sm': 'aa:bb:cc:dd:ee:ff'})))
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}

        # A new bridge entry does not get mapped.
        fake_bridge_table = {
            1: {'01:02:03:01:01:01', '01:02:03:04:01:01'},
            2: {'01:02:03:05:00:01', '01:02:03:01:01:02'}
        }
        self._await(switch.poll())
        net_mac_refs = self.ctx.netmeta_model.refs.get('net_mac') or {}
        self.assertEqual(net_mac_refs, {
            '01:02:03:04:01:01': NetworkMetadataReference(type='net_mac', value='01:02:03:04:01:01', attributes={'switch': 'test-sw-1-id'})
        })
        return
