import unittest
from site_filter import SiteFilter  # Replace with actual module name


class TestSiteFilter(unittest.TestCase):

    def test_default_all_included(self):
        f = SiteFilter({})
        self.assertTrue(f.should_include_site("anything"))
        self.assertTrue(f.should_include_site("Another-Site"))

    def test_include_only(self):
        f = SiteFilter({"include": ["site-*", "region-1"]})
        self.assertTrue(f.should_include_site("site-main"))
        self.assertTrue(f.should_include_site("region-1"))
        self.assertFalse(f.should_include_site("other"))

    def test_exclude_only(self):
        f = SiteFilter({"exclude": ["bad-*", "test"]})
        self.assertFalse(f.should_include_site("bad-site"))
        self.assertFalse(f.should_include_site("test"))
        self.assertTrue(f.should_include_site("good-site"))

    def test_include_and_exclude(self):
        f = SiteFilter({
            "include": ["site-*", "region-a"],
            "exclude": ["site-test", "region-a-deprecated"]
        })
        self.assertTrue(f.should_include_site("site-main"))
        self.assertFalse(
            f.should_include_site("site-test"))  # excluded even though matches include
        self.assertTrue(f.should_include_site("region-a"))
        self.assertFalse(f.should_include_site("region-a-deprecated"))
        self.assertFalse(f.should_include_site("other"))  # not in includes

    def test_case_insensitivity(self):
        f = SiteFilter({"include": ["SITE-*"], "exclude": ["SITE-BAD"]})
        self.assertTrue(f.should_include_site("site-good"))
        self.assertFalse(f.should_include_site("SITE-BAD"))
        self.assertFalse(f.should_include_site("bad-site"))

    def test_invalid_config_not_a_dict(self):
        with self.assertRaises(ValueError):
            SiteFilter("not-a-dict")

    def test_invalid_include_type(self):
        with self.assertRaises(ValueError):
            SiteFilter({"include": "not-a-list"})

    def test_invalid_exclude_type(self):
        with self.assertRaises(ValueError):
            SiteFilter({"exclude": 123})

    def test_unrecognized_key(self):
        with self.assertRaises(ValueError):
            SiteFilter({"foo": ["bar"]})


if __name__ == "__main__":
    unittest.main()
