"""Test the full NE/AP infrastructure together."""
import asyncio
import os.path
import sys
import unittest

from preseem import FakeNetworkMetricsModel, NetworkMetadataReference

sys.path.append(os.path.dirname(__file__))  # let code under test load stubs
import ap
import ap_data
from fake_snmp import FakeSnmpClient
from ne import BaseDevice, NetworkElementRegistry
from devices.ubnt import snmpBadMibError, snmpBadUpdateError, snmpUnsupportedFwError
from fake_context import FakeContext, FakeHttpClient
from preseem.network_element_update import NetworkElement, NetworkElementUpdate, ErrorContext
from fake_device import FakeDevice
import plugin
from uuid import UUID
from preseem_protobuf.network_poller import network_poller_pb2

class Ap(ap.Ap):
    """We subclass the Ap object to add some test synchronization helpers"""
    async def poll(self):
        FakeContext.ap_event.clear()
        await super().poll()
        FakeContext.ap_event.set()

class FakeModule:
    def __init__(self, device):
        self.device = device
        self.stations = []
        self.model = None
        self.mode = "ap"
        self.station_poller = None
        self.polled = False

    async def init(self):
        pass

    async def poll(self):
        self.polled = True


class TestNeAp(unittest.TestCase):
    def setUp(self):
        apdd = ap_data.ApData('ap_info.yaml')
        self.ctx = FakeContext()
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        self.loop = asyncio.new_event_loop() # needed to initialize asyncio
        self.reg = NetworkElementRegistry(self.ctx, {}, Ap)
        self.reg.ne_type = 'ap'
        asyncio.set_event_loop(self.loop)
        self._await(self.ctx.start())

    def tearDown(self):
        self._await(self.reg.close())
        self._await(self.ctx.close())
        self.loop.close()
        plugin._dm = None

    def _await(self, co):
        return self.loop.run_until_complete(co)

    def wait_for_ap_poll(self):
        """Wait for an AP poll to complete."""
        self._await(self.ctx.ap_event.wait())

    def wait_for_neu_posted(self):
        """Wait for a NetworkElementUpdateMessage to be posted and return it."""
        return self._await(asyncio.wait_for(self.ctx.network_poller_grpc.fut, timeout=1.0))

    def test_netconf_module(self):
        """PPA-1278 test plugin infrastructure changes for netconf-based polling."""
        FakeHttpClient.disable_http = True  # Disabling HTTP
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site'}))
        self.wait_for_ap_poll()
        self.assertIsNone(ne.module)  # nothing matches the default case
        self.assertIsInstance(ne.snmp_ne, FakeSnmpClient)  # it was reset

        match_device = None
        async def match(dev):
            nonlocal match_device
            match_device = dev
            return FakeModule(dev)

        # Try another run with my fake module in, to make sure it can work.
        plugin._dm.modules['test'] = match
        ne.last_match_time = None
        self._await(ne.poll())
        self.wait_for_ap_poll()
        self.assertIsInstance(match_device, BaseDevice)
        self.assertTrue(ne.module.polled)
