"""Test Ubiquiti specific code."""
import asyncio
from binascii import unhexlify
from collections import namedtuple
from dataclasses import dataclass
import ipaddress
import unittest
from unittest import mock

from test.fake_device import OctetString
from base_snmp_client import SnmpClientBase
from devices import ubnt
from ne import HttpCredentials


@dataclass
class FakeStation:
    """Class for manipulating a fake station."""
    mac_address: str
    name: str
    ip_address: str
    mode: str

    def __init__(self, mac_address, name, ip_address):
        self.http = FakeHttpCpe()
        self.mac_address = mac_address
        self.name = name
        self.ip_address = ip_address
        self.mode = 'bridge'

    def update_ip(self, ip_address):
        self.http = FakeHttpCpe()
        self.ip_address = ip_address

    def _replace(self, ip_address):
        return FakeStation(self.mac_address, self.name, ip_address)


class FakeHttpClient:
    """Fake HTTP client stub."""

    async def request(self,
                      path,
                      num_retries=5,
                      handler=None,
                      response_encoding=None,
                      **kwargs):
        """Stub http request method."""
        #print("REQ", path, handler, kwargs)

        if path == '/status.cgi':
            r = {
                'host': {
                    'uptime': 1586080,
                    'fwversion': 'v6.0.4',
                    'hostname': 'Hutch R&amp;B 136 W RM5 120',
                    'devmodel': 'Rocket M5 ',
                },
                'wireless': {
                    'mode': 'ap',
                    'essid': 'PRWN-HCRB-W',
                    'apmac': '00:27:22:01:02:03',
                    'frequency': '5205 MHz',
                    'signal': -67,
                    'rssi': 29,
                    'noisef': -102,
                    'txpower': 20,
                    'distance': 3300,
                    'count': 1,
                    'chanbw': 20,
                },
                'interfaces': [{
                    'ifname': 'eth0',
                    'hwaddr': '44:D9:E7:D7:5F:58',
                    'enabled': True,
                    'status': {
                        'plugged': 1,
                        'speed': 100,
                    },
                }, {
                    'ifname': 'ath0',
                    'hwaddr': '00:27:22:01:02:03',
                    'enabled': True,
                    'status': {
                        'plugged': 1,
                        'speed': 0,
                    },
                }]
            }
            return r
        elif path == '/sta.cgi':
            r = [{
                'mac': 'F0:9F:C2:5C:53:3B',
                'name': 'KBlankenship-P',
                'lastip': '172.16.123.204',
                'stats': {
                    'rx_bytes': 3242596639,
                    'tx_bytes': 96457526411,
                },
                'remote': {
                    'uptime': 20180988,
                    'hostname': 'KBlankenship-PBM5',
                    'platform': 'PowerBeam M5 400',
                    'version': 'XW.ar934x.v6.0.4.30805.170505.1510',
                }
            }]
            return r


class FakeHttpCpe:

    def __init__(self):
        self.apmac = '00:27:22:01:02:03'

    async def request(self,
                      path,
                      num_retries=5,
                      handler=None,
                      response_encoding=None,
                      **kwargs):
        """Stub http request method."""

        if path == '/status.cgi':
            self.r = {
                'wireless': {
                    'apmac': self.apmac,
                }
            }
            return self.r

        elif path == '/brmacs.cgi':
            brmacs_data = {
                'brmacs': [{
                    'bridge': 'br0',
                    'port': 'ath0',
                    'hwaddr': '00:15:6d:9a:f6:83',
                    'ageing': '34.02'
                }, {
                    'bridge': 'br0',
                    'port': 'ath0',
                    'hwaddr': '00:27:22:2a:13:46',
                    'ageing': '36.47'
                }]
            }
            return brmacs_data


class FakeDevice:
    """Fake device stub to pass to the module."""
    sysobjid = '1.3.6.1.4.1.41112.1.4'

    def __init__(self, host):
        self.cfg = {}
        self.http_credentials = [HttpCredentials('admin', 'password', None, None)]
        self.host = host
        self.http = FakeHttpClient()
        self.stations = []
        self.uptime = 1000

    def get_cfg(self, attr):
        """Stub get_cfg method."""
        return self.cfg.get(attr)

    async def get_vars(self, name, oidmap, decode_strings=None, bulk_size=45):
        """Stub get_vars method."""
        Vars = namedtuple(name, sorted(oidmap))
        if name == 'UbntApInfo':
            return Vars(None, None, None, None, None, None, None, None, 2)

    async def snmp_get(self, oid):
        """Fake snmp_get implementation."""
        pass

    async def read_if_table(self, counters=False, ifxtable=False, intfobj=False):
        """Stub method to return if info"""
        return [
            SnmpClientBase.IfEntry(1, 'lo', 24, 16436, 0, None, 1, 1, None, 0, 0, 0, 0,
                                   0, 0, 0, 0, 0, 0, 0),
            SnmpClientBase.IfEntry(2, 'eth0', 6, 1500, 100000000, '00:27:22:01:02:03',
                                   1, 1, None, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
            SnmpClientBase.IfEntry(3, 'ath0', 6, 1500, 130000000, '00:27:22:00:02:03',
                                   1, 1, None, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
        ]

    async def read_table3(self, name, oid, colmap, bulkwalk=None, bulksize=None):
        """Fake read_table3 implementation."""
        Row = namedtuple(name, colmap.values())
        if oid == ubnt.dot11ResourceInfoTable:
            # Row = namedtuple(name, ('manufacturer', 'name', 'version'))
            return [
                Row(OctetString(b'Ubiquiti Networks, Inc.'), OctetString(b'Rocket M5'),
                    OctetString(b'XW.ar934x.v6.2.0.33033.190703.1117'))
            ]
        if oid == ubnt.ubntRadioTable:
            return [Row(5765, 27)]
        if oid == ubnt.ubntWlStatTable:
            return [
                Row(OctetString(b'TestSSID'), b'\x00\x27\x22\x01\x02\x03', -67, 29, 91,
                    -90, 105016000, 106133000, 2, 20, 1)
            ]
        if oid == ubnt.ubntAirMaxTable:
            return [Row(1, 66, 68, 3, 2, None, None, None)]
        if oid == ubnt.ubntStaTable:
            return [
                Row(unhexlify(x.mac_address.replace(':', '')),
                    OctetString(x.name.encode()), -67, -90, 6150, 98, 0, 61, 61,
                    ipaddress.ip_address(x.ip_address).packed, 117000000, 78000000,
                    100000, 10000, 999, None, None, None, None, None, None)
                for x in self.stations
            ]

    async def snmp_walk2(self, oid, bulksize=None):
        """Fake snmp_walk implementation."""
        if oid == ubnt.dot11StationID:
            return {(5, ): b'\x00\x27\x22\x01\x02\x03'}


class TestUbntStubs(unittest.TestCase):
    """Tests for Ubiquiti code that use stubs"""

    def setUp(self):
        """Setup a test."""
        self.loop = asyncio.new_event_loop()  # needed to initialize asyncio
        asyncio.set_event_loop(self.loop)

    def tearDown(self):
        """Teardown a test."""
        self.loop.close()

    def _await(self, co):
        """hook into eventloop."""
        return self.loop.run_until_complete(co)

    def test_01(self):
        """Sample test."""
        # Setup the test
        dev = FakeDevice('192.0.2.2')
        # Force to use SNMP by disabling HTTP
        dev.http = None
        dev.stations = [FakeStation('f0:9f:c2:00:02:03', 'Station Name', '192.0.2.101')]
        m = ubnt.UbiquitiAP(dev)
        self._await(m.init())
        self._await(m.poll_config())

        # Do the first poll and check the result
        self._await(m.poll())
        self.assertEqual(len(m.stations), 1)
        self.assertEqual(m.stations[0].ip_address, '192.0.2.101')

        # Now make changes and do more polls.
        dev.stations[0].ip_address = '192.0.2.201'
        dev.uptime += 60
        self._await(m.poll())
        self.assertEqual(m.stations[0].ip_address, '192.0.2.201')

    def test_poll_station_01(self):
        """IP Flapping cases: No HTTP client present case
            i. within holdoff_interval. Fallback to lastIP.
            ii. holdoff_interval expired. Pick newIP."""
        # Setup the test
        dev = FakeDevice('192.0.2.2')
        Sta = namedtuple('Sta', ('mac_address', 'name', 'ip_address', 'mode'))
        dev.stations = [Sta('f0:9f:c2:5c:53:3b', 'Station Name', '192.0.2.101', None)]
        m = ubnt.UbiquitiAP(dev)
        self._await(m.init())
        self._await(m.poll_config())
        self._await(m.poll())
        # Run poll_station to update IP
        sta = self._await(m.poll_station(dev.stations[0], None, None))

        # Let initial IP expire, then assign a new one
        m.LAST_IP_HOLDOFF_TIME = 0
        sta = sta._replace(ip_address='192.0.2.108')
        sta = self._await(m.poll_station(sta, None, None))
        # Verify that IP did update as lastIP expired
        self.assertEqual(sta.ip_address, '192.0.2.108')

        # Now try assigning new one before the current one expires
        m.LAST_IP_HOLDOFF_TIME = 10
        sta = dev.stations[0]._replace(ip_address='192.0.2.103')
        sta = self._await(m.poll_station(sta, None, None))
        # Verify that IP didn't FLAP
        self.assertEqual(sta.ip_address, '192.0.2.108')

    def test_poll_station_02(self):
        """IP Flapping case, where station IP is set None."""
        dev = FakeDevice('172.16.123.233')
        m = ubnt.UbiquitiAP(dev)
        self._await(m.init())
        self._await(m.poll_config())
        self._await(m.poll())
        # Run initial poll_station
        sta = FakeStation('f0:9f:c2:5c:53:3b', 'Station Name', '172.16.123.204')
        sta = self._await(m.poll_station(sta, None, sta.http))
        # CASE1: Valid lastIP, so fallback to that
        sta.ip_address = None
        sta = self._await(m.poll_station(sta, None, None))
        # Verify that station IP falls back to last IP as its valid
        self.assertEqual(sta.ip_address, '172.16.123.204')

        # CASE2: lastIP expired, so set stationIP to None
        m.LAST_IP_HOLDOFF_TIME = 0
        sta.ip_address = None
        sta = self._await(m.poll_station(sta, None, None))
        # Verify that station IP falls back to last IP as its valid
        self.assertEqual(sta.ip_address, None)

    def test_poll_station_03(self):
        """IP Flapping case, with http_client flapping:
        i. holdoff_interval expired. Pick newIP."""
        # Setup the test
        dev = FakeDevice('172.16.123.233')
        dev.stations = [
            FakeStation('f0:9f:c2:5c:53:3b', 'KBlankenship-PBM5', '172.16.123.204')
        ]
        m = ubnt.UbiquitiAP(dev)
        self._await(m.init())
        self._await(m.poll_config())
        # Do the first poll and check the result
        self._await(m.poll())
        self.assertEqual(len(m.stations), 1)
        self.assertEqual(m.stations[0].ip_address, '172.16.123.204')

        # CASE: Sta_IP changes and old_IP expires (TRUE change)
        # Run initial poll_station
        sta = self._await(m.poll_station(dev.stations[0], None, dev.stations[0].http))
        # Requirements:
        m.LAST_IP_HOLDOFF_TIME = 0
        sta.update_ip('1.1.1.1')
        # Run poll_station replicating change of IP
        sta = self._await(m.poll_station(sta, None, sta.http))
        # Verify the station IP
        self.assertEqual(sta.ip_address, '1.1.1.1')

        # CASE:2.1.1: Sta_IP changes, cur_client APMAC check failed, old_IP valid
        # Run initial poll_station
        sta = FakeStation('f0:9f:c2:5c:53:3b', 'Station Name', '172.16.123.204')
        sta = self._await(m.poll_station(sta, None, sta.http))
        # Requirements:
        m.LAST_IP_HOLDOFF_TIME = 10
        sta.update_ip('1.1.1.1')
        sta.http.apmac = '00:27:22:ff:ff:ff'
        # Run poll_station replicating change of IP
        sta = self._await(m.poll_station(sta, None, sta.http))
        # Verify that station IP falls back to last IP using last client
        self.assertEqual(sta.ip_address, '172.16.123.204')

        # CASE:2.1.2: Sta_IP changes, both clients failed APMAC check
        # This would only happen when station IP changes and last_client APMAC check fails
        sta = FakeStation('f0:9f:c2:5c:53:3b', 'Station Name', '172.16.123.204')
        last_client = sta.http
        sta = self._await(m.poll_station(sta, None, last_client))
        # Requirements:
        sta.update_ip('1.1.1.1')
        sta.http.apmac = '00:27:22:ff:ff:ff'
        # Replicating that last_client's failure by setting APMAC to None
        last_client.apmac = None
        # Run poll_station replicating change of IP
        sta = self._await(m.poll_station(sta, None, sta.http))
        # Verify that station IP falls back to last IP using last client
        self.assertEqual(sta.ip_address, '172.16.123.204')

        # Case-2.2: No last_client or same as current client or last_client expired.
        # CASE:2.2.1 No useful client. But, valid last IP
        # Restart scenario
        m = ubnt.UbiquitiAP(dev)
        self._await(m.init())
        self._await(m.poll_config())
        self._await(m.poll())
        # Run initial poll_station
        sta = FakeStation('f0:9f:c2:5c:53:3b', 'Station Name', '172.16.123.204')
        sta = self._await(m.poll_station(sta, None, sta.http))
        # cur_client SETUP. Replicate APMAC mismatch on cur_client
        # Keep client same(don't use sta.update_ip) and keep lastIP valid
        sta.ip_address = '1.1.1.1'
        sta.http.apmac = '00:27:22:ff:ff:ff'
        sta = self._await(m.poll_station(sta, None, sta.http))
        # Verify that station IP falls back to last IP as its valid
        self.assertEqual(sta.ip_address, '172.16.123.204')

        # CASE:2.2.2: No valid client. Last IP expired.
        # Restart scenario
        m = ubnt.UbiquitiAP(dev)
        self._await(m.init())
        self._await(m.poll_config())
        self._await(m.poll())
        # Run initial poll_station
        sta = FakeStation('f0:9f:c2:5c:53:3b', 'Station Name', '172.16.123.204')
        sta = self._await(m.poll_station(sta, None, sta.http))
        # cur_client SETUP. Replicate APMAC mismatch on cur_client
        # Create new client
        sta.update_ip('1.1.1.1')
        # Expire lastIP
        m.LAST_IP_HOLDOFF_TIME = 0
        sta.http.apmac = '00:27:22:ff:ff:ff'
        sta = self._await(m.poll_station(sta, None, sta.http))
        # Verify that station IP is changed to NewIP as last IP expired
        self.assertEqual(sta.ip_address, '1.1.1.1')