import asyncio
import multiprocessing
import os
import sys
import unittest
from uuid import UUID

from preseem import NetworkMetadataReference, FakePreseemModel
from preseem.source_model import Account, Service
from preseem_protobuf.model.account_pb2 import AccountStatus

sys.path.append(os.path.dirname(__file__))  # let code under test load stubs
from fake_context import FakeContext
from service import ServiceSyncer

# uncomment this to see log output
#import logging
#logging.basicConfig(format='%(asctime)s %(levelname)s %(message)s',
#                        level=logging.INFO)

class TestServiceSync(unittest.TestCase):
    """Test the service syncer code.  For simplicity, we run both the producer and
       consumer sides in the same process."""
    def setUp(self):
        self._task = None
        self.ctx = FakeContext()
        self.ctx.service_q = multiprocessing.Queue()
        self.ctx.preseem_model = FakePreseemModel()
        self.loop = asyncio.new_event_loop() # needed to initialize asyncio
        asyncio.set_event_loop(self.loop)
        self._await(self.ctx.start())
        self._await(self.ctx.preseem_model.init())
        self.company = self._await(self.ctx.preseem_model.Company.Create(name='test-company'))
        self.ctx.company_uuid = UUID(bytes=self.company.uuid)

        self.service_syncer = ServiceSyncer(self.ctx)
        self.service_syncer.HOLDOFF_TIME = 0.5  # make the tests go fast

    def tearDown(self):
        if self._task:
            self._task.cancel()
            self._await(self._task)
        self.loop.close()

    def _await(self, co):
        return self.loop.run_until_complete(co)

    async def _run_service_sync(self):
        try:
            await self.service_syncer.run()
        finally:
            logging.info("Exit Service Sync")

    def test_service_sync(self):
        """Basic test of functionality: make sure services can be synced from >! id."""
        self._task = self.loop.create_task(self.service_syncer.run())
        self._await(asyncio.sleep(0)) # yield eventloop so it can start

        # setup something to Put (from two different sources)
        account = Account(id='a100', name='Account 100', status=AccountStatus.ACCOUNT_STATUS_ACTIVE)
        service1 = Service(id='s100', account_id='a100', subscriber_identifier='a100.s100')
        service2 = Service(id='s101', account_id='a100', subscriber_identifier='a100.s101')
        service1.download_rate = 100000
        service1.upload_rate = 20000
        service1.attachment.username = 's100'
        service2.download_rate = 500000
        service2.upload_rate = 100000
        service2.attachment.username = 's101'

        # Set the services to the syncer and let it run
        self._await(self.service_syncer.set('router.1', [account], [service1]))
        self._await(self.service_syncer.set('router.2', [account], [service2]))
        self._await(asyncio.sleep(2)) # let the syncer run past its holdoff

        # Check attributes are set
        account_ref = NetworkMetadataReference(type='account', value='a100', attributes={'subscriber_identifier': 'a100', 'subscriber_name': 'Account 100', 'system': 'net'})
        service1_ref = NetworkMetadataReference(type='service', value='s100', attributes={'account': 'a100', 'system': 'net', 'service_rate_down': 100000, 'service_rate_up': 20000, 'username': 's100'})
        service2_ref = NetworkMetadataReference(type='service', value='s101', attributes={'account': 'a100', 'system': 'net', 'service_rate_down': 500000, 'service_rate_up': 100000, 'username': 's101'})
        account_refs = self.ctx.netmeta_model.refs.get('account') or {}
        service_refs = self.ctx.netmeta_model.refs.get('service') or {}
        self.assertEqual(account_refs.get('a100'), account_ref)
        self.assertEqual(service_refs.get('s100'), service1_ref)
        self.assertEqual(service_refs.get('s101'), service2_ref)

        # Check model is set
        accounts = self._await(self.ctx.preseem_model.Account.List(company_uuid=self.company.uuid))
        self.assertEqual(len(accounts), 1)
        self.assertEqual(accounts[0].inactive, False)
        self.assertEqual(accounts[0].source, 'net')
        self.assertEqual(accounts[0].source_id, account.id)
        self.assertEqual(accounts[0].type, None)
        self.assertEqual(accounts[0].status, account.status)
        self.assertEqual(accounts[0].name, account.name)
        services = self._await(self.ctx.preseem_model.Service.List(company_uuid=self.company.uuid))
        self.assertEqual(len(services), 2)
        s1 = next(iter(x for x in services if x.source_id == service1.id), None)
        s2 = next(iter(x for x in services if x.source_id == service2.id), None)
        self.assertIsNotNone(s1)
        self.assertIsNotNone(s2)
        self.assertEqual(s1.inactive, False)
        self.assertEqual(s1.source, 'net')
        self.assertEqual(s1.download_rate, service1.download_rate)
        self.assertEqual(s1.upload_rate, service1.upload_rate)
        self.assertEqual(s1.attachment.username, service1.attachment.username)
        self.assertEqual(s2.inactive, False)
        self.assertEqual(s2.source, 'net')
        self.assertEqual(s2.download_rate, service2.download_rate)
        self.assertEqual(s2.upload_rate, service2.upload_rate)
        self.assertEqual(s2.attachment.username, service2.attachment.username)
