import asyncio
from collections import namedtuple
import logging
import os
import sys
import unittest

sys.path.append(os.path.dirname(__file__))  # let code under test load stubs
import olt
from preseem import NetworkMetadataReference, Reference
from preseem.network_element_update import NetworkElementUpdate
from fake_context import FakeContext

#logging.basicConfig(format='%(asctime)s %(levelname)s %(message)s',
#                        level=logging.INFO)

class Olt(olt.Olt):
    """We subclass the Olt object to add some test synchronization helpers"""
    async def poll(self):
        FakeContext.ap_event.clear()
        await super().poll()
        FakeContext.ap_event.set()

class FakeClient:
    def __init__(self, host=None):
        self.dft = {}
        self.host = host

class FakeModule:
    def __init__(self):
        self.stations = []
        self.model = None
        self.mode = "ap"  # I'm not sure yet if this should be changed to olt.
        self.next_neu = NetworkElementUpdate()

    async def poll(self):
        self.network_element_update_msg = self.next_neu


class TestOLT(unittest.TestCase):
    def setUp(self):
        self.ctx = FakeContext()
        self.loop = asyncio.new_event_loop() # needed to initialize asyncio
        asyncio.set_event_loop(self.loop)
        self._await(self.ctx.start())

    def tearDown(self):
        self._await(self.ctx.close())
        self.loop.close()

    def _await(self, co):
        return self.loop.run_until_complete(co)

    def test_olt_poll(self):
        """Test polling of an OLT element type."""
        olt = Olt(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'olt', None)
        olt.module = FakeModule()
        self._await(olt.load())

        # For now we're just testing some very basic functionality.
        topology1_snmp_refs = self.ctx.netmeta_model.refs.get('topology1_snmp') or {}
        neu = getattr(olt, 'network_element_update_msg', None)
        self.assertEqual(len(topology1_snmp_refs), 1)
        self.assertIsNotNone(neu)

    def test_new_pon(self):
        """Network metadata is setup correctly for a PON."""
        olt = Olt(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'olt', None)
        olt.module = FakeModule()
        pon_port = olt.module.next_neu.data.interfaces.add()
        pon = pon_port.pons.add()
        pon.id = '100001'
        self._await(olt.load())
        exp_ref = NetworkMetadataReference(type='ap', value='TestAP.100001', attributes={'topology1': 'PON.100001', 'topology2': 'OLT.TestAP', 'ap_config': 'TestAP.100001', 'topology1_snmp': 'TestAP', 'system': 'net'})
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 1)
        self.assertEqual(ap_refs.get('TestAP.100001'), exp_ref)

    def test_del_pon(self):
        """Network metadata is updated when a PON is removed or when an OLT goes offline."""
        olt = Olt(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'olt', None)
        olt.module = FakeModule()
        slot_1 = olt.module.next_neu.data.modules.add()
        slot_2 = olt.module.next_neu.data.modules.add()
        port_1_1 = slot_1.interfaces.add()
        port_1_2 = slot_1.interfaces.add()
        port_2_1 = slot_2.interfaces.add()
        port_2_2 = slot_2.interfaces.add()
        pon_1_1_1 = port_1_1.pons.add()
        pon_1_1_1.id = '1/1/1 GPON'
        pon_1_1_2 = port_1_1.pons.add()
        pon_1_1_2.id = '1/1/1 XGS-PON'
        pon_1_2_1 = port_1_2.pons.add()
        pon_1_2_1.id = '1/1/2 GPON'
        pon_2_1_1 = port_2_1.pons.add()
        pon_2_1_1.id = '1/2/1 GPON'
        pon_2_1_2 = port_2_1.pons.add()
        pon_2_1_2.id = '1/2/1 XGS-PON'
        pon_2_2_1 = port_2_2.pons.add()
        pon_2_2_1.id = '1/2/2 GPON'
        self._await(olt.load())
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 6)

        # Simulate an OLT line card being removed
        del olt.module.next_neu.data.modules[1]
        self._await(olt.load())
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 3)

        # Delete the other card
        card2 = olt.module.next_neu.data.modules[0]
        del olt.module.next_neu.data.modules[0]
        self._await(olt.load())
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 0)

        # Reinsert the card in slot 2
        olt.module.next_neu.data.modules.append(card2)
        self._await(olt.load())
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 3)

        # Test the refs are removed if the element goes offline
        olt.offline = True
        self._await(olt.remove_ref_offline())
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 0)

    def test_olt_bootstrap(self):
        """An OLT reads its previous state on initialization."""
        # Create existing state (that will no longer be present)
        cur_ref = NetworkMetadataReference(type='ap', value='TestAP.100001', attributes={'topology1': 'PON.100001', 'topology2': 'OLT.TestAP', 'ap_config': 'TestAP.100001', 'topology1_snmp': 'TestAP', 'system': 'net'})
        self._await(self.ctx.netmeta_model.set_ref(cur_ref))
        olt = Olt(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'olt', None)
        olt.module = FakeModule()
        pon_port = olt.module.next_neu.data.interfaces.add()
        pon = pon_port.pons.add()
        pon.id = '100002'
        self._await(olt.load())
        # old state should be removed and new state created
        exp_ref = NetworkMetadataReference(type='ap', value='TestAP.100002', attributes={'topology1': 'PON.100002', 'topology2': 'OLT.TestAP', 'ap_config': 'TestAP.100002', 'topology1_snmp': 'TestAP', 'system': 'net'})
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 1)
        self.assertEqual(ap_refs.get('TestAP.100002'), exp_ref)

    def test_olt_legacy_mode_boolean(self):
        """Network metadata is created for site/ap if legacy mode is selected."""
        self.ctx.olt_legacy_mapping = True
        olt = Olt(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'olt', None)
        olt.module = FakeModule()
        pon_port = olt.module.next_neu.data.interfaces.add()
        pon = pon_port.pons.add()
        pon.id = '100001'
        self._await(olt.load())
        exp_ref = NetworkMetadataReference(type='ap', value='TestAP.100001', attributes={'topology1': 'TestAP', 'topology2': 'MySite', 'ap_config': 'TestAP.100001', 'topology1_snmp': 'TestAP', 'system': 'net'})
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 1)
        self.assertEqual(ap_refs.get('TestAP.100001'), exp_ref)

    def test_olt_legacy_mode_list(self):
        """Network metadata is created for site/ap if this OLT is configured for legacy mode."""
        self.ctx.olt_legacy_mapping = ['SomeOtherAP', 'TestAP']
        olt = Olt(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'olt', None)
        olt.module = FakeModule()
        pon_port = olt.module.next_neu.data.interfaces.add()
        pon = pon_port.pons.add()
        pon.id = '100001'
        self._await(olt.load())
        exp_ref = NetworkMetadataReference(type='ap', value='TestAP.100001', attributes={'topology1': 'TestAP', 'topology2': 'MySite', 'ap_config': 'TestAP.100001', 'topology1_snmp': 'TestAP', 'system': 'net'})
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 1)
        self.assertEqual(ap_refs.get('TestAP.100001'), exp_ref)

    def test_olt_legacy_mode_list_other(self):
        """Network metadata is created for site/ap if this OLT is configured for legacy mode."""
        self.ctx.olt_legacy_mapping = ['SomeOtherAP']
        olt = Olt(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'olt', None)
        olt.module = FakeModule()
        pon_port = olt.module.next_neu.data.interfaces.add()
        pon = pon_port.pons.add()
        pon.id = '100001'
        self._await(olt.load())
        exp_ref = NetworkMetadataReference(type='ap', value='TestAP.100001', attributes={'topology1': 'PON.100001', 'topology2': 'OLT.TestAP', 'ap_config': 'TestAP.100001', 'topology1_snmp': 'TestAP', 'system': 'net'})
        ap_refs = self.ctx.netmeta_model.refs.get('ap') or {}
        self.assertEqual(len(ap_refs), 1)
        self.assertEqual(ap_refs.get('TestAP.100001'), exp_ref)

