"""Test network element life cycle logic etc"""
import asyncio
import os, os.path
import sys
import unittest

from preseem import FakeNetworkMetricsModel, NetworkMetadataReference

sys.path.append(os.path.dirname(__file__))  # let code under test load stubs
import ap_data
from fake_snmp import FakeSnmpClient
from ne import HttpCredentials, NetworkElementRegistry, HOLD_OFF, Element
from devices.ubnt import snmpBadMibError, snmpBadUpdateError, snmpUnsupportedFwError
from device_test import get_datafiles
from fake_context import FakeContext
from preseem.network_element_update import NetworkElement, NetworkElementUpdate, ErrorContext
from fake_device import FakeDevice
from uuid import UUID
from preseem_protobuf.network_poller import network_poller_pb2

# TODO this should actually do something useful, and probably be in preseem-sdk
class FakeHttpClient():
    def __init__(self, *args, **kwargs):
        self._url = None
        pass

    async def request(self, *args, **kwargs):
        pass

    async def close(self):
        pass


@staticmethod
async def fake_ping(host):
    if host == FakeContext.host:
        return True
    else:
        return False



class FakeContext(FakeContext):
    host='TEST'
    snmp_client_cls = FakeSnmpClient
    http_client_cls = FakeHttpClient
    ping = fake_ping
    verbose = False
    holdoff_loaded = True
    test_data = None
    poll_interval = None
    pysnmp_enabled = False
    pysnmp_fallback_enabled = False
    make_test_data = None
    network_poller_grpc = False
    company_uuid = None
    metrics_enabled = True
    bridge_mapping_enabled = False
    
    job = False
    cfg = {}
    name = 'unittest'

    async def start(self):
        self.metrics = []
        self.metrics_model = FakeNetworkMetricsModel(self.metrics_cbk)

    async def metrics_cbk(self, metrics):
        self.metrics.extend(metrics)

    async def close(self):
        if self.metrics_model:
            await self.metrics_model.close()
        if self.network_poller_grpc:
            await self.network_poller_grpc.close()


class TestNetworkElement(unittest.TestCase):
    def setUp(self):
        apdd = ap_data.ApData('ap_info.yaml')
        self.datafiles = get_datafiles(apdd)
        self.ctx = FakeContext()
        self.loop = asyncio.new_event_loop() # needed to initialize asyncio
        self.reg = NetworkElementRegistry(self.ctx)
        self.reg.ne_type = 'ap'
        asyncio.set_event_loop(self.loop)
        self._await(self.ctx.start())

    def tearDown(self):
        self._await(self.reg.close())
        self._await(self.ctx.close())
        self.loop.close()

    def _await(self, co):
        return self.loop.run_until_complete(co)

    def wait_for_neu_posted(self):
        """Wait for a NetworkElementUpdateMessage to be posted and return it."""
        return self._await(asyncio.wait_for(self.ctx.network_poller_grpc.fut, timeout=1.0))

    # Regular check status test
    def test_add_ap(self):
        """Add an AP object to the model.  It gets polled."""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 2,
            'sysname': 'Bear Valley North',
            'sysobjid': '1.3.6.1.4.1.10002.1',
            'snmp_error': '',
            'active': True,
            'pingable': None,
            'holdoff': 0
        })

    # Disconnection test
    # trigger a change of connection_timeout.
    def test_disconnected(self):
        """Add an AP object to the model. Check disconnected report."""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.snmp_client_cls.connection_lost = True
        ne.holdoff = 1
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'disconnected',
            'active': False,
            'pingable': None,
            'holdoff': 1
        })

    # Bad Mib error test
    # trigger a change of bad_mib_state.
    def test_bad_mib_error(self):
        """Add an AP object to the model. Check bad mib error report."""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ne.bad_mib_state = True
        ne.error_msg = snmpBadMibError
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 2,
            'sysname': 'Bear Valley North',
            'sysobjid': '1.3.6.1.4.1.10002.1',
            'snmp_error': ne.error_msg,
            'active': False,
            'pingable': None,
            'holdoff': 0
        })

    # Bad update MIB error test
    # trigger a change of bad_mib_state and relevant bad mib update message.
    def test_bad_update_mib_error(self):
        """Add an AP object to the model. Check bad mib update report."""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ne.bad_mib_state = True
        ne.error_msg = snmpBadUpdateError
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 2,
            'sysname': 'Bear Valley North',
            'sysobjid': '1.3.6.1.4.1.10002.1',
            'snmp_error': ne.error_msg,
            'active': False,
            'pingable': None,
            'holdoff': 0
        })

    # Unsupported error test
    # trigger a change of unsupported_firmware.
    def test_unsupported_firmware_error(self):
        """Add an AP object to the model. Check unsupported_firmware report."""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ne.unsupported_firmware = True
        ne.error_msg = snmpUnsupportedFwError
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 2,
            'sysname': 'Bear Valley North',
            'sysobjid': '1.3.6.1.4.1.10002.1',
            'snmp_error': ne.error_msg,
            'active': False,
            'pingable': None,
            'holdoff': 0
        })


    # Disconnection test 2
    # trigger a change of connection_timeout + no ping test.
    # Not My Site will invoke pingable false
    def test_disconnected_ping(self):
        """Add an AP object to the model. Check disconnected report."""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'Not My Site', 'path': df.path}))
        self.ctx.host='TEST2'
        self.ctx.snmp_client_cls.connection_lost = True
        ne.holdoff = 1
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'Not My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'disconnected',
            'active': False,
            'pingable': None,
            'holdoff': 1
        })

    #Offline test 1
    # trigger a change of connection_timeout and offline event.
    def test_offline_loaded(self):
        """Add an AP object to the model. Check offline report if device set to be disconnected for more than HOLD_OFF time."""
        # set 1 sec more than defined on static map that is loaded from start
        Element.ne_holdofd_ctrs[('TestNE', 'ap')] = HOLD_OFF+1
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.snmp_client_cls.connection_lost = True
        self.assertEqual(ne.holdoff, HOLD_OFF+1)
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'offline',
            'active': False,
            'pingable': None
        })

    # Offline test 2
    # trigger a change of connection_timeout and offline event.
    def test_offline_not_loaded(self):
        """Add an AP object to the model. Check offline report if device set to be disconnected for more than HOLD_OFF time."""

        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        # set 1 sec more than defined directly
        ne.holdoff=HOLD_OFF+1
        self.ctx.snmp_client_cls.connection_lost = True
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'offline',
            'active': False,
            'pingable': None
        })

    # Disconnection test
    # trigger a change of connection_timeout.
    def test_loaded_holdoff_sumup_to_new(self):
        """Add an AP object to the model. Check disconnected report."""
        Element.ne_holdofd_ctrs[('TestNE', 'ap')] = 20
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.snmp_client_cls.connection_lost = True
        ne.holdoff +=1
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'disconnected',
            'active': False,
            'pingable': None,
            'holdoff': 21
        })

    # Disconnection test
    # trigger a change in set during disconnection.
    def test_not_loaded_holdoff_when_set_with_change(self):
        """Add an AP object to the model. Check disconnected report and holdoff is continued and not loaded"""
        Element.ne_holdofd_ctrs[('TestNE','ap')] = 20
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.snmp_client_cls.connection_lost = True
        ne.holdoff +=1
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(-1)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'disconnected',
            'active': False,
            'pingable': None,
            'holdoff': 21
        })
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site2', 'path': df.path}))
        ne.holdoff +=1
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(-1)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site2',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'disconnected',
            'active': False,
            'pingable': None,
            'holdoff': 22
        })

    # Restore connection test after loaded holdoff
    def test_restore(self):
        """Add an AP object to the model. Check holdoff is reseted"""
        Element.ne_holdofd_ctrs[('TestNE', 'ap')] = HOLD_OFF-1
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        # holdoff as loaded
        self.assertEqual(HOLD_OFF-1, ne.holdoff)
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        # holdoff after reset
        self.assertEqual(0, ne.holdoff)
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(-1)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 2,
            'sysname': 'Bear Valley North',
            'sysobjid': '1.3.6.1.4.1.10002.1',
            'snmp_error': '',
            'active': True,
            'pingable': None,
            'holdoff': 0
        })

    #Restore test 2
    # trigger a change of connection_timeout and offline event , after it restore
    def test_offline_restore(self):
        """Add an AP object to the model. Check offline report if device set to be disconnected for more than HOLD_OFF time, then restore"""

        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        # set 1 sec more than defined directly
        ne.holdoff=HOLD_OFF+1
        self.ctx.snmp_client_cls.connection_lost = True
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(-1)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'offline',
            'active': False,
            'pingable': None
        })
        self.ctx.snmp_client_cls.connection_lost = False
        self._await(ne.poll())
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics),2)
        fm = self.ctx.metrics.pop(-1)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 2,
            'sysname': 'Bear Valley North',
            'sysobjid': '1.3.6.1.4.1.10002.1',
            'snmp_error': '',
            'active': True,
            'pingable': None,
            'holdoff': 0
        })

    #Restore test 3
    # trigger a change of connection_timeout and disconnect event , after it restore

    def test_restore_after_disconnect(self):
        """Add an AP object to the model. Check offline report if device set to be disconnected for more than HOLD_OFF time, then restore"""

        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        # set 1 sec more than defined directly
        ne.holdoff=HOLD_OFF-1
        self.ctx.snmp_client_cls.connection_lost = True
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(-1)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'disconnected',
            'active': False,
            'pingable': None,
            'holdoff': HOLD_OFF-1
        })
        self.ctx.snmp_client_cls.connection_lost = False
        self._await(ne.poll())
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self.assertEqual(len(self.ctx.metrics),1)
        fm = self.ctx.metrics.pop(-1)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 2,
            'sysname': 'Bear Valley North',
            'sysobjid': '1.3.6.1.4.1.10002.1',
            'snmp_error': '',
            'active': True,
            'pingable': None,
            'holdoff': 0
        })

        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics),1)
        fm = self.ctx.metrics.pop(-1)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 2,
            'sysname': 'Bear Valley North',
            'sysobjid': '1.3.6.1.4.1.10002.1',
            'snmp_error': '',
            'active': True,
            'pingable': None,
            'holdoff': 0
        })

    #holdoff load test 1
    # simulate holdoff > 0 is loaded from current instance and countinued to count and get offline
    def test_holdoff_loaded(self):
        """."""
        self.reg.ctx.telemetry_api = None
        self.reg.latest_saved_holdoff = {('TestNE', self.ctx.name, self.reg.ne_type): {'id': 'TestNE','instance':self.ctx.name, 'active' :False ,'holdoff': HOLD_OFF-1, 'type': self.reg.ne_type }}
        self._await(self.reg.load_holdoff())
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.snmp_client_cls.connection_lost = True
        ne.holdoff+=1
        self.assertEqual(ne.holdoff, HOLD_OFF)
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'offline',
            'active': False,
            'pingable': None 
        })

    #holdoff load test 2
    # simulate holdoff = None is set if AP wasn't active in all instance (during last holdoff time period not simulated)
    def test_holdoff_loaded_none(self):
        """."""
        self.reg.ctx.telemetry_api = None
        self.reg.latest_saved_holdoff = {('TestNE', self.ctx.name, self.reg.ne_type): {'id': 'TestNE','instance':self.ctx.name, 'active' :False ,'holdoff': None, 'type': self.reg.ne_type }}
        self.reg.latest_saved_holdoff[('TestNE', self.ctx.name+'1', self.reg.ne_type)] = {'id': 'TestNE','instance':self.ctx.name+'1', 'active' :False ,'holdoff': None, 'type': self.reg.ne_type }
        self.reg.latest_saved_holdoff[('TestNE', self.ctx.name+'2', self.reg.ne_type)] = {'id': 'TestNE','instance':self.ctx.name+'2', 'active' :False ,'holdoff': None, 'type': self.reg.ne_type }
        self._await(self.reg.load_holdoff())
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.snmp_client_cls.connection_lost = True
        self.assertEqual(ne.holdoff, None)
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'offline',
            'active': False,
            'pingable': None
        })

    # holdoff load test 3
    # simulate holdoff = None is set if AP was found active in some other instance
    def test_holdoff_loaded_none(self):
        """."""
        self.reg.ctx.telemetry_api = None
        self.reg.latest_saved_holdoff = {('TestNE', self.ctx.name, self.reg.ne_type): {'id': 'TestNE','instance':self.ctx.name, 'active' :False ,'holdoff': None, 'type': self.reg.ne_type }}
        self.reg.latest_saved_holdoff[('TestNE', self.ctx.name+'1', self.reg.ne_type)] = {'id': 'TestNE','instance':self.ctx.name+'1', 'active' :True ,'holdoff': 0, 'type': self.reg.ne_type }
        self.reg.latest_saved_holdoff[('TestNE', self.ctx.name+'2', self.reg.ne_type)] = {'id': 'TestNE','instance':self.ctx.name+'2', 'active' :True ,'holdoff': 0, 'type': self.reg.ne_type }
        self._await(self.reg.load_holdoff())
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.snmp_client_cls.connection_lost = True
        self.assertEqual(ne.holdoff, None)
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'offline',
            'active': False,
            'pingable': None,
        })

    # holdoff load test 4
    # simulate holdoff is set accoring to this current instance and continue to count
    def test_holdoff_loaded_from_instance(self):
        """."""
        self.reg.ctx.telemetry_api = None
        self.reg.latest_saved_holdoff = {('TestNE', self.ctx.name, self.reg.ne_type): {'id': 'TestNE','instance':self.ctx.name, 'active' :False ,'holdoff': 77, 'type': self.reg.ne_type }}
        self.reg.latest_saved_holdoff[('TestNE', self.ctx.name+'1', self.reg.ne_type)] = {'id': 'TestNE','instance':self.ctx.name+'1', 'active' :False ,'holdoff': 15, 'type': self.reg.ne_type }
        self.reg.latest_saved_holdoff[('TestNE', self.ctx.name+'2', self.reg.ne_type)] = {'id': 'TestNE','instance':self.ctx.name+'2', 'active' :False ,'holdoff': 16, 'type': self.reg.ne_type }
        self._await(self.reg.load_holdoff())
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.snmp_client_cls.connection_lost = True
        ne.holdoff+=17
        self.assertEqual(ne.holdoff, 94)
        self._await(asyncio.sleep(2)) # TODO better way to wait etc...
        self._await(self.reg.check_status())
        self.assertEqual(len(self.ctx.metrics), 1)
        fm = self.ctx.metrics.pop(0)
        self.assertEqual(fm.name, 'ne_info')
        self.assertEqual(fm.labels, {
            'instance': 'unittest',
            'source': 'net',
            'id': 'TestNE',
            'site': 'My Site',
            'name': 'Test Element',
            'host': self.ctx.host
        })
        self.assertEqual(fm.fields, {
            'type': 'ap',
            'snmp_version': 0,
            'sysname': None,
            'sysobjid': None,
            'snmp_error': 'disconnected',
            'active': False,
            'pingable': None,
            'holdoff': 94
        })

    def test_http_credentials_1(self):
        """Test that credentials are created from old http config."""
        cfg = {'http_user': 'username', 'http_pass': ['pass1', 'pass2']}
        # create a new registry object to pick up the config
        self._await(self.reg.close())
        self.reg = NetworkElementRegistry(self.ctx, cfg)
        self.reg.ne_type = 'ap'

        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site'}))

        self.assertEqual(ne.snmp_ne.http_credentials, [HttpCredentials('username', 'pass1', [], None), HttpCredentials('username', 'pass2', [], None)])
        
    def test_http_credentials_2(self):
        """Test that credentials are created from new credentials config."""
        cfg = {'credentials': [{'username': 'user', 'password': 'pass', 'role': 'ap'}]}
        # create a new registry object to pick up the config
        self._await(self.reg.close())
        self.reg = NetworkElementRegistry(self.ctx, cfg)
        self.reg.ne_type = 'ap'

        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site'}))

        self.assertEqual(ne.snmp_ne.http_credentials, [HttpCredentials('user', 'pass', [], 'ap')])
        
    def test_http_credentials_3(self):
        """Test that ports can be configured."""
        cfg = {'http_port': [12345, 80], 'credentials': [{'username': 'user', 'password': 'pass', 'role': 'ap'}]}
        # create a new registry object to pick up the config
        self._await(self.reg.close())
        self.reg = NetworkElementRegistry(self.ctx, cfg)
        self.reg.ne_type = 'ap'

        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site'}))

        self.assertEqual(ne.snmp_ne.http_credentials, [HttpCredentials('user', 'pass', [12345, 80], 'ap')])
        
    def test_http_credentials_4(self):
        """Test that Mikrotik credentials can be configured."""
        cfg = {
            'http_port': [12345, 80],
            'credentials': [{'username': 'user', 'password': 'pass', 'role': 'ap'}],
            'mikrotik_username': 'mikrotik_user',
            'mikrotik_password': 'mikrotik_pass',
        }
        # create a new registry object to pick up the config
        self._await(self.reg.close())
        self.reg = NetworkElementRegistry(self.ctx, cfg)
        self.reg.ne_type = 'ap'

        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site'}))

        self.assertEqual(ne.snmp_ne.http_credentials, [HttpCredentials('user', 'pass', [12345, 80], 'ap'), HttpCredentials('mikrotik_user', 'mikrotik_pass', [], 'router')])

    def test_http_credentials_5(self):
        """Test that Mikrotik credentials will not be configured when (username, password, role) exists."""
        cfg = {
            'http_port': [12345, 80],
            'credentials': [{'username': 'user', 'password': 'pass', 'role': 'router'}],
            'mikrotik_username': 'user',
            'mikrotik_password': 'pass',
        }
        # create a new registry object to pick up the config
        self._await(self.reg.close())
        self.reg = NetworkElementRegistry(self.ctx, cfg)
        self.reg.ne_type = 'ap'

        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site'}))

        self.assertEqual(ne.snmp_ne.http_credentials, [HttpCredentials('user', 'pass', [12345, 80], 'router')])

    def test_http_credentials_6(self):
        """Test that Mikrotik credentials will be configured when username and password pair exists, but the role does not match. This allows duplicate credentials where roles are configured differently."""
        cfg = {
            'http_port': [12345, 80],
            'credentials': [{'username': 'user', 'password': 'pass'}],
            'mikrotik_username': 'user',
            'mikrotik_password': 'pass',
        }
        # create a new registry object to pick up the config
        self._await(self.reg.close())
        self.reg = NetworkElementRegistry(self.ctx, cfg)
        self.reg.ne_type = 'ap'

        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site'}))

        self.assertEqual(ne.snmp_ne.http_credentials, [HttpCredentials('user', 'pass', [12345, 80], None), HttpCredentials('user', 'pass', [], "router")])

    def test_send_update_1(self):
        """Verify that send_update sends network_element_update_msg when matching elem_discovery
            reference is found."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1', 'uuid': '491c9831-7597-4a2e-8464-000000000000'}
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference(type='ap', value='TestNE', attributes=attrs)))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference(
            'elem_discovery', '491c9831-7597-4a2e-8464-000000000000',{'element_uuid': '491c9831-7597-4a2e-8464-111111111111'})))
        ne = Element(self.ctx, 'TestNE', 'TestName', 'TestSite', None, None, 'ap', None, {'uuid': '491c9831-7597-4a2e-8464-000000000000'})
        ne.device = FakeDevice(None)
        nw_ele = NetworkElement(errctx=ErrorContext())
        nw_ele.errors = []
        nw_ele.management_ip = '192.0.2.1'
        nw_ele.name = 'abra-RP- 60'
        nw_ele.manufacturer = 'Ubiquiti'
        nw_ele.model = 'Rocket Prism 5AC Gen2'
        nw_ele.system_mac_address = '78:8a:20:1c:71:8d'
        ne.network_element_update_msg = NetworkElementUpdate()
        ne.network_element_update_msg.data = nw_ele
        ne.poll_start_time = 1655500145
        self._await(ne.send_update())
        posted_neu = self.wait_for_neu_posted()[0]
        self.assertIsInstance(posted_neu, network_poller_pb2.NetworkElementUpdate)
        self.assertEqual(posted_neu.data.name, 'abra-RP- 60')
        self.assertEqual(posted_neu.data.system_mac_address, '78:8a:20:1c:71:8d')

    def test_send_update_2(self):
        """Verify that send_update sends network_element_discovery when matching elem_discovery reference is not found."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1', 'uuid': '491c9831-7597-4a2e-8464-000000000000'}
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference(type='ap', value='TestNE', attributes=attrs)))
        ne = Element(self.ctx, 'TestNE', 'TestName', 'TestSite', None, None, 'ap', None, {'uuid': '491c9831-7597-4a2e-8464-000000000000'})
        ne.device = FakeDevice(None)
        # ne.ctx.cfg['uuid'] = '491c9831-7597-4a2e-8464-000000000000'
        self._await(ne.send_update())
        posted_neu = self.wait_for_neu_posted()[0]
        self.assertIsInstance(posted_neu, network_poller_pb2.NetworkElementDiscovery)

    def test_offline_update(self):
        """The poller sets an OFFLINE message to send if snmp is offline."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1', 'uuid': '491c9831-7597-4a2e-8464-000000000000'}
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference(type='ap', value='TestNE', attributes=attrs)))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference(
            'elem_discovery', '491c9831-7597-4a2e-8464-000000000000',{'element_uuid': '491c9831-7597-4a2e-8464-111111111111'})))
        ne = Element(self.ctx, 'TestNE', 'TestName', 'TestSite', None, None, 'ap', None, {'uuid': '491c9831-7597-4a2e-8464-000000000000'})
        ne.device = FakeDevice(None)
        ne.snmp_error = 'offline'
        nw_ele = NetworkElement(errctx=ErrorContext())
        nw_ele.errors = []
        nw_ele.management_ip = '192.0.2.1'
        nw_ele.name = 'abra-RP- 60'
        nw_ele.manufacturer = 'Ubiquiti'
        nw_ele.model = 'Rocket Prism 5AC Gen2'
        nw_ele.system_mac_address = '78:8a:20:1c:71:8d'
        ne.network_element_update_msg = NetworkElementUpdate()
        ne.network_element_update_msg.data = nw_ele
        self._await(ne.send_update())  # this just sets an internal msg
        msg = ne.offline_disconnected_update_msg
        self.assertIsNotNone(ne)
        self.assertEqual(msg.data.status, network_poller_pb2.ELEMENT_STATUS_OFFLINE)

    def test_suppress_offline_update_to_agent_polled_element(self):
        """STM-9828 the poller doesn't send OFFLINE updates for elements
           polled by the netmeta agent."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1', 'uuid': '491c9831-7597-4a2e-8464-000000000000'}
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference(type='ap', value='TestNE', attributes=attrs)))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference(
            'elem_discovery', '491c9831-7597-4a2e-8464-000000000000',{'element_uuid': '491c9831-7597-4a2e-8464-111111111111'})))
        self.ctx.agent_polled_elems = {'TestNE'}
        ne = Element(self.ctx, 'TestNE', 'TestName', 'TestSite', None, None, 'ap', None, {'uuid': '491c9831-7597-4a2e-8464-000000000000'})
        ne.device = FakeDevice(None)
        ne.snmp_error = 'offline'
        nw_ele = NetworkElement(errctx=ErrorContext())
        nw_ele.errors = []
        nw_ele.management_ip = '192.0.2.1'
        nw_ele.name = 'abra-RP- 60'
        nw_ele.manufacturer = 'Ubiquiti'
        nw_ele.model = 'Rocket Prism 5AC Gen2'
        nw_ele.system_mac_address = '78:8a:20:1c:71:8d'
        ne.network_element_update_msg = NetworkElementUpdate()
        ne.network_element_update_msg.data = nw_ele
        self._await(ne.send_update())  # this just sets an internal msg
        self.assertIsNone(ne.offline_disconnected_update_msg)
