"""Test the check module."""
import asyncio
import os
import sys
import unittest

sys.path.append(os.path.dirname(__file__))  # let code under test load stubs
from context import Context
from fake_snmp import FakeSnmpClient
from preseem import NetworkMetadataReference
from preseem.ssl_probe import SslProbeResult
from check import check

async def fake_ping(host):
    return 1.1

async def fake_ssl_probe(host, port):
    return SslProbeResult('NoConnection', None, None)

class TestCheck(unittest.TestCase):
    def setUp(self):
        self.ctx = Context('test', {}, None, {}, fake=True, job=True)
        self.ctx.ping = fake_ping
        self.ctx.snmp_client_cls = FakeSnmpClient
        self.ctx.ssl_probe = fake_ssl_probe
        self.loop = asyncio.new_event_loop()
        initialized = self.loop.create_future()
        self._task = self.loop.create_task(self.ctx.start(load_metadata=False, load_elements=False, fut=initialized))
        self._await(initialized)
        check.PING_ENABLED = False  # make results consistent whether root or not

    def tearDown(self):
        self._task.cancel()
        self._await(self.ctx.close())
        self.loop.close()

    def _await(self, co):
        return self.loop.run_until_complete(co)

    def test_ap_probe(self):
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'ap01', {'topology2': 'Site 1', 'topology1': 'AP 1', 'ip_address': '192.0.2.1'})))

        def check_output(name, rows, cols, csvfile):
            self.assertIn(csvfile, ('/tmp/preseem-aps.csv', '/tmp/preseem-routers.csv', '/tmp/preseem-switches.csv', '/tmp/preseem-olts.csv'))
            if csvfile == '/tmp/preseem-aps.csv':
                self.assertEqual(len(rows), 1)
                self.assertEqual(rows[0], ('ap01', 'Site 1', 'AP 1', '192.0.2.1', True, None, None, None))
            elif csvfile == '/tmp/preseem-routers.csv':
                self.assertEqual(len(rows), 0)
        self._await(check(self.ctx, check_output))

    def test_ap_probe_verbose(self):
        async def fake_ssl_probe(host, port):
            return SslProbeResult(error=None, org='Tarana Wireless', cn='S134F2215100421.radio.taranawireless.com')
        self.ctx.ssl_probe = fake_ssl_probe
        self.ctx.verbose = 1
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'ap01', {'topology2': 'Site 1', 'topology1': 'AP 1', 'ip_address': '192.0.2.1'})))

        def check_output(name, rows, cols, csvfile):
            self.assertIn(csvfile, ('/tmp/preseem-aps.csv', '/tmp/preseem-routers.csv', '/tmp/preseem-switches.csv', '/tmp/preseem-olts.csv'))
            if csvfile == '/tmp/preseem-aps.csv':
                self.assertEqual(len(rows), 1)
                self.assertEqual(rows[0], ('ap01', 'Site 1', 'AP 1', '192.0.2.1', True, None, None, None, None, None, 'Tarana Wireless', 'S134F2215100421.radio.taranawireless.com'))
            elif csvfile == '/tmp/preseem-routers.csv':
                self.assertEqual(len(rows), 0)
        self._await(check(self.ctx, check_output))

    def test_router_probe(self):
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'ap01', {'topology2': 'Site 1', 'topology1': 'AP 1', 'ip_address': '192.0.2.1'})))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('router', 'rtr01', {'site': 'Site 1', 'name': 'Router', 'host': '192.0.2.254'})))

        def check_output(name, rows, cols, csvfile):
            self.assertIn(csvfile, ('/tmp/preseem-aps.csv', '/tmp/preseem-routers.csv', '/tmp/preseem-switches.csv', '/tmp/preseem-olts.csv'))
            if csvfile == '/tmp/preseem-aps.csv':
                self.assertEqual(len(rows), 1)
                self.assertEqual(rows[0], ('ap01', 'Site 1', 'AP 1', '192.0.2.1', True, None, None, None))
            elif csvfile == '/tmp/preseem-routers.csv':
                self.assertEqual(len(rows), 1)
                self.assertEqual(rows[0], ('rtr01', 'Router', '192.0.2.254', True, None, None, None, None))
        self._await(check(self.ctx, check_output))

    def test_switch_probe(self):
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'ap01', {'topology2': 'Site 1', 'topology1': 'AP 1', 'ip_address': '192.0.2.1'})))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('switch', 'sw01', {'site': 'Site 1', 'name': 'Switch', 'host': '192.0.2.254'})))

        def check_output(name, rows, cols, csvfile):
            self.assertIn(csvfile, ('/tmp/preseem-aps.csv', '/tmp/preseem-routers.csv', '/tmp/preseem-switches.csv', '/tmp/preseem-olts.csv'))
            if csvfile == '/tmp/preseem-aps.csv':
                self.assertEqual(len(rows), 1)
                self.assertEqual(rows[0], ('ap01', 'Site 1', 'AP 1', '192.0.2.1', True, None, None, None))
            elif csvfile == '/tmp/preseem-switches.csv':
                self.assertEqual(len(rows), 1)
                self.assertEqual(rows[0], ('sw01', 'Switch', '192.0.2.254', True, None, None, None, None))
        self._await(check(self.ctx, check_output))

    def test_olt_probe(self):
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'ap01', {'topology2': 'Site 1', 'topology1': 'AP 1', 'ip_address': '192.0.2.1'})))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('olt', 'olt01', {'site': 'Site 1', 'name': 'OLT 1', 'host': '192.0.2.254'})))

        def check_output(name, rows, cols, csvfile):
            self.assertIn(csvfile, ('/tmp/preseem-aps.csv', '/tmp/preseem-routers.csv', '/tmp/preseem-switches.csv', '/tmp/preseem-olts.csv'))
            if csvfile == '/tmp/preseem-aps.csv':
                self.assertEqual(len(rows), 1)
                self.assertEqual(rows[0], ('ap01', 'Site 1', 'AP 1', '192.0.2.1', True, None, None, None))
            elif csvfile == '/tmp/preseem-olts.csv':
                self.assertEqual(len(rows), 1)
                self.assertEqual(rows[0], ('olt01', 'OLT 1', '192.0.2.254', True, None, None, None, None))
        self._await(check(self.ctx, check_output))
