import asyncio
import ipaddress
import unittest
from uuid import UUID

from context import Context
from ne import NetworkElementRegistry
from preseem import NetworkMetadataReference

class FakeAp:
    ne_type = 'ap'
    def __init__(self, ctx, neid, name, site, snmp_ne, snmp_ne_v3, ne_type, holdoff, extra=None):
        self.ctx = ctx
        self.neid = neid
        self.name = name
        self.site = site
        self.snmp_ne = snmp_ne
        self.snmp_ne_v3 = None
        self.ne_type = ne_type
        self.holdoff = holdoff

    async def close(self):
        if self.snmp_ne:
            self.snmp_ne.close()
            self.snmp_ne = None

    def start(self):
        pass


class TestConfig(unittest.TestCase):
    def setUp(self):
        self.ctx = Context('test', {}, None, {}, fake=True)
        self.loop = asyncio.new_event_loop() # needed to initialize asyncio
        asyncio.set_event_loop(self.loop)
        self.ctx.ap_registry = NetworkElementRegistry(self.ctx, self.ctx.defaults, FakeAp)
        initialized = self.loop.create_future()
        self._task = self.loop.create_task(self.ctx.start(fut=initialized))
        self._await(initialized)

    def tearDown(self):
        self._task.cancel()
        self._await(self.ctx.close())
        self.loop.close()

    def _await(self, co):
        return self.loop.run_until_complete(co)

    def test_snmp_config_01(self):
        """SNMP community on the AP ref is passed to the SNMP client."""
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1'}
        attrs['community'] = 'testcomm'
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'TESTAP1', attrs)))
        ap = self.ctx.ap_registry.nes.get('TESTAP1')
        self.assertIsNotNone(ap)
        cfg = ap.snmp_ne.cfg
        self.assertEqual(cfg.get('community'), 'testcomm')

    def test_snmp_config_02(self):
        """SNMP community on the ap_config ref is passed to the SNMP client."""
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1'}
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'TESTAP1', attrs)))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap_config', 'TESTAP1', {'ap_snmp_community': 'cfgcomm'})))
        ap = self.ctx.ap_registry.nes.get('TESTAP1')
        self.assertIsNotNone(ap)
        cfg = ap.snmp_ne.cfg
        self.assertEqual(cfg.get('community'), 'cfgcomm')

    def test_snmp_config_03(self):
        """SNMP community on the ap_config ref is passed to the SNMP client if ap is added after."""
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1'}
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap_config', 'TESTAP1', {'ap_snmp_community': 'cfgcomm'})))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'TESTAP1', attrs)))
        ap = self.ctx.ap_registry.nes.get('TESTAP1')
        self.assertIsNotNone(ap)
        cfg = ap.snmp_ne.cfg
        self.assertEqual(cfg.get('community'), 'cfgcomm')

    def test_snmp_config_04(self):
        """SNMPv3 config on the ap_config ref is passed to the SNMP client."""
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1'}
        v3attrs = {
            'ap_snmp3_security_name': 'PLAYER1',
            'ap_snmp3_auth_key': 'AUTHKEY!',
            'ap_snmp3_auth_protocol': 'SHA',
            'ap_snmp3_priv_key': 'PRIVKEY!',
            'ap_snmp3_priv_protocol': 'AES'
        }
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap_config', 'TESTAP1', v3attrs)))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'TESTAP1', attrs)))
        ap = self.ctx.ap_registry.nes.get('TESTAP1')
        self.assertIsNotNone(ap)
        cfg = ap.snmp_ne.cfg
        self.assertEqual(cfg.get('username'), 'PLAYER1')
        self.assertEqual(cfg.get('priv_key'), 'PRIVKEY!')
        self.assertEqual(cfg.get('priv_protocol'), 'AES')
        self.assertEqual(cfg.get('auth_key'), 'AUTHKEY!')
        self.assertEqual(cfg.get('auth_protocol'), 'SHA')

    def test_ignore_subnets_config_01(self):
        """Test input values for "ignore_subnets" configuration."""
        self._await(self.ctx.close())
        self.ctx = Context('test', {}, None, {}, netpoller_cfg={'ignore_subnets': 'asdf'}, fake=True)
        self.assertEqual(self.ctx.ignore_subnets, [])
        self._await(self.ctx.close())

        self.ctx = Context('test', {}, None, {}, netpoller_cfg={'ignore_subnets': ['BAD_SUBNET', '192.0.2.0/24', '']}, fake=True)
        self.assertEqual(self.ctx.ignore_subnets, [ipaddress.ip_network('192.0.2.0/24')])
        self._await(self.ctx.close())

        self.ctx = Context('test', {}, None, {}, netpoller_cfg={'ignore_subnets': ['192.0.2.0/25', '192.0.2.255']}, fake=True)
        self.assertEqual(self.ctx.ignore_subnets, [ipaddress.ip_network('192.0.2.0/25'), ipaddress.ip_network('192.0.2.255/32')])

    def test_company_uuid_from_ap_ref(self):
        """Test that the company UUID can be read from an AP reference."""
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1'}
        attrs['company_uuid'] = 'd116c9b2-25f2-48a6-9538-91ca712791ef'
        self.assertIsNone(self.ctx.company_uuid)
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'TESTAP1', attrs)))
        self.assertEqual(str(self.ctx.company_uuid), attrs['company_uuid'])

    def test_company_uuid_collision(self):
        """Test that the company UUID is not set if it differs from a previously set UUID"""
        company_uuid = UUID('f7d8c925-15f9-4ad8-b4bd-3f7a548684ed')
        self.ctx.company_uuid = company_uuid
        attrs = {'topoloy2': 'T2', 'topology1': 'T1', 'ip_address': '192.0.2.1'}
        attrs['company_uuid'] = 'd116c9b2-25f2-48a6-9538-91ca712791ef'
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap', 'TESTAP1', attrs)))
        self.assertEqual(self.ctx.company_uuid, company_uuid)

    def test_ipv4_mapping_config(self):
        """Test that IPv4 mapping is enabled by default but can be disabled (STM-8681)"""
        self._await(self.ctx.close())
        self.ctx = Context('test', {}, None, {}, netpoller_cfg={}, fake=True)
        self.assertEqual(self.ctx.ipv4_mapping_enabled, True)
        self._await(self.ctx.close())

        self.ctx = Context('test', {}, None, {}, netpoller_cfg={'map-ipv4': False}, fake=True)
        self.assertEqual(self.ctx.ipv4_mapping_enabled, False)
        self._await(self.ctx.close())

        self.ctx = Context('test', {}, None, {}, netpoller_cfg={'map_ipv4': False}, fake=True)
        self.assertEqual(self.ctx.ipv4_mapping_enabled, False)
        self._await(self.ctx.close())

        self.ctx = Context('test', {}, None, {}, netpoller_cfg={'map_ipv4': False, 'map-ipv4': False}, fake=True)
        self.assertEqual(self.ctx.ipv4_mapping_enabled, False)
        self._await(self.ctx.close())
