"""Test AP infrastructure"""
import asyncio
from collections import namedtuple, Counter
import os, os.path
import sys
import time
import unittest
from uuid import UUID

from preseem import FakeNetworkMetadataModel, FakeNetworkMetricsModel, NetworkMetadataReference, NetworkMetadataEntity, Reference
from preseem.ssl_probe import SslProbeResult

sys.path.append(os.path.dirname(__file__))  # let code under test load stubs
import ap
import ap_data
from fake_snmp import FakeSnmpClient
from ne import NetworkElementRegistry
from preseem import network_element_update
from devices.ubnt import snmpBadMibError, snmpBadUpdateError, snmpUnsupportedFwError
from device_test import get_datafiles
from preseem_protobuf.network_poller import network_poller_pb2
from preseem.network_element_update import ErrorContext, NetworkElementUpdate, NetworkElement
from fake_context import FakeHttpClient, fake_ping, FakeContext
from fake_device import FakeDevice

import pprint
def pp(name, d, indent=8):
    print(' ' * indent + '{} = '.format(name) + '{\n ', end='')
    for line in pprint.pformat(d, indent=indent)[1:-1].split('\n'):
        print(' ' * indent + line)
    print(' ' * indent + '}')

class Ap(ap.Ap):
    """We subclass the Ap object to add some test synchronization helpers"""
    async def poll(self):
        FakeContext.ap_event.clear()
        await super().poll()
        FakeContext.ap_event.set()

# What I want here is a Module I can add to ap.
# Really I would rather not even do any SNMP here, we're not testing that logic
# and in theory SNMP would be optional anyway.
# I could monkey-patch ap.poll() - sort of already did inAp above..could do
# similar and just skip all the polling logic, which I think may do away
# with any snmp requirements?
# YES - then just call self.load() to do the AP algorithms, after I set
# self.module to my fake module.
#
# May not even need anything fake: just construct a couple AP objects and
# call load on them?

# Ok so try this:
# 1. Create Ap object manually instead of going through registry.
# 2. Set module on it to a FakeModule that I can control to return whatever
# stations list I want, move those around.
class FakeModule:
    def __init__(self):
        self.stations = []
        self.model = None
        self.mode = "ap"
        self.station_poller = None

    async def poll(self):
        pass

    async def poll_station(self, sta, snmp_client, http_client):
        if self.station_poller:
            return self.station_poller(sta, snmp_client, http_client)
        return sta

class FakeClient:
    def __init__(self, host=None):
        self.dft = {}
        self.host = host


class TestAP(unittest.TestCase):
    def setUp(self):
        apdd = ap_data.ApData('ap_info.yaml')
        self.datafiles = get_datafiles(apdd)
        self.ctx = FakeContext()
        self.loop = asyncio.new_event_loop() # needed to initialize asyncio
        self.reg = NetworkElementRegistry(self.ctx, {}, Ap)
        self.reg.ne_type = 'ap'
        asyncio.set_event_loop(self.loop)
        self._await(self.ctx.start())

    def tearDown(self):
        self._await(self.reg.close())
        self._await(self.ctx.close())
        self.loop.close()

    def _await(self, co):
        return self.loop.run_until_complete(co)

    def wait_for_ap_poll(self):
        """Wait for an AP poll to complete."""
        self._await(self.ctx.ap_event.wait())

    def wait_for_neu_posted(self):
        """Wait for a NetworkElementUpdateMessage to be posted and return it."""
        return self._await(asyncio.wait_for(self.ctx.network_poller_grpc.fut, timeout=1.0))

    def test_topology1_snmp_1(self):
        """Test the topology1_snmp attributes gets set properly."""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.wait_for_ap_poll()
        refs = self.ctx.netmeta_model.refs.get('topology1_snmp')
        self.assertEqual(len(refs), 1)
        ref = refs.get('TestNE')
        self.assertEqual(ref, NetworkMetadataReference(type='topology1_snmp', value='TestNE', attributes={'topology1_vendor': 'Ubiquiti', 'topology1_model': 'Rocket Prism 5AC Gen2'}))

    def test_ap_info(self):
        """Test the ap_info flex message and all attributes get set properly."""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket Prism 5AC Gen2.v8.5.7.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        fm = self._await(ap.collect_ap_info())
        self.assertEqual(len(fm), 4)
        self.assertEqual(fm.name, 'ap_info')
        self.assertEqual(fm.labels, {
            'ap': 'TestNE',
            'topology1': 'Test Element',
            'topology2': 'My Site',
            'instance': 'unittest'
        })
        self.assertEqual(fm.fields, {
            'vendor': 'Ubiquiti',
            'model': 'Rocket Prism 5AC Gen2',
            'sw_version': 'v8.5.7',
            'rf_freq': 5740,
            'rf_chan_width': 20,
            'rf_tx_power': 21,
            'sta_count': 24,
            'gps_sync': True,
            'framing': 'FIXED'
        })

    def test_cpe_mac_match_sm(self):
        """Test the cpe_mac attributes gets set properly , case cpe_mac matched sm. should produce 1 cpe_mac per station"""
        # set sm ref that cpe_mac will match
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('sm', '00:04:56:c4:fe:b8',{'service': '1'})))
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 1000.3.4.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.wait_for_ap_poll()
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 1)
        sta_mac_ref = refs.get('00:04:56:c4:fe:b8')
        self.assertEqual(sta_mac_ref, NetworkMetadataReference(type='cpe_mac', value='00:04:56:c4:fe:b8', attributes={'ap': 'TestNE', 'sm':'00:04:56:c4:fe:b8'}))


    def test_cpe_mac_default(self):
        """Test the cpe_mac attributes gets set properly , case cpe_mac and alt_macs not matched sm. Shpuld produce 1 cpe_mac per station, total 4 in this case"""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket 5AC PTMP.v8.5.0.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.wait_for_ap_poll()
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 4)
        for mac, ref in refs.items():
            self.assertEqual(ref, NetworkMetadataReference(type='cpe_mac', value=mac, attributes={'ap': 'TestNE', 'sm':mac}))


    def test_cpe_mac_alt_mac_match_sm(self):
        """Test the cpe_mac attributes gets set properly , case alt_mac matched sm. should produce 2 cpe_mac per station"""
        # set sm ref that alt_mac will match
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('sm', '00:04:56:c4:fe:b7',{'service': '1'})))
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 1000.3.4.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.wait_for_ap_poll()
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 2)
        alt_mac_ref = refs.get('00:04:56:c4:fe:b7')
        sta_mac_ref = refs.get('00:04:56:c4:fe:b8')
        self.assertEqual(alt_mac_ref, NetworkMetadataReference(type='cpe_mac', value='00:04:56:c4:fe:b7', attributes={'ap': 'TestNE', 'sm':'00:04:56:c4:fe:b7'}))
        self.assertEqual(sta_mac_ref, NetworkMetadataReference(type='cpe_mac', value='00:04:56:c4:fe:b8', attributes={'ap': 'TestNE', 'sm':'00:04:56:c4:fe:b7'}))

    def test_cpe_mac_match_sm_and_delete_ap_offline(self):
        """Test the cpe_mac attributes gets set properly , case cpe_mac matched sm. Should produce 1 cpe_mac per station, that should be removed after offline event"""
        # set sm ref that cpe_mac will match
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('sm', '00:04:56:c4:fe:b8',{'service': '1'})))
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 1000.3.4.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.wait_for_ap_poll()
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 1)
        sta_mac_ref = refs.get('00:04:56:c4:fe:b8')
        self.assertEqual(sta_mac_ref, NetworkMetadataReference(type='cpe_mac', value='00:04:56:c4:fe:b8', attributes={'ap': 'TestNE', 'sm':'00:04:56:c4:fe:b8'}))
        # triggering offline event
        ne.offline = True
        self._await(ne.poll())
        self.wait_for_ap_poll()
        # should be deleted == 0
        self.assertEqual(len(refs), 0)

    def test_cpe_mac_match_sm_and_delete_not_in_station(self):
        """Test the cpe_mac attributes gets set properly , case when both cpe_mac and alt_macs are not matched sm. Should produce 1 cpe_mac per station, total 4 in this case, then remove one mac from station"""
        df = self.datafiles.get(('ubnt.airmax-ac.Rocket 5AC PTMP.v8.5.0.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.wait_for_ap_poll()
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 4)
        for mac, ref in refs.items():
            self.assertEqual(ref, NetworkMetadataReference(type='cpe_mac', value=mac, attributes={'ap': 'TestNE', 'sm':mac}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        # remove one station
        ap.stations = [x for x in ap.stations if x.mac_address!='78:8a:20:1c:2e:e5']

        self._await(ap.cpe_mac_update())
        self.wait_for_ap_poll()
        self.assertEqual(len(refs), 3)
        deleted_ref = refs.get('78:8a:20:1c:2e:e5')
        self.assertEqual(deleted_ref, None)

    def test_cpe_mac_alt_mac_match_sm_delete_not_in_station(self):
        """Test the cpe_mac attributes gets set properly , case alt_mac matched sm. should produce 2 cpe_mac per 1 station and 1 default cpe_mac. remove 1st case station, then both cpe_macs should be deleted"""
        # set sm ref that alt_mac will match
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('sm', '58:c1:7a:40:5b:ba',{'service': '1'})))
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 3)
        alt_mac_ref_1 = refs.get('58:c1:7a:40:5b:ba')
        sta_mac_ref_1 = refs.get('58:c1:7a:40:5b:bb')
        self.assertEqual(sta_mac_ref_1, NetworkMetadataReference(type='cpe_mac', value='58:c1:7a:40:5b:bb', attributes={'ap': 'TestNE', 'sm':'58:c1:7a:40:5b:ba'}))
        self.assertEqual(alt_mac_ref_1, NetworkMetadataReference(type='cpe_mac', value='58:c1:7a:40:5b:ba', attributes={'ap': 'TestNE', 'sm':'58:c1:7a:40:5b:ba'}))

        # remove 1st station where the sm was match alt_mac
        ap.stations = [x for x in ap.stations if x.mac_address!="58:c1:7a:40:5b:bb"]

        self._await(ap.cpe_mac_update())
        self.wait_for_ap_poll()
        # as result , 2 cpe_mac should be removed
        self.assertEqual(len(refs), 1)
        deleted_ref_1 = refs.get('58:c1:7a:40:5b:ba')
        self.assertEqual(deleted_ref_1, None)
        deleted_ref_2 = refs.get('58:c1:7a:40:5b:bb')
        self.assertEqual(deleted_ref_2, None)

    def test_cpe_moving_between_aps_01(self):
        """Test a CPE can move between APs."""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        Sta = namedtuple('Sta', ('mac_address',))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne, None, 'ap', None)
        ap1.module = FakeModule()
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1', ne.snmp_ne, None, 'ap', None)
        ap2.module = FakeModule()
        sta = Sta('02:00:00:00:00:01')

        # Station is on AP 1
        ap1.module.stations = [sta]
        self._await(ap1.load())
        self._await(ap2.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-1-id', 'sm': sta.mac_address})

        # Station moves to AP 2
        ap1.module.stations = []
        ap2.module.stations = [sta]
        self._await(ap1.load())
        self._await(ap2.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-2-id', 'sm': sta.mac_address})

    def test_cpe_moving_between_aps_02(self):
        """Test a CPE can move between APs in "Activity Observer" mode"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        def poll(n=1):  # simulate a poll loop
            for _ in range(n):
                self._await(ap1.load())
                self._await(ap2.load())
                self.ctx.dl_rf_packets +=1
                sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
                if len(ap1.module.stations) > 0:
                    ap1.module.stations = [sta]
                if len(ap2.module.stations) > 0:
                    ap2.module.stations = [sta]

        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True
        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
        self.ctx.cpe_mapping_polls = 3

        # Station is on AP 1
        ap1.module.stations = [sta]
        poll(3)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-1-id', 'sm': sta.mac_address})

        # Station moves to AP 2
        ap1.module.stations = []
        ap2.module.stations = [sta]
        poll(3)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-2-id', 'sm': sta.mac_address})


    def test_cpe_map_multiple_aps_01(self):
        """Test a CPE MAC is created for an one active AP from two APs"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        def poll(n=1):  # simulate a poll loop
            for _ in range(n):
                self._await(ap1.load())
                self._await(ap2.load())
                self.ctx.dl_rf_packets +=1
                # making active ap2 only
                sta_active = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
                if len(ap2.module.stations) > 0:
                    ap2.module.stations = [sta_active]


        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True
        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
        self.ctx.cpe_mapping_polls = 3

        # Station is on AP 1
        ap1.module.stations = [sta]
        ap2.module.stations = [sta]
        poll(5)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-2-id', 'sm': sta.mac_address})


    def test_cpe_map_multiple_aps_02(self):
        """Test a CPE MAC is created for an one active AP from three APs"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        def poll(n=1):  # simulate a poll loop
            for _ in range(n):
                self._await(ap3.load())
                self._await(ap1.load())
                self._await(ap2.load())

                self.ctx.dl_rf_packets +=1
                # making active ap3 only
                sta_active = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
                if len(ap3.module.stations) > 0:
                    ap3.module.stations = [sta_active]

        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True
        ap3 = ap.Ap(self.ctx, 'test-ap-3-id', 'test-ap-3', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap3.module = FakeModule()
        ap3.module.activity_observer = True
        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
        self.ctx.cpe_mapping_polls = 3

        ap1.module.stations = [sta]
        ap2.module.stations = [sta]
        ap3.module.stations = [sta]
        poll(3)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-3-id', 'sm': sta.mac_address})


    def test_cpe_map_multiple_aps_03(self):
        """Test a CPE MAC is never created for any non active AP"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        def poll(n=1):  # simulate a poll loop
            for _ in range(n):
                self._await(ap1.load())
                self._await(ap2.load())

        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True
        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)

        self.ctx.cpe_mapping_polls = 3

        ap1.module.stations = [sta]
        ap2.module.stations = [sta]
        poll(5)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNone(ref)

    def test_cpe_map_multiple_aps_04(self):
        """Test a CPE MAC is created for an one active AP from two APs when dl_rf_packets of one of them is None"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        def poll(n=1):  # simulate a poll loop
            for _ in range(n):
                self._await(ap1.load())
                self._await(ap2.load())
                self.ctx.dl_rf_packets +=1
                # making active ap2 only
                sta_active = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
                if len(ap2.module.stations) > 0:
                    ap2.module.stations = [sta_active]


        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True
        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
        self.ctx.cpe_mapping_polls = 3

        ap1.module.stations = [Sta('02:00:00:00:00:01', None)]
        ap2.module.stations = [sta]
        poll(3)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-2-id', 'sm': sta.mac_address})

    def test_ap_swap_1(self):
        """Test a correct swapping of APs in "Activity Observer" mode and netmeta created for both"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))

        def poll(n=1, load=None):  # simulate a poll loop
            for _ in range(n):
                if load==1:
                    self._await(ap1.load())
                if load==2:
                    self._await(ap2.load())
                self.ctx.dl_rf_packets +=1
                sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
                if load ==1 and len(ap1.module.stations) > 0:
                    ap1.module.stations = [sta]
                if load ==2 and len(ap2.module.stations) > 0:
                    ap2.module.stations = [sta]
                    ap1.module.stations = []

        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1',  ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1',  ne.snmp_ne,  None, 'ap', None)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True
        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
        self.ctx.cpe_mapping_polls = 3

        # Station is on AP 1
        ap1.module.stations = [sta]
        poll(3,1)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-1-id', 'sm': sta.mac_address})

        # Station moves to AP 2
        ap1.module.stations = []
        ap2.module.stations = [sta]
        poll(3,2)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-2-id', 'sm': sta.mac_address})

    def test_ap_swap_2(self):
        """Test a correct swapping of APs in "Activity Observer" mode when netmeta wasn't created for first"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        def poll(n=1, load=None):  # simulate a poll loop
            for _ in range(n):
                if load==1:
                    self._await(ap1.load())
                if load==2:
                    self._await(ap2.load())
                self.ctx.dl_rf_packets +=1
                sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
                if load ==1 and len(ap1.module.stations) > 0:
                    ap1.module.stations = [sta]
                if load ==2 and len(ap2.module.stations) > 0:
                    ap2.module.stations = [sta]
                    ap1.module.stations = []

        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True
        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
        self.ctx.cpe_mapping_polls = 4

        # Stations is on AP 1
        ap1.module.stations = [sta]
        poll(3,1)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNone(ref)

        # Stations moves to AP 2
        ap1.module.stations = []
        ap2.module.stations = [sta]
        poll(4,2)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-2-id', 'sm': sta.mac_address})



    def test_cpe_moving_between_aps_03(self):
        """Test a CPE can move between APs in "Activity Observer" mode (AP1-AP2-AP1) and only set for AP1 twice"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        def poll(n=1):  # simulate a poll loop
            for _ in range(n):
                self._await(ap1.load())
                self._await(ap2.load())
                self.ctx.dl_rf_packets +=1
                sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
                if len(ap1.module.stations) > 0:
                    ap1.module.stations = [sta]
                if len(ap2.module.stations) > 0:
                    ap2.module.stations = [sta]

        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True
        ap2 = ap.Ap(self.ctx, 'test-ap-2-id', 'test-ap-2', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True
        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
        self.ctx.cpe_mapping_polls = 3

        # Station is on AP 1
        ap1.module.stations = [sta]
        poll(3)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-1-id', 'sm': sta.mac_address})

        # Station moves to AP 2
        ap1.module.stations = []
        ap2.module.stations = [sta]
        poll(2)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNone(ref)
        #self.assertEqual(ref.attributes, {'ap': 'test-ap-2-id', 'sm': sta.mac_address})

        # Station is on AP 1
        ap1.module.stations = [sta]
        ap2.module.stations = []
        poll(15)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': 'test-ap-1-id', 'sm': sta.mac_address})

    def _check_ue_mapping(self, ue, ap_id, system=None):
        ip_address = getattr(ue, 'ip_address', None)
        name = getattr(ue, 'name', None)
        if ip_address:
            nm = self.ctx.netmeta_model.nms.get(ip_address)
            self.assertIsNotNone(nm)
            self.assertEqual(nm.attributes, {'session': Reference(ip_address), 'snmp_session': Reference(ip_address)})
            session = (self.ctx.netmeta_model.refs.get('snmp_session') or {}).get(ip_address)
            attrs = {'cpe_imsi': ue.imsi}
            if system:
                attrs['system'] = system
            self.assertEqual(session, NetworkMetadataReference('snmp_session', ip_address, attrs))
        cpe_imsi = (self.ctx.netmeta_model.refs.get('cpe_imsi') or {}).get(ue.imsi)
        attrs = {'ap': Reference(ap_id), 'imsi_service': Reference(ue.imsi)}
        if system:
            attrs['system'] = system
        self.assertEqual(cpe_imsi, NetworkMetadataReference('cpe_imsi', ue.imsi, attrs))

    def test_lte_01(self):
        """Test LTE eNodeB/UE mapping, including IP address."""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        Ue = namedtuple('Ue', ('imsi', 'ip_address', 'name', 'ul_rssi', 'gauge'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ues = [Ue('310120265624299', '192.0.2.101', 'test ue 1', -99.999, 1), Ue('502130123456789', None, 'test ue 2', -100.00, 1)]
        ap1.module.stations = ues
        self._await(ap1.load())
        self.assertEqual(len(self.ctx.netmeta_model.nms), 1)

        # Check Mappings
        self._check_ue_mapping(ues[0], ap1.id)
        self._check_ue_mapping(ues[1], ap1.id)

        # Check Metrics
        metrics = {m.labels.get('imsi'): m for m in self.ctx.metrics if m.name == 'cpe_stats'}
        self.assertEqual(len(metrics), 2)
        for ue in ues:
            m = metrics.get(ue.imsi)
            self.assertIsNotNone(m)
            fields = {f: getattr(ue, f) for f in ('ip_address', 'name', 'gauge') if getattr(ue, f, None) is not None}
            self.assertEqual(m.fields, fields)

        # Check Unmap
        ap1.module.stations = ues[:1]
        self._await(ap1.load())
        self._check_ue_mapping(ues[0], ap1.id)
        self.assertIsNone(self.ctx.netmeta_model.refs.get('cpe_imsi').get(ues[1].imsi))

        ap1.module.stations = []
        self._await(ap1.load())
        self.assertIsNone(self.ctx.netmeta_model.refs.get('cpe_imsi'))
        self.assertIsNone(self.ctx.netmeta_model.refs.get('snmp_session'))

    def test_lte_02(self):
        """Test UE with no IP address, make sure nothing breaks"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        Ue = namedtuple('Ue', ('imsi', 'name', 'ul_rssi'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ues = [Ue('310120265624299', 'test ue 1', -99.999)]
        ap1.module.stations = ues
        self._await(ap1.load())
        self.assertEqual(len(self.ctx.netmeta_model.nms), 0)
        self._check_ue_mapping(ues[0], ap1.id)

    def test_lte_03(self):
        """Delete an AP and make sure its IMSI mappings are deleted."""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        Ue = namedtuple('Ue', ('imsi', 'ip_address', 'name', 'ul_rssi'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ues = [Ue('310120265624299', '192.0.2.101', 'test ue 1', -99.999), Ue('502130123456789', None, 'test ue 2', -100.00)]
        ap1.module.stations = ues
        self._await(ap1.load())
        self.assertEqual(len(self.ctx.netmeta_model.nms), 1)
        # this is what the registry calls when an AP is deleted or times out
        ap1.offline = True
        self._await(ap1.remove_ref_offline())
        self.assertIsNone(self.ctx.netmeta_model.refs.get('cpe_imsi'))
        self.assertIsNone(self.ctx.netmeta_model.refs.get('snmp_session'))

    def test_lte_04(self):
        """Make sure an AP can be timed out properly multiple times."""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        Ue = namedtuple('Ue', ('imsi', 'ip_address', 'name', 'ul_rssi'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        ues = [Ue('310120265624299', '192.0.2.101', 'test ue 1', -99.999), Ue('502130123456789', None, 'test ue 2', -100.00)]
        ap1.module.stations = ues
        self._await(ap1.load())
        self.assertEqual(len(self.ctx.netmeta_model.nms), 1)
        self._check_ue_mapping(ues[0], ap1.id)
        self._check_ue_mapping(ues[1], ap1.id)
        # simulate it offline
        ap1.offline = True
        self._await(ap1.remove_ref_offline())
        self.assertIsNone(self.ctx.netmeta_model.refs.get('cpe_imsi'))
        self.assertIsNone(self.ctx.netmeta_model.refs.get('snmp_session'))
        # simulate it back online
        ap1.offline = False
        self._await(ap1.load())
        self._check_ue_mapping(ues[0], ap1.id)
        self._check_ue_mapping(ues[1], ap1.id)
        # simulate offline again
        ap1.offline = True
        self._await(ap1.remove_ref_offline())
        self.assertIsNone(self.ctx.netmeta_model.refs.get('cpe_imsi'))
        self.assertIsNone(self.ctx.netmeta_model.refs.get('snmp_session'))

    def test_lte_05(self):
        """Make sure we don't delete mappings created by a different system."""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.3.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        Ue = namedtuple('Ue', ('imsi', 'ip_address', 'name', 'ul_rssi'))
        ap1 = ap.Ap(self.ctx, 'test-ap-1-id', 'test-ap-1', 'site1', ne.snmp_ne,  None, 'ap', None)
        ap1.module = FakeModule()
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('cpe_imsi', '502130123456789', {'ap': Reference(ap1.id), 'imsi_service': Reference('502130123456789'), 'system': 'other'})))
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('snmp_session', '192.0.2.102', {'cpe_imsi': Reference('502130123456789'), 'system': 'other'})))
        self._await(self.ctx.netmeta_model.set_nm(NetworkMetadataEntity('192.0.2.102', time.time(), {'session': Reference('192.0.2.102'), 'snmp_session': Reference('192.0.2.102')})))
        ues = [Ue('310120265624299', '192.0.2.101', 'test ue 1', -99.999)]
        ap1.module.stations = ues
        self._await(ap1.load())
        self._check_ue_mapping(ues[0], ap1.id)
        self._check_ue_mapping(Ue('502130123456789', '192.0.2.102', 'test ue 2', -100.00), ap1.id, 'other')
        ap1.offline = True
        self._await(ap1.remove_ref_offline())
        self._check_ue_mapping(Ue('502130123456789', '192.0.2.102', 'test ue 2', -100.00), ap1.id, 'other')

    def test_cpe_mac_failed_to_be_deleted(self):
        """Test the cpe_mac is failed to be deleted if cpe mapping is disabled"""
        # set sm ref that cpe_mac will match
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('sm', '00:04:56:c4:fe:b8',{'service': '1'})))
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 1000.3.4.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.wait_for_ap_poll()
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 1)
        sta_mac_ref = refs.get('00:04:56:c4:fe:b8')
        self.assertEqual(sta_mac_ref, NetworkMetadataReference(type='cpe_mac', value='00:04:56:c4:fe:b8', attributes={'ap': 'TestNE', 'sm':'00:04:56:c4:fe:b8'}))

        # change cpe_mapping_enabled to False
        self.ctx.cpe_mapping_enabled = False
        # triggering delete
        self._await(self.reg.delete(ne.id))
        self.wait_for_ap_poll()
        # cpe_mac should not be deleted if cpe_mapping is not enabled
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 1)
        sta_mac_ref = refs.get('00:04:56:c4:fe:b8')
        self.assertEqual(sta_mac_ref, NetworkMetadataReference(type='cpe_mac', value='00:04:56:c4:fe:b8', attributes={'ap': 'TestNE', 'sm':'00:04:56:c4:fe:b8'}))

    def test_ne_update_1(self):
        """Test the creation of network_element_update_msg message and basic attributes get set properly."""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.5.6.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        self.assertEqual(ne_u.instance,  'unittest')
        self.assertEqual(ne_u.data.management_ip, ap.host)
        self.assertEqual(ne_u.active, ap.active)
        ne_u.poller_version = self.ctx.version
        self.assertEqual(ne_u.poller_version, "1.2.3")
        self.assertEqual(ne_u.pingable, False)  # STM-5609
        # wlan interface has sector
        self.assertEqual(ne_u.data.interfaces[1].sectors[0].ssid, str(ap.module.ssid))
        # wlan interface has links
        self.assertEqual(len(ne_u.data.interfaces[1].links), 5)
        # wlan interface has radios
        self.assertEqual(len(ne_u.data.interfaces[1].radios[0].streams[0].links), 5)

        # lan interface has no sector
        self.assertEqual(len(ne_u.data.interfaces[0].sectors), 0)
        # wlan interface has no links
        self.assertEqual(len(ne_u.data.interfaces[0].links), 0)
        # wlan interface has no radios
        self.assertEqual(len(ne_u.data.interfaces[0].radios), 0)

        # total interfaces num
        self.assertEqual(len(ne_u.data.interfaces), 2)
        # test peers len is equal num of stations
        self.assertEqual(len(ne_u.peers), len(ap.module.stations))
        # test peer has only one link (AP)
        self.assertEqual(len(ne_u.peers[0].interfaces[0].links), 1)
        # test peer's link mac address is wifi mac_address of AP
        self.assertEqual(ne_u.peers[0].interfaces[0].links[0].mac_address, ne_u.data.interfaces[1].mac_address)
        # test peer's link mac address is wifi mac_address of AP
        self.assertEqual(ne_u.peers[0].interfaces[0].links[0].mac_address, ne_u.data.interfaces[1].mac_address)
        # test peer's wifi mac address is equal to specific AP links mac_address
        self.assertEqual(ne_u.peers[0].interfaces[0].mac_address, ne_u.data.interfaces[1].links[0].mac_address)

    def test_ne_update_2(self):
        """Test that the company_uuid and element_uuid can be populated when the message is sent."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('ap_model', 'TestNE', {'uuid': '491c9831-7597-4a2e-8464-0e1eb84d4963'})))
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        ap.errctx = getattr(ap.module, 'errctx', ErrorContext())
        ap.network_element_update_msg = getattr(ap.module, 'network_element_update_msg', NetworkElementUpdate(errctx=ap.errctx))
        Sta = namedtuple('Sta', ('mac_address', 'poller_hash_id'))
        msg = ap.module.network_element_update_msg = network_element_update.NetworkElementUpdate()
        msg.data.poller_hash = b'\x01'
        self._await(ap.load())
        self._await(ap.send_update())
        ne_u = ap.network_element_update_msg
        self.assertEqual(UUID(bytes=ne_u.pb.company_uuid), self.ctx.company_uuid)
        self.assertEqual(UUID(bytes=ne_u.pb.element_uuid), UUID('491c9831-7597-4a2e-8464-0e1eb84d4963'))


    def test_poller_hash_01(self):
        """Poller hash uses the system mac address by default."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.5.6.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        ne_u= ap.module.network_element_update_msg
        self.assertEqual(ne_u.data.poller_hash, ap.make_poller_hash(mac=ap.module.system_mac_address).digest())
        for peer in ne_u.peers:
            self.assertEqual(peer.poller_hash, ap.make_poller_hash(mac=peer.system_mac_address).digest())

    def test_poller_hash_02(self):
        """Poller hash can be provided explicitly as part of the update message."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        Sta = namedtuple('Sta', ('mac_address', 'poller_hash_id'))
        ap.module.stations = [Sta('01:02:03:04:05:06', b'\x02')]
        msg = ap.module.network_element_update_msg = network_element_update.NetworkElementUpdate()
        msg.data.poller_hash = b'\x01'
        peer = msg.peers.add()
        peer.poller_hash = b'\x02'
        self._await(ap.load())
        ne_u = ap.module.network_element_update_msg
        self.assertEqual(ne_u.data.poller_hash, ap.make_poller_hash(hash_id=b'\x01').digest())
        self.assertEqual(ne_u.peers[0].poller_hash, ap.make_poller_hash(hash_id=b'\x02').digest())

    def test_poller_hash_03(self):
        """Poller hash can be provided for interfaces."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        msg = ap.module.network_element_update_msg = network_element_update.NetworkElementUpdate()
        msg.data.poller_hash = b'\x01'
        intf = msg.data.interfaces.add()
        intf.id = 'eth'
        peer = msg.peers.add()
        peer.poller_hash = b'\x02'
        # interface with poller_hash
        peer_intf = peer.interfaces.add()
        peer_intf.id = '1'
        peer_intf.name = 'wlan0'
        peer_intf.mac_address = '01:02:03:04:05:0a'
        peer_intf.poller_hash = b'wlan'
        # interface with mac
        peer_intf = peer.interfaces.add()
        peer_intf.id = '2'
        peer_intf.name = 'eth0'
        peer_intf.mac_address = '01:02:03:04:05:0b'
        # interface with name
        peer_intf = peer.interfaces.add()
        peer_intf.id = '3'
        peer_intf.name = 'eth1'
        # interface with id
        peer_intf = peer.interfaces.add()
        peer_intf.id = '4'
        self._await(ap.load())
        ne_u = ap.module.network_element_update_msg
        self.assertEqual(ne_u.data.poller_hash, ap.make_poller_hash(hash_id=b'\x01').digest())
        self.assertEqual(ne_u.peers[0].poller_hash, ap.make_poller_hash(hash_id=b'\x02').digest())
        ph = ap.make_poller_hash(hash_id=b'\x01')
        ph.update(b'eth')
        self.assertEqual(ne_u.data.interfaces[0].poller_hash, ph.digest())
        ph = ap.make_poller_hash(hash_id=b'\x02')
        ph.update(b'wlan')
        self.assertEqual(ne_u.peers[0].interfaces[0].poller_hash, ph.digest())
        ph = ap.make_poller_hash(hash_id=b'\x02')
        ph.update(int('01:02:03:04:05:0b'.replace(':', ''), 16).to_bytes(6, byteorder='big'))
        self.assertEqual(ne_u.peers[0].interfaces[1].poller_hash, ph.digest())
        ph = ap.make_poller_hash(hash_id=b'\x02')
        ph.update(b'eth1')
        self.assertEqual(ne_u.peers[0].interfaces[2].poller_hash, ph.digest())
        ph = ap.make_poller_hash(hash_id=b'\x02')
        ph.update(b'4')
        self.assertEqual(ne_u.peers[0].interfaces[3].poller_hash, ph.digest())

    def test_poller_hash_04(self):
        """STM-7056 poller hash should not be sent if no system mac address or
           poller_hash from the module is provided."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        msg = ap.module.network_element_update_msg = network_element_update.NetworkElementUpdate()
        self._await(ap.load())
        self.assertFalse(ap.module.network_element_update_msg.data.pb.HasField('poller_hash'))

    def test_poller_hash_links_predefined_hash(self):
        """Poller hash can be provided for links."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        msg = ap.module.network_element_update_msg = network_element_update.NetworkElementUpdate()
        msg.data.poller_hash = b'serAP'

        intf = msg.data.interfaces.add()
        intf.poller_hash = b'ser1wl1'
        radio = intf.radios.add()
        radio.id = '1'
        stream = radio.streams.add()

        peer = msg.peers.add()
        peer.poller_hash = b'serCPE1'
        # interface with poller_hash
        peer_intf = peer.interfaces.add()
        peer_intf.id = 'wl1'
        peer_intf.poller_hash = b'serCPE1wl1'
        p1link = peer_intf.links.add()
        p1link.poller_hash = intf.poller_hash
        p1r = peer_intf.radios.add()
        p1s = p1r.streams.add()
        p1wlink = p1s.links.add()
        p1wlink.poller_hash = intf.poller_hash

        peer2 = msg.peers.add()
        peer2.poller_hash = b'serCPE2'
        # interface with poller_hash
        peer2_intf = peer2.interfaces.add()
        peer2_intf.id = 'wl1'
        peer2_intf.poller_hash =  b'serCPE2wl1'
        p2link = peer2_intf.links.add()
        p2link.poller_hash = intf.poller_hash
        p2r = peer2_intf.radios.add()
        p2s = p2r.streams.add()
        p2wlink = p2s.links.add()
        p2wlink.poller_hash = intf.poller_hash


        link1 = intf.links.add()
        wlink1 = stream.links.add()
        link1.poller_hash = peer_intf.poller_hash
        wlink1.poller_hash = peer_intf.poller_hash

        link2 = intf.links.add()
        wlink2 = stream.links.add()
        link2.poller_hash = peer2_intf.poller_hash
        wlink2.poller_hash = peer2_intf.poller_hash


        self._await(ap.load())
        ne_u = ap.module.network_element_update_msg
        phwl1 = ap.make_poller_hash(hash_id=b'serAP')
        phwl1.update(b'ser1wl1')
        self.assertEqual(ne_u.data.poller_hash, ap.make_poller_hash(hash_id=b'serAP').digest())
        self.assertEqual(ne_u.data.interfaces[0].poller_hash, phwl1.digest())

        #test link 1 (link1 poller hash == peer1 wlan interface poller hash)
        self.assertEqual(ne_u.data.interfaces[0].links[0].poller_hash, ne_u.peers[0].interfaces[0].poller_hash)
        self.assertEqual(ne_u.data.interfaces[0].radios[0].streams[0].links[0].poller_hash, ne_u.peers[0].interfaces[0].poller_hash)

        #test link 2 (link2 poller hash == peer2 wlan interface poller hash)
        self.assertEqual(ne_u.data.interfaces[0].links[1].poller_hash, ne_u.peers[1].interfaces[0].poller_hash)
        self.assertEqual(ne_u.data.interfaces[0].radios[0].streams[0].links[1].poller_hash, ne_u.peers[1].interfaces[0].poller_hash)

        #test peer 1
        self.assertEqual(ne_u.peers[0].poller_hash, ap.make_poller_hash(hash_id=b'serCPE1').digest())
        ph = ap.make_poller_hash(hash_id=b'serCPE1')
        ph.update(b'serCPE1wl1')
        self.assertEqual(ne_u.peers[0].interfaces[0].poller_hash, ph.digest())

        self.assertEqual(ne_u.peers[0].interfaces[0].links[0].poller_hash, ne_u.data.interfaces[0].poller_hash)
        self.assertEqual(ne_u.peers[0].interfaces[0].radios[0].streams[0].links[0].poller_hash, ne_u.data.interfaces[0].poller_hash)

        #test peer 2
        self.assertEqual(ne_u.peers[1].poller_hash, ap.make_poller_hash(hash_id=b'serCPE2').digest())
        ph = ap.make_poller_hash(hash_id=b'serCPE2')
        ph.update(b'serCPE2wl1')
        self.assertEqual(ne_u.peers[1].interfaces[0].poller_hash, ph.digest())

        self.assertEqual(ne_u.peers[1].interfaces[0].links[0].poller_hash, ne_u.data.interfaces[0].poller_hash)
        self.assertEqual(ne_u.peers[1].interfaces[0].radios[0].streams[0].links[0].poller_hash, ne_u.data.interfaces[0].poller_hash)

    def test_poller_hash_links_macs(self):
        """Poller hash can be provided for links."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        msg = ap.module.network_element_update_msg = network_element_update.NetworkElementUpdate()
        msg.data.system_mac_address = '01:02:03:04:05:0a'
        # msg.data.poller_hash = b'\x01'

        lan_intf = msg.data.interfaces.add()
        lan_intf.id = "1"
        lan_intf.name = "LAN"
        lan_intf.mac_address = '01:02:03:04:05:0a'

        radio_intf = msg.data.interfaces.add()
        radio_intf.id = "2"
        radio_intf.name = "WLAN"
        radio_intf.mac_address = '01:02:03:04:05:0b'
        radio = radio_intf.radios.add()
        radio.id = '1'
        stream = radio.streams.add()

        peer = msg.peers.add()
        # peer.poller_hash = b'\x02'
        peer.system_mac_address = '01:02:03:04:05:01'
        peer_radio_intf = peer.interfaces.add()
        peer_radio_intf.id = '1'
        peer_radio_intf.mac_address = b'01:02:03:04:05:02'
        p1link = peer_radio_intf.links.add()
        p1link.mac_address = radio_intf.mac_address
        p1r = peer_radio_intf.radios.add()
        p1s = p1r.streams.add()
        p1wlink = p1s.links.add()
        p1wlink.mac_address = radio_intf.mac_address

        peer2 = msg.peers.add()
        # peer2.poller_hash = b'\x03'
        peer2.system_mac_address = '11:02:03:04:05:01'
        peer2_radio_intf = peer2.interfaces.add()
        peer2_radio_intf.id = '1'
        peer2_radio_intf.mac_address = b'11:02:03:04:05:02'
        p2link = peer2_radio_intf.links.add()
        p2link.mac_address = radio_intf.mac_address
        p2r = peer2_radio_intf.radios.add()
        p2s = p2r.streams.add()
        p2wlink = p2s.links.add()
        p2wlink.mac_address = radio_intf.mac_address


        link1 = radio_intf.links.add()
        wlink1 = stream.links.add()
        link1.mac_address = peer_radio_intf.mac_address
        wlink1.mac_address = peer_radio_intf.mac_address

        link2 = radio_intf.links.add()
        wlink2 = stream.links.add()
        link2.mac_address = peer2_radio_intf.mac_address
        wlink2.mac_address = peer2_radio_intf.mac_address


        self._await(ap.load())

        ne_u = ap.module.network_element_update_msg
        self.assertEqual(ne_u.data.poller_hash, ap.make_poller_hash(mac=msg.data.system_mac_address).digest())
        ph = ap.make_poller_hash(mac=msg.data.system_mac_address)
        ph.update(int(radio_intf.mac_address.replace(':', ''), 16).to_bytes(6, byteorder='big'))
        self.assertEqual(ne_u.data.interfaces[1].poller_hash, ph.digest())

        # test link 1 (link1 poller hash == peer1 wlan interface poller hash)
        self.assertEqual(ne_u.data.interfaces[1].links[0].poller_hash, ne_u.peers[0].interfaces[0].poller_hash)
        self.assertEqual(ne_u.data.interfaces[1].radios[0].streams[0].links[0].poller_hash, ne_u.peers[0].interfaces[0].poller_hash)

        # test link 2 (link2 poller hash == peer2 wlan interface poller hash)
        self.assertEqual(ne_u.data.interfaces[1].links[1].poller_hash, ne_u.peers[1].interfaces[0].poller_hash)
        self.assertEqual(ne_u.data.interfaces[1].radios[0].streams[0].links[1].poller_hash, ne_u.peers[1].interfaces[0].poller_hash)


        # test peer 1
        self.assertEqual(ne_u.peers[0].poller_hash,ap.make_poller_hash(mac=peer.system_mac_address).digest())
        ph = ap.make_poller_hash(mac=peer.system_mac_address)
        ph.update(int(peer_radio_intf.mac_address.replace(':', ''), 16).to_bytes(6, byteorder='big'))
        self.assertEqual(ne_u.peers[0].interfaces[0].poller_hash, ph.digest())

        self.assertEqual(ne_u.peers[0].interfaces[0].links[0].poller_hash, ne_u.data.interfaces[1].poller_hash)
        self.assertEqual(ne_u.peers[0].interfaces[0].radios[0].streams[0].links[0].poller_hash, ne_u.data.interfaces[1].poller_hash)

        # test peer 2
        self.assertEqual(ne_u.peers[1].poller_hash, ap.make_poller_hash(mac=peer2.system_mac_address).digest())
        ph = ap.make_poller_hash(mac=peer2.system_mac_address)
        ph.update(int(peer2_radio_intf.mac_address.replace(':', ''), 16).to_bytes(6, byteorder='big'))
        self.assertEqual(ne_u.peers[1].interfaces[0].poller_hash, ph.digest())

        self.assertEqual(ne_u.peers[1].interfaces[0].links[0].poller_hash, ne_u.data.interfaces[1].poller_hash)
        self.assertEqual(ne_u.peers[1].interfaces[0].radios[0].streams[0].links[0].poller_hash, ne_u.data.interfaces[1].poller_hash)


    # STM-4730
    def test_cpe_map_multiple_aps_05(self):
        """Test a CPE MAC is created for an one AP from three active APs by alphabetical order """

        def poll(n=1):  # simulate a poll loop
            for _ in range(n):
                if ap3:
                    self._await(ap3.load())
                if ap1:
                    self._await(ap1.load())
                if ap2:
                    self._await(ap2.load())

                self.ctx.dl_rf_packets +=1
                # making all 3 active APs
                sta_active = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)

                if ap3 and len(ap3.module.stations) > 0:
                    ap3.module.stations = [sta_active]
                if ap1 and len(ap1.module.stations) > 0:
                    ap1.module.stations = [sta_active]
                if ap2 and len(ap2.module.stations) > 0:
                    ap2.module.stations = [sta_active]

        ne1 = self._await(self.reg.set('TestNE_B', {'name': 'Test Element 1', 'site': 'My Site'}))
        ap1 = Ap(self.ctx, ne1.id, ne1.name, ne1.site, ne1.snmp_ne, ne1.snmp_ne_v3, ne1.ne_type, ne1.holdoff)
        ap1.module = FakeModule()
        ap1.module.activity_observer = True

        ne2 = self._await(self.reg.set('TestNE_C', {'name': 'Test Element 2', 'site': 'My Site'}))
        ap2 = Ap(self.ctx, ne2.id, ne2.name, ne2.site, ne2.snmp_ne, ne2.snmp_ne_v3, ne2.ne_type, ne2.holdoff)
        ap2.module = FakeModule()
        ap2.module.activity_observer = True

        ne3 = self._await(self.reg.set('TestNE_A', {'name': 'Test Element 3', 'site': 'My Site'}))
        ap3 = Ap(self.ctx, ne3.id, ne3.name, ne3.site, ne3.snmp_ne, ne3.snmp_ne_v3, ne3.ne_type, ne3.holdoff)
        ap3.module = FakeModule()
        ap3.module.activity_observer = True

        Sta = namedtuple('Sta', ('mac_address','dl_rf_packets'))

        sta = Sta('02:00:00:00:00:01',self.ctx.dl_rf_packets)
        self.ctx.cpe_mapping_polls = 4

        ap3.module.stations = [sta]
        ap1.module.stations = [sta]
        ap2.module.stations = [sta]
        poll(4)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': ap3.id, 'sm': sta.mac_address})

        # triggering offline event and removing ap3 from polling , should be assigned to next AP by alphabetcal order
        ne3.offline = True
        ap3 = None
        poll(10)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac') or {}
        ref = refs.get(sta.mac_address)
        self.assertIsNotNone(ref)
        self.assertEqual(ref.attributes, {'ap': ap1.id, 'sm': sta.mac_address})

    def test_bridge_map_01(self):
        """Test that bridged MACs are mapped to the AP if bridge table extraction is enabled."""
        self.ctx.bridge_mapping_enabled = True
        # This snmp dump has two stations: one with five bridged MACs and one with none.
        df = self.datafiles.get(('cambium-epmp.epmp3000.ePMP 3000.4.6.0.1.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.wait_for_ap_poll()
        self.assertEqual(len(ne.module.stations), 2)
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 7)
        checked = 0
        for sta in ne.module.stations:
            ref = refs.get(sta.mac_address)
            exp_ref = NetworkMetadataReference(type='cpe_mac', value=sta.mac_address, attributes={'ap': 'TestNE', 'sm': sta.mac_address})
            self.assertEqual(exp_ref, ref)
            for dev_mac in sta.dev_macs or []:
                ref = refs.get(dev_mac)
                exp_ref = exp_ref._replace(value = dev_mac)
                self.assertEqual(exp_ref, ref)
                checked += 1
            checked += 1
        self.assertEqual(checked, len(refs))


    def test_bridge_map_02(self):
        """Test edge cases around bridge table mapping"""
        self.ctx.bridge_mapping_enabled = True
        ap = Ap(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'ap', None)
        ap.module = FakeModule()
        Sta = namedtuple('Sta', ('mac_address', 'dev_macs'))
        ap.module.stations = [Sta('01:00:00:00:00:01', ['02:00:00:00:00:01'])]
        self._await(ap.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 2)
        self.assertEqual(refs['01:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='01:00:00:00:00:01', attributes={'ap': 'TestAP', 'sm': '01:00:00:00:00:01'}))
        self.assertEqual(refs['02:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='02:00:00:00:00:01', attributes={'ap': 'TestAP', 'sm': '01:00:00:00:00:01'}))
        # remove the device mac
        ap.module.stations = [Sta('01:00:00:00:00:01', {})]
        self._await(ap.load())
        self.assertEqual(len(refs), 1)
        self.assertEqual(refs['01:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='01:00:00:00:00:01', attributes={'ap': 'TestAP', 'sm': '01:00:00:00:00:01'}))
        # remove the device
        ap.module.stations = []
        self._await(ap.load())
        self.assertEqual(len(refs), 0)
        # put the original mappings back
        ap.module.stations = [Sta('01:00:00:00:00:01', ['02:00:00:00:00:01'])]
        self._await(ap.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 2)
        # offline the AP
        ap.offline = True
        self._await(ap.remove_ref_offline())  # would be called by poll()
        self.assertEqual(len(refs), 0)

    def test_bridge_map_03(self):
        """Test case where a MAC is bridged through two APs."""
        self.ctx.bridge_mapping_enabled = True
        ap1 = Ap(self.ctx, 'TestAP1', 'TestAP1', 'MySite', FakeClient('MyHost1'), None, 'ap', None)
        ap1.module = FakeModule()
        ap2 = Ap(self.ctx, 'TestAP2', 'TestAP2', 'MySite', FakeClient('MyHost2'), None, 'ap', None)
        ap2.module = FakeModule()
        Sta = namedtuple('Sta', ('mac_address', 'dev_macs'))
        ap1.module.stations = [Sta('01:00:00:00:00:01', ['02:00:00:00:00:01', '03:00:00:00:00:01'])]
        ap2.module.stations = [Sta('a1:00:00:00:00:01', ['03:00:00:00:00:01'])]
        self._await(ap1.load())
        self._await(ap2.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 4)
        self.assertEqual(refs['01:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='01:00:00:00:00:01', attributes={'ap': 'TestAP1', 'sm': '01:00:00:00:00:01'}))
        self.assertEqual(refs['a1:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='a1:00:00:00:00:01', attributes={'ap': 'TestAP2', 'sm': 'a1:00:00:00:00:01'}))
        self.assertEqual(refs['02:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='02:00:00:00:00:01', attributes={'ap': 'TestAP1', 'sm': '01:00:00:00:00:01'}))
        # The cpe mapped to both APs gets mapped to the one with the smallest bridge table.
        self.assertEqual(refs['03:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='03:00:00:00:00:01', attributes={'ap': 'TestAP2', 'sm': 'a1:00:00:00:00:01'}))

    def test_net_macs_01(self):
        """Test that MAC addresses given by "net_macs" are mapped."""
        ap = Ap(self.ctx, 'TestAP', 'TestAP', 'MySite', FakeClient('MyHost'), None, 'ap', None)
        ap.module = FakeModule()
        Sta = namedtuple('Sta', ('mac_address', 'alt_macs', 'net_macs'))
        ap.module.stations = [Sta('01:00:00:00:00:01', ['01:00:00:00:00:02', '01:00:00:00:00:03'], ['01:00:00:00:00:03'])]
        self._await(ap.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 2)
        self.assertEqual(refs['01:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='01:00:00:00:00:01', attributes={'ap': 'TestAP', 'sm': '01:00:00:00:00:01'}))
        self.assertEqual(refs['01:00:00:00:00:03'], NetworkMetadataReference(type='cpe_mac', value='01:00:00:00:00:03', attributes={'ap': 'TestAP', 'sm': '01:00:00:00:00:01'}))

    def test_duplicate_ips(self):
        """STM-5600 test that duplicate station IPs are filtered from metrics."""
        df = self.datafiles.get(("ubnt.airmax-m.NanoStation loco M900.v6.2.0.01", 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        # this dump has several stations with IP address 192.168.1.20
        polled_ips = {x.ip_address for x in ap.module.stations}
        self.assertIn("192.168.1.20", polled_ips)
        metrics_ips = {x.fields['ip_address'] for x in self.ctx.metrics if x.name == 'cpe_stats' and 'ip_address' in x.fields}
        self.assertEqual(metrics_ips, polled_ips - {'192.168.1.20'})
        ne_u = ap.module.network_element_update_msg
        peers_ips = {x.management_ip for x in ne_u.peers if x.management_ip}
        self.assertEqual(metrics_ips, peers_ips - {'192.168.1.20'})

    def test_model_match_01(self):
        """Test model matching based on ap_info.yaml"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.6.1.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        msg = ap.module.network_element_update_msg
        self.assertEqual(len(msg.peers), 12)
        peer_models = Counter([x.model for x in msg.peers])
        self.assertEqual(peer_models, {'ePMP Force 180': 11, 'ePMP Force 190': 1})

    def test_poll_ap_router_01(self):
        """Test that an AP can also be polled as a router."""
        self.ctx.bridge_mapping_enabled = True
        df = self.datafiles.get(('cmts.cisco.uBR10012.12.2(33)SCG6.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        self._await(ap.poll())
        self.wait_for_ap_poll()
        snmp_session_refs = {
                '10.0.196.10': NetworkMetadataReference(type='snmp_session', value='10.0.196.10', attributes={'cpe_mac': '94:87:7c:00:27:12', 'dev_mac': '94:87:7c:00:27:12', 'net_mac': '94:87:7c:00:27:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.100': NetworkMetadataReference(type='snmp_session', value='10.0.196.100', attributes={'cpe_mac': '14:ab:f0:d0:22:e2', 'dev_mac': '14:ab:f0:d0:22:e2', 'net_mac': '14:ab:f0:d0:22:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.101': NetworkMetadataReference(type='snmp_session', value='10.0.196.101', attributes={'cpe_mac': '08:3e:0c:ec:ce:c2', 'dev_mac': '08:3e:0c:ec:ce:c2', 'net_mac': '08:3e:0c:ec:ce:c2', 'router_id': 'ap.TestNE'}),
                '10.0.196.102': NetworkMetadataReference(type='snmp_session', value='10.0.196.102', attributes={'cpe_mac': '38:70:0c:8f:5b:5a', 'dev_mac': '38:70:0c:8f:5b:5a', 'net_mac': '38:70:0c:8f:5b:5a', 'router_id': 'ap.TestNE'}),
                '10.0.196.103': NetworkMetadataReference(type='snmp_session', value='10.0.196.103', attributes={'cpe_mac': '2c:99:24:8e:56:13', 'dev_mac': '2c:99:24:8e:56:13', 'net_mac': '2c:99:24:8e:56:13', 'router_id': 'ap.TestNE'}),
                '10.0.196.104': NetworkMetadataReference(type='snmp_session', value='10.0.196.104', attributes={'cpe_mac': '78:71:9c:94:20:72', 'dev_mac': '78:71:9c:94:20:72', 'net_mac': '78:71:9c:94:20:72', 'router_id': 'ap.TestNE'}),
                '10.0.196.105': NetworkMetadataReference(type='snmp_session', value='10.0.196.105', attributes={'cpe_mac': '38:70:0c:8f:6b:ec', 'dev_mac': '38:70:0c:8f:6b:ec', 'net_mac': '38:70:0c:8f:6b:ec', 'router_id': 'ap.TestNE'}),
                '10.0.196.106': NetworkMetadataReference(type='snmp_session', value='10.0.196.106', attributes={'cpe_mac': '60:19:71:ee:44:72', 'dev_mac': '60:19:71:ee:44:72', 'net_mac': '60:19:71:ee:44:72', 'router_id': 'ap.TestNE'}),
                '10.0.196.107': NetworkMetadataReference(type='snmp_session', value='10.0.196.107', attributes={'cpe_mac': '38:70:0c:6a:7e:c7', 'dev_mac': '38:70:0c:6a:7e:c7', 'net_mac': '38:70:0c:6a:7e:c7', 'router_id': 'ap.TestNE'}),
                '10.0.196.109': NetworkMetadataReference(type='snmp_session', value='10.0.196.109', attributes={'cpe_mac': '38:70:0c:63:a7:54', 'dev_mac': '38:70:0c:63:a7:54', 'net_mac': '38:70:0c:63:a7:54', 'router_id': 'ap.TestNE'}),
                '10.0.196.11': NetworkMetadataReference(type='snmp_session', value='10.0.196.11', attributes={'cpe_mac': 'ac:b3:13:8f:85:32', 'dev_mac': 'ac:b3:13:8f:85:32', 'net_mac': 'ac:b3:13:8f:85:32', 'router_id': 'ap.TestNE'}),
                '10.0.196.110': NetworkMetadataReference(type='snmp_session', value='10.0.196.110', attributes={'cpe_mac': '5c:8f:e0:27:ed:12', 'dev_mac': '5c:8f:e0:27:ed:12', 'net_mac': '5c:8f:e0:27:ed:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.111': NetworkMetadataReference(type='snmp_session', value='10.0.196.111', attributes={'cpe_mac': 'ac:b3:13:ac:c4:c2', 'dev_mac': 'ac:b3:13:ac:c4:c2', 'net_mac': 'ac:b3:13:ac:c4:c2', 'router_id': 'ap.TestNE'}),
                '10.0.196.112': NetworkMetadataReference(type='snmp_session', value='10.0.196.112', attributes={'cpe_mac': '38:70:0c:ea:0e:fa', 'dev_mac': '38:70:0c:ea:0e:fa', 'net_mac': '38:70:0c:ea:0e:fa', 'router_id': 'ap.TestNE'}),
                '10.0.196.113': NetworkMetadataReference(type='snmp_session', value='10.0.196.113', attributes={'cpe_mac': '40:70:09:ac:13:82', 'dev_mac': '40:70:09:ac:13:82', 'net_mac': '40:70:09:ac:13:82', 'router_id': 'ap.TestNE'}),
                '10.0.196.114': NetworkMetadataReference(type='snmp_session', value='10.0.196.114', attributes={'cpe_mac': '38:70:0c:f2:84:3a', 'dev_mac': '38:70:0c:f2:84:3a', 'net_mac': '38:70:0c:f2:84:3a', 'router_id': 'ap.TestNE'}),
                '10.0.196.115': NetworkMetadataReference(type='snmp_session', value='10.0.196.115', attributes={'cpe_mac': '3c:7a:8a:f2:92:12', 'dev_mac': '3c:7a:8a:f2:92:12', 'net_mac': '3c:7a:8a:f2:92:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.116': NetworkMetadataReference(type='snmp_session', value='10.0.196.116', attributes={'cpe_mac': '2c:99:24:52:71:e3', 'dev_mac': '2c:99:24:52:71:e3', 'net_mac': '2c:99:24:52:71:e3', 'router_id': 'ap.TestNE'}),
                '10.0.196.118': NetworkMetadataReference(type='snmp_session', value='10.0.196.118', attributes={'cpe_mac': '5c:8f:e0:1b:c1:5a', 'dev_mac': '5c:8f:e0:1b:c1:5a', 'net_mac': '5c:8f:e0:1b:c1:5a', 'router_id': 'ap.TestNE'}),
                '10.0.196.119': NetworkMetadataReference(type='snmp_session', value='10.0.196.119', attributes={'cpe_mac': '38:70:0c:02:e8:b8', 'dev_mac': '38:70:0c:02:e8:b8', 'net_mac': '38:70:0c:02:e8:b8', 'router_id': 'ap.TestNE'}),
                '10.0.196.12': NetworkMetadataReference(type='snmp_session', value='10.0.196.12', attributes={'cpe_mac': 'ac:b3:13:0c:b0:52', 'dev_mac': 'ac:b3:13:0c:b0:52', 'net_mac': 'ac:b3:13:0c:b0:52', 'router_id': 'ap.TestNE'}),
                '10.0.196.120': NetworkMetadataReference(type='snmp_session', value='10.0.196.120', attributes={'cpe_mac': '94:87:7c:0b:19:02', 'dev_mac': '94:87:7c:0b:19:02', 'net_mac': '94:87:7c:0b:19:02', 'router_id': 'ap.TestNE'}),
                '10.0.196.121': NetworkMetadataReference(type='snmp_session', value='10.0.196.121', attributes={'cpe_mac': '78:71:9c:9b:e1:12', 'dev_mac': '78:71:9c:9b:e1:12', 'net_mac': '78:71:9c:9b:e1:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.122': NetworkMetadataReference(type='snmp_session', value='10.0.196.122', attributes={'cpe_mac': '78:71:9c:90:f2:b2', 'dev_mac': '78:71:9c:90:f2:b2', 'net_mac': '78:71:9c:90:f2:b2', 'router_id': 'ap.TestNE'}),
                '10.0.196.123': NetworkMetadataReference(type='snmp_session', value='10.0.196.123', attributes={'cpe_mac': '3c:7a:8a:7e:91:fa', 'dev_mac': '3c:7a:8a:7e:91:fa', 'net_mac': '3c:7a:8a:7e:91:fa', 'router_id': 'ap.TestNE'}),
                '10.0.196.124': NetworkMetadataReference(type='snmp_session', value='10.0.196.124', attributes={'cpe_mac': '3c:7a:8a:ec:11:52', 'dev_mac': '3c:7a:8a:ec:11:52', 'net_mac': '3c:7a:8a:ec:11:52', 'router_id': 'ap.TestNE'}),
                '10.0.196.125': NetworkMetadataReference(type='snmp_session', value='10.0.196.125', attributes={'cpe_mac': '3c:7a:8a:ec:11:2a', 'dev_mac': '3c:7a:8a:ec:11:2a', 'net_mac': '3c:7a:8a:ec:11:2a', 'router_id': 'ap.TestNE'}),
                '10.0.196.126': NetworkMetadataReference(type='snmp_session', value='10.0.196.126', attributes={'cpe_mac': '90:1a:ca:d9:7a:d2', 'dev_mac': '90:1a:ca:d9:7a:d2', 'net_mac': '90:1a:ca:d9:7a:d2', 'router_id': 'ap.TestNE'}),
                '10.0.196.127': NetworkMetadataReference(type='snmp_session', value='10.0.196.127', attributes={'cpe_mac': 'ac:b3:13:85:c2:72', 'dev_mac': 'ac:b3:13:85:c2:72', 'net_mac': 'ac:b3:13:85:c2:72', 'router_id': 'ap.TestNE'}),
                '10.0.196.128': NetworkMetadataReference(type='snmp_session', value='10.0.196.128', attributes={'cpe_mac': '08:3e:0c:ec:43:02', 'dev_mac': '08:3e:0c:ec:43:02', 'net_mac': '08:3e:0c:ec:43:02', 'router_id': 'ap.TestNE'}),
                '10.0.196.129': NetworkMetadataReference(type='snmp_session', value='10.0.196.129', attributes={'cpe_mac': '78:71:9c:a1:7d:c2', 'dev_mac': '78:71:9c:a1:7d:c2', 'net_mac': '78:71:9c:a1:7d:c2', 'router_id': 'ap.TestNE'}),
                '10.0.196.13': NetworkMetadataReference(type='snmp_session', value='10.0.196.13', attributes={'cpe_mac': 'ac:b3:13:b6:cb:92', 'dev_mac': 'ac:b3:13:b6:cb:92', 'net_mac': 'ac:b3:13:b6:cb:92', 'router_id': 'ap.TestNE'}),
                '10.0.196.130': NetworkMetadataReference(type='snmp_session', value='10.0.196.130', attributes={'cpe_mac': 'ac:b3:13:b5:5b:72', 'dev_mac': 'ac:b3:13:b5:5b:72', 'net_mac': 'ac:b3:13:b5:5b:72', 'router_id': 'ap.TestNE'}),
                '10.0.196.132': NetworkMetadataReference(type='snmp_session', value='10.0.196.132', attributes={'cpe_mac': '38:70:0c:40:aa:0a', 'dev_mac': '38:70:0c:40:aa:0a', 'net_mac': '38:70:0c:40:aa:0a', 'router_id': 'ap.TestNE'}),
                '10.0.196.133': NetworkMetadataReference(type='snmp_session', value='10.0.196.133', attributes={'cpe_mac': '2c:99:24:8e:59:2b', 'dev_mac': '2c:99:24:8e:59:2b', 'net_mac': '2c:99:24:8e:59:2b', 'router_id': 'ap.TestNE'}),
                '10.0.196.134': NetworkMetadataReference(type='snmp_session', value='10.0.196.134', attributes={'cpe_mac': '9c:34:26:4e:92:5a', 'dev_mac': '9c:34:26:4e:92:5a', 'net_mac': '9c:34:26:4e:92:5a', 'router_id': 'ap.TestNE'}),
                '10.0.196.135': NetworkMetadataReference(type='snmp_session', value='10.0.196.135', attributes={'cpe_mac': '3c:7a:8a:75:a7:42', 'dev_mac': '3c:7a:8a:75:a7:42', 'net_mac': '3c:7a:8a:75:a7:42', 'router_id': 'ap.TestNE'}),
                '10.0.196.136': NetworkMetadataReference(type='snmp_session', value='10.0.196.136', attributes={'cpe_mac': '2c:99:24:8e:4d:23', 'dev_mac': '2c:99:24:8e:4d:23', 'net_mac': '2c:99:24:8e:4d:23', 'router_id': 'ap.TestNE'}),
                '10.0.196.138': NetworkMetadataReference(type='snmp_session', value='10.0.196.138', attributes={'cpe_mac': '38:70:0c:85:c3:8a', 'dev_mac': '38:70:0c:85:c3:8a', 'net_mac': '38:70:0c:85:c3:8a', 'router_id': 'ap.TestNE'}),
                '10.0.196.139': NetworkMetadataReference(type='snmp_session', value='10.0.196.139', attributes={'cpe_mac': '38:70:0c:6a:bf:5d', 'dev_mac': '38:70:0c:6a:bf:5d', 'net_mac': '38:70:0c:6a:bf:5d', 'router_id': 'ap.TestNE'}),
                '10.0.196.14': NetworkMetadataReference(type='snmp_session', value='10.0.196.14', attributes={'cpe_mac': 'ac:b3:13:80:8a:02', 'dev_mac': 'ac:b3:13:80:8a:02', 'net_mac': 'ac:b3:13:80:8a:02', 'router_id': 'ap.TestNE'}),
                '10.0.196.140': NetworkMetadataReference(type='snmp_session', value='10.0.196.140', attributes={'cpe_mac': '38:70:0c:8f:01:b1', 'dev_mac': '38:70:0c:8f:01:b1', 'net_mac': '38:70:0c:8f:01:b1', 'router_id': 'ap.TestNE'}),
                '10.0.196.141': NetworkMetadataReference(type='snmp_session', value='10.0.196.141', attributes={'cpe_mac': '38:70:0c:8f:85:7d', 'dev_mac': '38:70:0c:8f:85:7d', 'net_mac': '38:70:0c:8f:85:7d', 'router_id': 'ap.TestNE'}),
                '10.0.196.142': NetworkMetadataReference(type='snmp_session', value='10.0.196.142', attributes={'cpe_mac': '38:70:0c:8f:a6:0e', 'dev_mac': '38:70:0c:8f:a6:0e', 'net_mac': '38:70:0c:8f:a6:0e', 'router_id': 'ap.TestNE'}),
                '10.0.196.143': NetworkMetadataReference(type='snmp_session', value='10.0.196.143', attributes={'cpe_mac': '38:70:0c:8f:bc:a4', 'dev_mac': '38:70:0c:8f:bc:a4', 'net_mac': '38:70:0c:8f:bc:a4', 'router_id': 'ap.TestNE'}),
                '10.0.196.144': NetworkMetadataReference(type='snmp_session', value='10.0.196.144', attributes={'cpe_mac': '9c:34:26:44:02:3a', 'dev_mac': '9c:34:26:44:02:3a', 'net_mac': '9c:34:26:44:02:3a', 'router_id': 'ap.TestNE'}),
                '10.0.196.145': NetworkMetadataReference(type='snmp_session', value='10.0.196.145', attributes={'cpe_mac': '5c:8f:e0:28:69:92', 'dev_mac': '5c:8f:e0:28:69:92', 'net_mac': '5c:8f:e0:28:69:92', 'router_id': 'ap.TestNE'}),
                '10.0.196.146': NetworkMetadataReference(type='snmp_session', value='10.0.196.146', attributes={'cpe_mac': 'c0:c5:22:f9:75:a4', 'dev_mac': 'c0:c5:22:f9:75:a4', 'net_mac': 'c0:c5:22:f9:75:a4', 'router_id': 'ap.TestNE'}),
                '10.0.196.147': NetworkMetadataReference(type='snmp_session', value='10.0.196.147', attributes={'cpe_mac': '5c:8f:e0:1b:14:a2', 'dev_mac': '5c:8f:e0:1b:14:a2', 'net_mac': '5c:8f:e0:1b:14:a2', 'router_id': 'ap.TestNE'}),
                '10.0.196.148': NetworkMetadataReference(type='snmp_session', value='10.0.196.148', attributes={'cpe_mac': '5c:8f:e0:1a:e7:82', 'dev_mac': '5c:8f:e0:1a:e7:82', 'net_mac': '5c:8f:e0:1a:e7:82', 'router_id': 'ap.TestNE'}),
                '10.0.196.149': NetworkMetadataReference(type='snmp_session', value='10.0.196.149', attributes={'cpe_mac': '5c:8f:e0:28:74:9a', 'dev_mac': '5c:8f:e0:28:74:9a', 'net_mac': '5c:8f:e0:28:74:9a', 'router_id': 'ap.TestNE'}),
                '10.0.196.15': NetworkMetadataReference(type='snmp_session', value='10.0.196.15', attributes={'cpe_mac': '40:70:09:91:e4:c2', 'dev_mac': '40:70:09:91:e4:c2', 'net_mac': '40:70:09:91:e4:c2', 'router_id': 'ap.TestNE'}),
                '10.0.196.150': NetworkMetadataReference(type='snmp_session', value='10.0.196.150', attributes={'cpe_mac': '9c:34:26:b0:f7:d2', 'dev_mac': '9c:34:26:b0:f7:d2', 'net_mac': '9c:34:26:b0:f7:d2', 'router_id': 'ap.TestNE'}),
                '10.0.196.152': NetworkMetadataReference(type='snmp_session', value='10.0.196.152', attributes={'cpe_mac': '3c:7a:8a:7a:f6:e2', 'dev_mac': '3c:7a:8a:7a:f6:e2', 'net_mac': '3c:7a:8a:7a:f6:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.153': NetworkMetadataReference(type='snmp_session', value='10.0.196.153', attributes={'cpe_mac': 'c0:c5:22:e1:67:3a', 'dev_mac': 'c0:c5:22:e1:67:3a', 'net_mac': 'c0:c5:22:e1:67:3a', 'router_id': 'ap.TestNE'}),
                '10.0.196.156': NetworkMetadataReference(type='snmp_session', value='10.0.196.156', attributes={'cpe_mac': '3c:7a:8a:7e:c0:32', 'dev_mac': '3c:7a:8a:7e:c0:32', 'net_mac': '3c:7a:8a:7e:c0:32', 'router_id': 'ap.TestNE'}),
                '10.0.196.157': NetworkMetadataReference(type='snmp_session', value='10.0.196.157', attributes={'cpe_mac': '2c:99:24:d6:60:c9', 'dev_mac': '2c:99:24:d6:60:c9', 'net_mac': '2c:99:24:d6:60:c9', 'router_id': 'ap.TestNE'}),
                '10.0.196.158': NetworkMetadataReference(type='snmp_session', value='10.0.196.158', attributes={'cpe_mac': '54:65:de:6f:c4:d2', 'dev_mac': '54:65:de:6f:c4:d2', 'net_mac': '54:65:de:6f:c4:d2', 'router_id': 'ap.TestNE'}),
                '10.0.196.159': NetworkMetadataReference(type='snmp_session', value='10.0.196.159', attributes={'cpe_mac': 'ac:b3:13:0e:bb:02', 'dev_mac': 'ac:b3:13:0e:bb:02', 'net_mac': 'ac:b3:13:0e:bb:02', 'router_id': 'ap.TestNE'}),
                '10.0.196.16': NetworkMetadataReference(type='snmp_session', value='10.0.196.16', attributes={'cpe_mac': '98:6b:3d:dc:a2:82', 'dev_mac': '98:6b:3d:dc:a2:82', 'net_mac': '98:6b:3d:dc:a2:82', 'router_id': 'ap.TestNE'}),
                '10.0.196.160': NetworkMetadataReference(type='snmp_session', value='10.0.196.160', attributes={'cpe_mac': '38:70:0c:92:91:7f', 'dev_mac': '38:70:0c:92:91:7f', 'net_mac': '38:70:0c:92:91:7f', 'router_id': 'ap.TestNE'}),
                '10.0.196.162': NetworkMetadataReference(type='snmp_session', value='10.0.196.162', attributes={'cpe_mac': '9c:34:26:b0:be:6a', 'dev_mac': '9c:34:26:b0:be:6a', 'net_mac': '9c:34:26:b0:be:6a', 'router_id': 'ap.TestNE'}),
                '10.0.196.163': NetworkMetadataReference(type='snmp_session', value='10.0.196.163', attributes={'cpe_mac': '38:70:0c:96:38:30', 'dev_mac': '38:70:0c:96:38:30', 'net_mac': '38:70:0c:96:38:30', 'router_id': 'ap.TestNE'}),
                '10.0.196.164': NetworkMetadataReference(type='snmp_session', value='10.0.196.164', attributes={'cpe_mac': '38:70:0c:6d:47:d2', 'dev_mac': '38:70:0c:6d:47:d2', 'net_mac': '38:70:0c:6d:47:d2', 'router_id': 'ap.TestNE'}),
                '10.0.196.165': NetworkMetadataReference(type='snmp_session', value='10.0.196.165', attributes={'cpe_mac': '38:70:0c:93:98:3e', 'dev_mac': '38:70:0c:93:98:3e', 'net_mac': '38:70:0c:93:98:3e', 'router_id': 'ap.TestNE'}),
                '10.0.196.17': NetworkMetadataReference(type='snmp_session', value='10.0.196.17', attributes={'cpe_mac': '40:70:09:aa:b2:e2', 'dev_mac': '40:70:09:aa:b2:e2', 'net_mac': '40:70:09:aa:b2:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.19': NetworkMetadataReference(type='snmp_session', value='10.0.196.19', attributes={'cpe_mac': '40:70:09:94:7a:02', 'dev_mac': '40:70:09:94:7a:02', 'net_mac': '40:70:09:94:7a:02', 'router_id': 'ap.TestNE'}),
                '10.0.196.2': NetworkMetadataReference(type='snmp_session', value='10.0.196.2', attributes={'cpe_mac': '14:cf:e2:a0:15:c2', 'dev_mac': '14:cf:e2:a0:15:c2', 'net_mac': '14:cf:e2:a0:15:c2', 'router_id': 'ap.TestNE'}),
                '10.0.196.20': NetworkMetadataReference(type='snmp_session', value='10.0.196.20', attributes={'cpe_mac': 'cc:a4:62:37:0c:12', 'dev_mac': 'cc:a4:62:37:0c:12', 'net_mac': 'cc:a4:62:37:0c:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.21': NetworkMetadataReference(type='snmp_session', value='10.0.196.21', attributes={'cpe_mac': '90:1a:ca:7c:2b:62', 'dev_mac': '90:1a:ca:7c:2b:62', 'net_mac': '90:1a:ca:7c:2b:62', 'router_id': 'ap.TestNE'}),
                '10.0.196.22': NetworkMetadataReference(type='snmp_session', value='10.0.196.22', attributes={'cpe_mac': '10:86:8c:16:1e:12', 'dev_mac': '10:86:8c:16:1e:12', 'net_mac': '10:86:8c:16:1e:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.23': NetworkMetadataReference(type='snmp_session', value='10.0.196.23', attributes={'cpe_mac': 'e8:89:2c:7e:83:e2', 'dev_mac': 'e8:89:2c:7e:83:e2', 'net_mac': 'e8:89:2c:7e:83:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.24': NetworkMetadataReference(type='snmp_session', value='10.0.196.24', attributes={'cpe_mac': '98:6b:3d:c7:56:62', 'dev_mac': '98:6b:3d:c7:56:62', 'net_mac': '98:6b:3d:c7:56:62', 'router_id': 'ap.TestNE'}),
                '10.0.196.25': NetworkMetadataReference(type='snmp_session', value='10.0.196.25', attributes={'cpe_mac': '14:ab:f0:d3:47:d2', 'dev_mac': '14:ab:f0:d3:47:d2', 'net_mac': '14:ab:f0:d3:47:d2', 'router_id': 'ap.TestNE'}),
                '10.0.196.26': NetworkMetadataReference(type='snmp_session', value='10.0.196.26', attributes={'cpe_mac': '94:87:7c:10:62:22', 'dev_mac': '94:87:7c:10:62:22', 'net_mac': '94:87:7c:10:62:22', 'router_id': 'ap.TestNE'}),
                '10.0.196.27': NetworkMetadataReference(type='snmp_session', value='10.0.196.27', attributes={'cpe_mac': 'ac:b3:13:04:d1:12', 'dev_mac': 'ac:b3:13:04:d1:12', 'net_mac': 'ac:b3:13:04:d1:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.28': NetworkMetadataReference(type='snmp_session', value='10.0.196.28', attributes={'cpe_mac': '3c:7a:8a:ef:02:0a', 'dev_mac': '3c:7a:8a:ef:02:0a', 'net_mac': '3c:7a:8a:ef:02:0a', 'router_id': 'ap.TestNE'}),
                '10.0.196.29': NetworkMetadataReference(type='snmp_session', value='10.0.196.29', attributes={'cpe_mac': '38:70:0c:68:4f:ca', 'dev_mac': '38:70:0c:68:4f:ca', 'net_mac': '38:70:0c:68:4f:ca', 'router_id': 'ap.TestNE'}),
                '10.0.196.3': NetworkMetadataReference(type='snmp_session', value='10.0.196.3', attributes={'cpe_mac': '38:70:0c:8f:ac:90', 'dev_mac': '38:70:0c:8f:ac:90', 'net_mac': '38:70:0c:8f:ac:90', 'router_id': 'ap.TestNE'}),
                '10.0.196.30': NetworkMetadataReference(type='snmp_session', value='10.0.196.30', attributes={'cpe_mac': '90:1a:ca:5e:14:c2', 'dev_mac': '90:1a:ca:5e:14:c2', 'net_mac': '90:1a:ca:5e:14:c2', 'router_id': 'ap.TestNE'}),
                '10.0.196.31': NetworkMetadataReference(type='snmp_session', value='10.0.196.31', attributes={'cpe_mac': '54:65:de:6f:6a:52', 'dev_mac': '54:65:de:6f:6a:52', 'net_mac': '54:65:de:6f:6a:52', 'router_id': 'ap.TestNE'}),
                '10.0.196.33': NetworkMetadataReference(type='snmp_session', value='10.0.196.33', attributes={'cpe_mac': 'ac:b3:13:a1:a9:e2', 'dev_mac': 'ac:b3:13:a1:a9:e2', 'net_mac': 'ac:b3:13:a1:a9:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.34': NetworkMetadataReference(type='snmp_session', value='10.0.196.34', attributes={'cpe_mac': 'ac:b3:13:78:36:72', 'dev_mac': 'ac:b3:13:78:36:72', 'net_mac': 'ac:b3:13:78:36:72', 'router_id': 'ap.TestNE'}),
                '10.0.196.35': NetworkMetadataReference(type='snmp_session', value='10.0.196.35', attributes={'cpe_mac': '90:1a:ca:71:7f:52', 'dev_mac': '90:1a:ca:71:7f:52', 'net_mac': '90:1a:ca:71:7f:52', 'router_id': 'ap.TestNE'}),
                '10.0.196.36': NetworkMetadataReference(type='snmp_session', value='10.0.196.36', attributes={'cpe_mac': '38:70:0c:6a:ea:4b', 'dev_mac': '38:70:0c:6a:ea:4b', 'net_mac': '38:70:0c:6a:ea:4b', 'router_id': 'ap.TestNE'}),
                '10.0.196.37': NetworkMetadataReference(type='snmp_session', value='10.0.196.37', attributes={'cpe_mac': '5c:8f:e0:1b:06:7a', 'dev_mac': '5c:8f:e0:1b:06:7a', 'net_mac': '5c:8f:e0:1b:06:7a', 'router_id': 'ap.TestNE'}),
                '10.0.196.38': NetworkMetadataReference(type='snmp_session', value='10.0.196.38', attributes={'cpe_mac': 'f8:ed:a5:dc:5e:e2', 'dev_mac': 'f8:ed:a5:dc:5e:e2', 'net_mac': 'f8:ed:a5:dc:5e:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.4': NetworkMetadataReference(type='snmp_session', value='10.0.196.4', attributes={'cpe_mac': 'f8:ed:a5:f2:aa:b2', 'dev_mac': 'f8:ed:a5:f2:aa:b2', 'net_mac': 'f8:ed:a5:f2:aa:b2', 'router_id': 'ap.TestNE'}),
                '10.0.196.40': NetworkMetadataReference(type='snmp_session', value='10.0.196.40', attributes={'cpe_mac': '24:94:cb:28:6f:1d', 'dev_mac': '24:94:cb:28:6f:1d', 'net_mac': '24:94:cb:28:6f:1d', 'router_id': 'ap.TestNE'}),
                '10.0.196.42': NetworkMetadataReference(type='snmp_session', value='10.0.196.42', attributes={'cpe_mac': '00:ac:e0:bf:40:92', 'dev_mac': '00:ac:e0:bf:40:92', 'net_mac': '00:ac:e0:bf:40:92', 'router_id': 'ap.TestNE'}),
                '10.0.196.43': NetworkMetadataReference(type='snmp_session', value='10.0.196.43', attributes={'cpe_mac': 'ac:b3:13:a2:05:12', 'dev_mac': 'ac:b3:13:a2:05:12', 'net_mac': 'ac:b3:13:a2:05:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.44': NetworkMetadataReference(type='snmp_session', value='10.0.196.44', attributes={'cpe_mac': '08:3e:0c:e8:51:b2', 'dev_mac': '08:3e:0c:e8:51:b2', 'net_mac': '08:3e:0c:e8:51:b2', 'router_id': 'ap.TestNE'}),
                '10.0.196.45': NetworkMetadataReference(type='snmp_session', value='10.0.196.45', attributes={'cpe_mac': '94:87:7c:06:b4:02', 'dev_mac': '94:87:7c:06:b4:02', 'net_mac': '94:87:7c:06:b4:02', 'router_id': 'ap.TestNE'}),
                '10.0.196.46': NetworkMetadataReference(type='snmp_session', value='10.0.196.46', attributes={'cpe_mac': '90:1a:ca:7b:69:92', 'dev_mac': '90:1a:ca:7b:69:92', 'net_mac': '90:1a:ca:7b:69:92', 'router_id': 'ap.TestNE'}),
                '10.0.196.47': NetworkMetadataReference(type='snmp_session', value='10.0.196.47', attributes={'cpe_mac': '08:3e:0c:10:67:72', 'dev_mac': '08:3e:0c:10:67:72', 'net_mac': '08:3e:0c:10:67:72', 'router_id': 'ap.TestNE'}),
                '10.0.196.48': NetworkMetadataReference(type='snmp_session', value='10.0.196.48', attributes={'cpe_mac': '5c:8f:e0:1b:a7:b2', 'dev_mac': '5c:8f:e0:1b:a7:b2', 'net_mac': '5c:8f:e0:1b:a7:b2', 'router_id': 'ap.TestNE'}),
                '10.0.196.49': NetworkMetadataReference(type='snmp_session', value='10.0.196.49', attributes={'cpe_mac': '8c:09:f4:04:06:62', 'dev_mac': '8c:09:f4:04:06:62', 'net_mac': '8c:09:f4:04:06:62', 'router_id': 'ap.TestNE'}),
                '10.0.196.5': NetworkMetadataReference(type='snmp_session', value='10.0.196.5', attributes={'cpe_mac': '94:87:7c:01:e3:a2', 'dev_mac': '94:87:7c:01:e3:a2', 'net_mac': '94:87:7c:01:e3:a2', 'router_id': 'ap.TestNE'}),
                '10.0.196.50': NetworkMetadataReference(type='snmp_session', value='10.0.196.50', attributes={'cpe_mac': '9c:34:26:ac:47:5a', 'dev_mac': '9c:34:26:ac:47:5a', 'net_mac': '9c:34:26:ac:47:5a', 'router_id': 'ap.TestNE'}),
                '10.0.196.51': NetworkMetadataReference(type='snmp_session', value='10.0.196.51', attributes={'cpe_mac': '40:70:09:71:a2:32', 'dev_mac': '40:70:09:71:a2:32', 'net_mac': '40:70:09:71:a2:32', 'router_id': 'ap.TestNE'}),
                '10.0.196.52': NetworkMetadataReference(type='snmp_session', value='10.0.196.52', attributes={'cpe_mac': '40:70:09:8c:07:a2', 'dev_mac': '40:70:09:8c:07:a2', 'net_mac': '40:70:09:8c:07:a2', 'router_id': 'ap.TestNE'}),
                '10.0.196.53': NetworkMetadataReference(type='snmp_session', value='10.0.196.53', attributes={'cpe_mac': '40:70:09:86:0b:62', 'dev_mac': '40:70:09:86:0b:62', 'net_mac': '40:70:09:86:0b:62', 'router_id': 'ap.TestNE'}),
                '10.0.196.54': NetworkMetadataReference(type='snmp_session', value='10.0.196.54', attributes={'cpe_mac': '54:65:de:fe:9d:22', 'dev_mac': '54:65:de:fe:9d:22', 'net_mac': '54:65:de:fe:9d:22', 'router_id': 'ap.TestNE'}),
                '10.0.196.55': NetworkMetadataReference(type='snmp_session', value='10.0.196.55', attributes={'cpe_mac': '8c:09:f4:10:7b:82', 'dev_mac': '8c:09:f4:10:7b:82', 'net_mac': '8c:09:f4:10:7b:82', 'router_id': 'ap.TestNE'}),
                '10.0.196.56': NetworkMetadataReference(type='snmp_session', value='10.0.196.56', attributes={'cpe_mac': 'ac:b3:13:b2:b4:52', 'dev_mac': 'ac:b3:13:b2:b4:52', 'net_mac': 'ac:b3:13:b2:b4:52', 'router_id': 'ap.TestNE'}),
                '10.0.196.58': NetworkMetadataReference(type='snmp_session', value='10.0.196.58', attributes={'cpe_mac': '0c:f8:93:de:7d:e2', 'dev_mac': '0c:f8:93:de:7d:e2', 'net_mac': '0c:f8:93:de:7d:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.6': NetworkMetadataReference(type='snmp_session', value='10.0.196.6', attributes={'cpe_mac': '94:87:7c:0a:91:22', 'dev_mac': '94:87:7c:0a:91:22', 'net_mac': '94:87:7c:0a:91:22', 'router_id': 'ap.TestNE'}),
                '10.0.196.60': NetworkMetadataReference(type='snmp_session', value='10.0.196.60', attributes={'cpe_mac': '98:6b:3d:d2:0d:b2', 'dev_mac': '98:6b:3d:d2:0d:b2', 'net_mac': '98:6b:3d:d2:0d:b2', 'router_id': 'ap.TestNE'}),
                '10.0.196.61': NetworkMetadataReference(type='snmp_session', value='10.0.196.61', attributes={'cpe_mac': '98:6b:3d:c3:17:e2', 'dev_mac': '98:6b:3d:c3:17:e2', 'net_mac': '98:6b:3d:c3:17:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.62': NetworkMetadataReference(type='snmp_session', value='10.0.196.62', attributes={'cpe_mac': 'ac:b3:13:9f:dc:c2', 'dev_mac': 'ac:b3:13:9f:dc:c2', 'net_mac': 'ac:b3:13:9f:dc:c2', 'router_id': 'ap.TestNE'}),
                '10.0.196.63': NetworkMetadataReference(type='snmp_session', value='10.0.196.63', attributes={'cpe_mac': 'f8:ed:a5:df:27:12', 'dev_mac': 'f8:ed:a5:df:27:12', 'net_mac': 'f8:ed:a5:df:27:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.66': NetworkMetadataReference(type='snmp_session', value='10.0.196.66', attributes={'cpe_mac': '40:70:09:8a:27:92', 'dev_mac': '40:70:09:8a:27:92', 'net_mac': '40:70:09:8a:27:92', 'router_id': 'ap.TestNE'}),
                '10.0.196.67': NetworkMetadataReference(type='snmp_session', value='10.0.196.67', attributes={'cpe_mac': '98:6b:3d:dc:60:42', 'dev_mac': '98:6b:3d:dc:60:42', 'net_mac': '98:6b:3d:dc:60:42', 'router_id': 'ap.TestNE'}),
                '10.0.196.69': NetworkMetadataReference(type='snmp_session', value='10.0.196.69', attributes={'cpe_mac': 'cc:a4:62:26:71:92', 'dev_mac': 'cc:a4:62:26:71:92', 'net_mac': 'cc:a4:62:26:71:92', 'router_id': 'ap.TestNE'}),
                '10.0.196.7': NetworkMetadataReference(type='snmp_session', value='10.0.196.7', attributes={'cpe_mac': '38:70:0c:3e:cc:3a', 'dev_mac': '38:70:0c:3e:cc:3a', 'net_mac': '38:70:0c:3e:cc:3a', 'router_id': 'ap.TestNE'}),
                '10.0.196.71': NetworkMetadataReference(type='snmp_session', value='10.0.196.71', attributes={'cpe_mac': '00:ac:e0:b8:07:92', 'dev_mac': '00:ac:e0:b8:07:92', 'net_mac': '00:ac:e0:b8:07:92', 'router_id': 'ap.TestNE'}),
                '10.0.196.73': NetworkMetadataReference(type='snmp_session', value='10.0.196.73', attributes={'cpe_mac': 'f8:ed:a5:db:b3:f2', 'dev_mac': 'f8:ed:a5:db:b3:f2', 'net_mac': 'f8:ed:a5:db:b3:f2', 'router_id': 'ap.TestNE'}),
                '10.0.196.75': NetworkMetadataReference(type='snmp_session', value='10.0.196.75', attributes={'cpe_mac': '3c:7a:8a:7d:b7:aa', 'dev_mac': '3c:7a:8a:7d:b7:aa', 'net_mac': '3c:7a:8a:7d:b7:aa', 'router_id': 'ap.TestNE'}),
                '10.0.196.76': NetworkMetadataReference(type='snmp_session', value='10.0.196.76', attributes={'cpe_mac': '90:1a:ca:5e:3a:82', 'dev_mac': '90:1a:ca:5e:3a:82', 'net_mac': '90:1a:ca:5e:3a:82', 'router_id': 'ap.TestNE'}),
                '10.0.196.77': NetworkMetadataReference(type='snmp_session', value='10.0.196.77', attributes={'cpe_mac': 'ac:b3:13:a9:70:02', 'dev_mac': 'ac:b3:13:a9:70:02', 'net_mac': 'ac:b3:13:a9:70:02', 'router_id': 'ap.TestNE'}),
                '10.0.196.78': NetworkMetadataReference(type='snmp_session', value='10.0.196.78', attributes={'cpe_mac': '98:6b:3d:e5:51:e2', 'dev_mac': '98:6b:3d:e5:51:e2', 'net_mac': '98:6b:3d:e5:51:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.79': NetworkMetadataReference(type='snmp_session', value='10.0.196.79', attributes={'cpe_mac': '3c:7a:8a:76:94:4a', 'dev_mac': '3c:7a:8a:76:94:4a', 'net_mac': '3c:7a:8a:76:94:4a', 'router_id': 'ap.TestNE'}),
                '10.0.196.8': NetworkMetadataReference(type='snmp_session', value='10.0.196.8', attributes={'cpe_mac': 'f8:ed:a5:e8:1a:d2', 'dev_mac': 'f8:ed:a5:e8:1a:d2', 'net_mac': 'f8:ed:a5:e8:1a:d2', 'router_id': 'ap.TestNE'}),
                '10.0.196.80': NetworkMetadataReference(type='snmp_session', value='10.0.196.80', attributes={'cpe_mac': '38:70:0c:8f:c2:70', 'dev_mac': '38:70:0c:8f:c2:70', 'net_mac': '38:70:0c:8f:c2:70', 'router_id': 'ap.TestNE'}),
                '10.0.196.81': NetworkMetadataReference(type='snmp_session', value='10.0.196.81', attributes={'cpe_mac': 'ac:b3:13:77:26:e2', 'dev_mac': 'ac:b3:13:77:26:e2', 'net_mac': 'ac:b3:13:77:26:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.82': NetworkMetadataReference(type='snmp_session', value='10.0.196.82', attributes={'cpe_mac': '40:70:09:71:53:5c', 'dev_mac': '40:70:09:71:53:5c', 'net_mac': '40:70:09:71:53:5c', 'router_id': 'ap.TestNE'}),
                '10.0.196.84': NetworkMetadataReference(type='snmp_session', value='10.0.196.84', attributes={'cpe_mac': '2c:99:24:d8:34:79', 'dev_mac': '2c:99:24:d8:34:79', 'net_mac': '2c:99:24:d8:34:79', 'router_id': 'ap.TestNE'}),
                '10.0.196.85': NetworkMetadataReference(type='snmp_session', value='10.0.196.85', attributes={'cpe_mac': '98:6b:3d:c5:50:42', 'dev_mac': '98:6b:3d:c5:50:42', 'net_mac': '98:6b:3d:c5:50:42', 'router_id': 'ap.TestNE'}),
                '10.0.196.86': NetworkMetadataReference(type='snmp_session', value='10.0.196.86', attributes={'cpe_mac': '14:cf:e2:a9:1f:12', 'dev_mac': '14:cf:e2:a9:1f:12', 'net_mac': '14:cf:e2:a9:1f:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.87': NetworkMetadataReference(type='snmp_session', value='10.0.196.87', attributes={'cpe_mac': 'ac:b3:13:a4:98:02', 'dev_mac': 'ac:b3:13:a4:98:02', 'net_mac': 'ac:b3:13:a4:98:02', 'router_id': 'ap.TestNE'}),
                '10.0.196.88': NetworkMetadataReference(type='snmp_session', value='10.0.196.88', attributes={'cpe_mac': '90:1a:ca:61:ca:52', 'dev_mac': '90:1a:ca:61:ca:52', 'net_mac': '90:1a:ca:61:ca:52', 'router_id': 'ap.TestNE'}),
                '10.0.196.89': NetworkMetadataReference(type='snmp_session', value='10.0.196.89', attributes={'cpe_mac': '08:3e:0c:ec:39:b2', 'dev_mac': '08:3e:0c:ec:39:b2', 'net_mac': '08:3e:0c:ec:39:b2', 'router_id': 'ap.TestNE'}),
                '10.0.196.9': NetworkMetadataReference(type='snmp_session', value='10.0.196.9', attributes={'cpe_mac': '5c:8f:e0:4f:ac:1a', 'dev_mac': '5c:8f:e0:4f:ac:1a', 'net_mac': '5c:8f:e0:4f:ac:1a', 'router_id': 'ap.TestNE'}),
                '10.0.196.90': NetworkMetadataReference(type='snmp_session', value='10.0.196.90', attributes={'cpe_mac': '54:65:de:bd:4e:62', 'dev_mac': '54:65:de:bd:4e:62', 'net_mac': '54:65:de:bd:4e:62', 'router_id': 'ap.TestNE'}),
                '10.0.196.91': NetworkMetadataReference(type='snmp_session', value='10.0.196.91', attributes={'cpe_mac': '90:1a:ca:db:eb:f2', 'dev_mac': '90:1a:ca:db:eb:f2', 'net_mac': '90:1a:ca:db:eb:f2', 'router_id': 'ap.TestNE'}),
                '10.0.196.92': NetworkMetadataReference(type='snmp_session', value='10.0.196.92', attributes={'cpe_mac': '3c:7a:8a:78:0d:3a', 'dev_mac': '3c:7a:8a:78:0d:3a', 'net_mac': '3c:7a:8a:78:0d:3a', 'router_id': 'ap.TestNE'}),
                '10.0.196.94': NetworkMetadataReference(type='snmp_session', value='10.0.196.94', attributes={'cpe_mac': '78:71:9c:8d:0f:82', 'dev_mac': '78:71:9c:8d:0f:82', 'net_mac': '78:71:9c:8d:0f:82', 'router_id': 'ap.TestNE'}),
                '10.0.196.95': NetworkMetadataReference(type='snmp_session', value='10.0.196.95', attributes={'cpe_mac': '14:cf:e2:9c:ce:e2', 'dev_mac': '14:cf:e2:9c:ce:e2', 'net_mac': '14:cf:e2:9c:ce:e2', 'router_id': 'ap.TestNE'}),
                '10.0.196.96': NetworkMetadataReference(type='snmp_session', value='10.0.196.96', attributes={'cpe_mac': '2c:99:24:8e:5f:03', 'dev_mac': '2c:99:24:8e:5f:03', 'net_mac': '2c:99:24:8e:5f:03', 'router_id': 'ap.TestNE'}),
                '10.0.196.97': NetworkMetadataReference(type='snmp_session', value='10.0.196.97', attributes={'cpe_mac': '78:71:9c:8b:f9:b2', 'dev_mac': '78:71:9c:8b:f9:b2', 'net_mac': '78:71:9c:8b:f9:b2', 'router_id': 'ap.TestNE'}),
                '10.0.196.98': NetworkMetadataReference(type='snmp_session', value='10.0.196.98', attributes={'cpe_mac': '14:cf:e2:a0:26:12', 'dev_mac': '14:cf:e2:a0:26:12', 'net_mac': '14:cf:e2:a0:26:12', 'router_id': 'ap.TestNE'}),
                '10.0.196.99': NetworkMetadataReference(type='snmp_session', value='10.0.196.99', attributes={'cpe_mac': '98:6b:3d:c6:d7:92', 'dev_mac': '98:6b:3d:c6:d7:92', 'net_mac': '98:6b:3d:c6:d7:92', 'router_id': 'ap.TestNE'}),
                '10.1.1.249': NetworkMetadataReference(type='snmp_session', value='10.1.1.249', attributes={'cpe_mac': 'e4:8d:8c:2a:98:1f', 'dev_mac': 'e4:8d:8c:2a:98:1f', 'net_mac': 'e4:8d:8c:2a:98:1f', 'router_id': 'ap.TestNE'}),
                '10.255.255.10': NetworkMetadataReference(type='snmp_session', value='10.255.255.10', attributes={'cpe_mac': '3c:7a:8a:7a:f6:e3', 'dev_mac': '3c:7a:8a:7a:f6:e3', 'net_mac': '3c:7a:8a:7a:f6:e3', 'router_id': 'ap.TestNE'}),
                '10.255.255.130': NetworkMetadataReference(type='snmp_session', value='10.255.255.130', attributes={'cpe_mac': '38:70:0c:02:e8:b9', 'dev_mac': '38:70:0c:02:e8:b9', 'net_mac': '38:70:0c:02:e8:b9', 'router_id': 'ap.TestNE'}),
                '10.255.255.132': NetworkMetadataReference(type='snmp_session', value='10.255.255.132', attributes={'cpe_mac': '40:70:09:8c:07:a4', 'dev_mac': '40:70:09:8c:07:a4', 'net_mac': '40:70:09:8c:07:a4', 'router_id': 'ap.TestNE'}),
                '10.255.255.134': NetworkMetadataReference(type='snmp_session', value='10.255.255.134', attributes={'cpe_mac': 'ac:b3:13:a1:a9:e4', 'dev_mac': 'ac:b3:13:a1:a9:e4', 'net_mac': 'ac:b3:13:a1:a9:e4', 'router_id': 'ap.TestNE'}),
                '10.255.255.136': NetworkMetadataReference(type='snmp_session', value='10.255.255.136', attributes={'cpe_mac': 'ac:b3:13:78:36:74', 'dev_mac': 'ac:b3:13:78:36:74', 'net_mac': 'ac:b3:13:78:36:74', 'router_id': 'ap.TestNE'}),
                '10.255.255.137': NetworkMetadataReference(type='snmp_session', value='10.255.255.137', attributes={'cpe_mac': '40:70:09:8a:27:94', 'dev_mac': '40:70:09:8a:27:94', 'net_mac': '40:70:09:8a:27:94', 'router_id': 'ap.TestNE'}),
                '10.255.255.14': NetworkMetadataReference(type='snmp_session', value='10.255.255.14', attributes={'cpe_mac': '54:65:de:bd:4e:64', 'dev_mac': '54:65:de:bd:4e:64', 'net_mac': '54:65:de:bd:4e:64', 'router_id': 'ap.TestNE'}),
                '10.255.255.15': NetworkMetadataReference(type='snmp_session', value='10.255.255.15', attributes={'cpe_mac': 'f8:ed:a5:db:b3:f4', 'dev_mac': 'f8:ed:a5:db:b3:f4', 'net_mac': 'f8:ed:a5:db:b3:f4', 'router_id': 'ap.TestNE'}),
                '10.255.255.150': NetworkMetadataReference(type='snmp_session', value='10.255.255.150', attributes={'cpe_mac': 'ac:b3:13:0e:bb:04', 'dev_mac': 'ac:b3:13:0e:bb:04', 'net_mac': 'ac:b3:13:0e:bb:04', 'router_id': 'ap.TestNE'}),
                '10.255.255.151': NetworkMetadataReference(type='snmp_session', value='10.255.255.151', attributes={'cpe_mac': 'ac:b3:13:ac:c4:c4', 'dev_mac': 'ac:b3:13:ac:c4:c4', 'net_mac': 'ac:b3:13:ac:c4:c4', 'router_id': 'ap.TestNE'}),
                '10.255.255.152': NetworkMetadataReference(type='snmp_session', value='10.255.255.152', attributes={'cpe_mac': '5c:8f:e0:1b:06:7b', 'dev_mac': '5c:8f:e0:1b:06:7b', 'net_mac': '5c:8f:e0:1b:06:7b', 'router_id': 'ap.TestNE'}),
                '10.255.255.154': NetworkMetadataReference(type='snmp_session', value='10.255.255.154', attributes={'cpe_mac': '78:71:9c:8b:f9:b4', 'dev_mac': '78:71:9c:8b:f9:b4', 'net_mac': '78:71:9c:8b:f9:b4', 'router_id': 'ap.TestNE'}),
                '10.255.255.155': NetworkMetadataReference(type='snmp_session', value='10.255.255.155', attributes={'cpe_mac': '3c:7a:8a:78:0d:3b', 'dev_mac': '3c:7a:8a:78:0d:3b', 'net_mac': '3c:7a:8a:78:0d:3b', 'router_id': 'ap.TestNE'}),
                '10.255.255.157': NetworkMetadataReference(type='snmp_session', value='10.255.255.157', attributes={'cpe_mac': '8c:09:f4:04:06:64', 'dev_mac': '8c:09:f4:04:06:64', 'net_mac': '8c:09:f4:04:06:64', 'router_id': 'ap.TestNE'}),
                '10.255.255.158': NetworkMetadataReference(type='snmp_session', value='10.255.255.158', attributes={'cpe_mac': '40:70:09:71:a2:34', 'dev_mac': '40:70:09:71:a2:34', 'net_mac': '40:70:09:71:a2:34', 'router_id': 'ap.TestNE'}),
                '10.255.255.159': NetworkMetadataReference(type='snmp_session', value='10.255.255.159', attributes={'cpe_mac': 'ac:b3:13:85:c2:74', 'dev_mac': 'ac:b3:13:85:c2:74', 'net_mac': 'ac:b3:13:85:c2:74', 'router_id': 'ap.TestNE'}),
                '10.255.255.160': NetworkMetadataReference(type='snmp_session', value='10.255.255.160', attributes={'cpe_mac': '3c:7a:8a:7e:91:fb', 'dev_mac': '3c:7a:8a:7e:91:fb', 'net_mac': '3c:7a:8a:7e:91:fb', 'router_id': 'ap.TestNE'}),
                '10.255.255.161': NetworkMetadataReference(type='snmp_session', value='10.255.255.161', attributes={'cpe_mac': '38:70:0c:3e:cc:3b', 'dev_mac': '38:70:0c:3e:cc:3b', 'net_mac': '38:70:0c:3e:cc:3b', 'router_id': 'ap.TestNE'}),
                '10.255.255.163': NetworkMetadataReference(type='snmp_session', value='10.255.255.163', attributes={'cpe_mac': '3c:7a:8a:76:94:4b', 'dev_mac': '3c:7a:8a:76:94:4b', 'net_mac': '3c:7a:8a:76:94:4b', 'router_id': 'ap.TestNE'}),
                '10.255.255.20': NetworkMetadataReference(type='snmp_session', value='10.255.255.20', attributes={'cpe_mac': 'c0:c5:22:f9:75:a5', 'dev_mac': 'c0:c5:22:f9:75:a5', 'net_mac': 'c0:c5:22:f9:75:a5', 'router_id': 'ap.TestNE'}),
                '10.255.255.22': NetworkMetadataReference(type='snmp_session', value='10.255.255.22', attributes={'cpe_mac': '98:6b:3d:c7:56:64', 'dev_mac': '98:6b:3d:c7:56:64', 'net_mac': '98:6b:3d:c7:56:64', 'router_id': 'ap.TestNE'}),
                '10.255.255.23': NetworkMetadataReference(type='snmp_session', value='10.255.255.23', attributes={'cpe_mac': '38:70:0c:8f:85:7e', 'dev_mac': '38:70:0c:8f:85:7e', 'net_mac': '38:70:0c:8f:85:7e', 'router_id': 'ap.TestNE'}),
                '10.255.255.24': NetworkMetadataReference(type='snmp_session', value='10.255.255.24', attributes={'cpe_mac': 'c0:c1:c0:fb:af:a1', 'dev_mac': 'c0:c1:c0:fb:af:a1', 'net_mac': 'c0:c1:c0:fb:af:a1', 'router_id': 'ap.TestNE'}),
                '10.255.255.3': NetworkMetadataReference(type='snmp_session', value='10.255.255.3', attributes={'cpe_mac': '5c:8f:e0:27:ed:13', 'dev_mac': '5c:8f:e0:27:ed:13', 'net_mac': '5c:8f:e0:27:ed:13', 'router_id': 'ap.TestNE'}),
                '10.255.255.6': NetworkMetadataReference(type='snmp_session', value='10.255.255.6', attributes={'cpe_mac': '2c:99:24:52:71:e4', 'dev_mac': '2c:99:24:52:71:e4', 'net_mac': '2c:99:24:52:71:e4', 'router_id': 'ap.TestNE'}),
                '10.255.255.8': NetworkMetadataReference(type='snmp_session', value='10.255.255.8', attributes={'cpe_mac': '90:1a:ca:61:ca:54', 'dev_mac': '90:1a:ca:61:ca:54', 'net_mac': '90:1a:ca:61:ca:54', 'router_id': 'ap.TestNE'}),
                '107.181.195.10': NetworkMetadataReference(type='snmp_session', value='107.181.195.10', attributes={'cpe_mac': 'dc:ef:09:17:05:e9', 'dev_mac': 'dc:ef:09:17:05:e9', 'net_mac': 'dc:ef:09:17:05:e9', 'router_id': 'ap.TestNE'}),
                '107.181.195.100': NetworkMetadataReference(type='snmp_session', value='107.181.195.100', attributes={'cpe_mac': '3c:7a:8a:75:a7:43', 'dev_mac': '3c:7a:8a:75:a7:43', 'net_mac': '3c:7a:8a:75:a7:43', 'router_id': 'ap.TestNE'}),
                '107.181.195.101': NetworkMetadataReference(type='snmp_session', value='107.181.195.101', attributes={'cpe_mac': '14:cf:e2:a0:26:14', 'dev_mac': '14:cf:e2:a0:26:14', 'net_mac': '14:cf:e2:a0:26:14', 'router_id': 'ap.TestNE'}),
                '107.181.195.104': NetworkMetadataReference(type='snmp_session', value='107.181.195.104', attributes={'cpe_mac': 'f8:ed:a5:e8:1a:d4', 'dev_mac': 'f8:ed:a5:e8:1a:d4', 'net_mac': 'f8:ed:a5:e8:1a:d4', 'router_id': 'ap.TestNE'}),
                '107.181.195.105': NetworkMetadataReference(type='snmp_session', value='107.181.195.105', attributes={'cpe_mac': '40:70:09:91:e4:c4', 'dev_mac': '40:70:09:91:e4:c4', 'net_mac': '40:70:09:91:e4:c4', 'router_id': 'ap.TestNE'}),
                '107.181.195.106': NetworkMetadataReference(type='snmp_session', value='107.181.195.106', attributes={'cpe_mac': '38:70:0c:8f:c2:71', 'dev_mac': '38:70:0c:8f:c2:71', 'net_mac': '38:70:0c:8f:c2:71', 'router_id': 'ap.TestNE'}),
                '107.181.195.107': NetworkMetadataReference(type='snmp_session', value='107.181.195.107', attributes={'cpe_mac': '98:6b:3d:c3:17:e4', 'dev_mac': '98:6b:3d:c3:17:e4', 'net_mac': '98:6b:3d:c3:17:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.109': NetworkMetadataReference(type='snmp_session', value='107.181.195.109', attributes={'cpe_mac': 'ac:b3:13:a4:98:04', 'dev_mac': 'ac:b3:13:a4:98:04', 'net_mac': 'ac:b3:13:a4:98:04', 'router_id': 'ap.TestNE'}),
                '107.181.195.110': NetworkMetadataReference(type='snmp_session', value='107.181.195.110', attributes={'cpe_mac': '38:70:0c:8f:bc:a5', 'dev_mac': '38:70:0c:8f:bc:a5', 'net_mac': '38:70:0c:8f:bc:a5', 'router_id': 'ap.TestNE'}),
                '107.181.195.112': NetworkMetadataReference(type='snmp_session', value='107.181.195.112', attributes={'cpe_mac': '14:cf:e2:a9:1f:14', 'dev_mac': '14:cf:e2:a9:1f:14', 'net_mac': '14:cf:e2:a9:1f:14', 'router_id': 'ap.TestNE'}),
                '107.181.195.113': NetworkMetadataReference(type='snmp_session', value='107.181.195.113', attributes={'cpe_mac': '9c:34:26:ac:47:5b', 'dev_mac': '9c:34:26:ac:47:5b', 'net_mac': '9c:34:26:ac:47:5b', 'router_id': 'ap.TestNE'}),
                '107.181.195.115': NetworkMetadataReference(type='snmp_session', value='107.181.195.115', attributes={'cpe_mac': '44:a5:6e:45:d0:d5', 'dev_mac': '44:a5:6e:45:d0:d5', 'net_mac': '44:a5:6e:45:d0:d5', 'router_id': 'ap.TestNE'}),
                '107.181.195.116': NetworkMetadataReference(type='snmp_session', value='107.181.195.116', attributes={'cpe_mac': '8c:09:f4:10:7b:84', 'dev_mac': '8c:09:f4:10:7b:84', 'net_mac': '8c:09:f4:10:7b:84', 'router_id': 'ap.TestNE'}),
                '107.181.195.118': NetworkMetadataReference(type='snmp_session', value='107.181.195.118', attributes={'cpe_mac': 'ac:b3:13:04:d1:14', 'dev_mac': 'ac:b3:13:04:d1:14', 'net_mac': 'ac:b3:13:04:d1:14', 'router_id': 'ap.TestNE'}),
                '107.181.195.119': NetworkMetadataReference(type='snmp_session', value='107.181.195.119', attributes={'cpe_mac': '40:70:09:71:53:24', 'dev_mac': '40:70:09:71:53:24', 'net_mac': '40:70:09:71:53:24', 'router_id': 'ap.TestNE'}),
                '107.181.195.120': NetworkMetadataReference(type='snmp_session', value='107.181.195.120', attributes={'cpe_mac': '40:70:09:86:0b:64', 'dev_mac': '40:70:09:86:0b:64', 'net_mac': '40:70:09:86:0b:64', 'router_id': 'ap.TestNE'}),
                '107.181.195.121': NetworkMetadataReference(type='snmp_session', value='107.181.195.121', attributes={'cpe_mac': '08:3e:0c:10:67:74', 'dev_mac': '08:3e:0c:10:67:74', 'net_mac': '08:3e:0c:10:67:74', 'router_id': 'ap.TestNE'}),
                '107.181.195.122': NetworkMetadataReference(type='snmp_session', value='107.181.195.122', attributes={'cpe_mac': '60:19:71:ee:44:74', 'dev_mac': '60:19:71:ee:44:74', 'net_mac': '60:19:71:ee:44:74', 'router_id': 'ap.TestNE'}),
                '107.181.195.123': NetworkMetadataReference(type='snmp_session', value='107.181.195.123', attributes={'cpe_mac': '94:87:7c:06:b4:04', 'dev_mac': '94:87:7c:06:b4:04', 'net_mac': '94:87:7c:06:b4:04', 'router_id': 'ap.TestNE'}),
                '107.181.195.124': NetworkMetadataReference(type='snmp_session', value='107.181.195.124', attributes={'cpe_mac': 'b0:39:56:6b:c8:09', 'dev_mac': 'b0:39:56:6b:c8:09', 'net_mac': 'b0:39:56:6b:c8:09', 'router_id': 'ap.TestNE'}),
                '107.181.195.125': NetworkMetadataReference(type='snmp_session', value='107.181.195.125', attributes={'cpe_mac': '3c:7a:8a:7e:c0:33', 'dev_mac': '3c:7a:8a:7e:c0:33', 'net_mac': '3c:7a:8a:7e:c0:33', 'router_id': 'ap.TestNE'}),
                '107.181.195.127': NetworkMetadataReference(type='snmp_session', value='107.181.195.127', attributes={'cpe_mac': 'f8:ed:a5:df:27:14', 'dev_mac': 'f8:ed:a5:df:27:14', 'net_mac': 'f8:ed:a5:df:27:14', 'router_id': 'ap.TestNE'}),
                '107.181.195.128': NetworkMetadataReference(type='snmp_session', value='107.181.195.128', attributes={'cpe_mac': '14:91:82:fd:e6:89', 'dev_mac': '14:91:82:fd:e6:89', 'net_mac': '14:91:82:fd:e6:89', 'router_id': 'ap.TestNE'}),
                '107.181.195.129': NetworkMetadataReference(type='snmp_session', value='107.181.195.129', attributes={'cpe_mac': 'f8:ed:a5:f2:aa:b4', 'dev_mac': 'f8:ed:a5:f2:aa:b4', 'net_mac': 'f8:ed:a5:f2:aa:b4', 'router_id': 'ap.TestNE'}),
                '107.181.195.13': NetworkMetadataReference(type='snmp_session', value='107.181.195.13', attributes={'cpe_mac': 'ac:b3:13:b2:b4:54', 'dev_mac': 'ac:b3:13:b2:b4:54', 'net_mac': 'ac:b3:13:b2:b4:54', 'router_id': 'ap.TestNE'}),
                '107.181.195.130': NetworkMetadataReference(type='snmp_session', value='107.181.195.130', attributes={'cpe_mac': '14:91:82:29:0a:63', 'dev_mac': '14:91:82:29:0a:63', 'net_mac': '14:91:82:29:0a:63', 'router_id': 'ap.TestNE'}),
                '107.181.195.131': NetworkMetadataReference(type='snmp_session', value='107.181.195.131', attributes={'cpe_mac': 'e4:90:7e:f6:83:a0', 'dev_mac': 'e4:90:7e:f6:83:a0', 'net_mac': 'e4:90:7e:f6:83:a0', 'router_id': 'ap.TestNE'}),
                '107.181.195.132': NetworkMetadataReference(type='snmp_session', value='107.181.195.132', attributes={'cpe_mac': '78:71:9c:94:20:74', 'dev_mac': '78:71:9c:94:20:74', 'net_mac': '78:71:9c:94:20:74', 'router_id': 'ap.TestNE'}),
                '107.181.195.135': NetworkMetadataReference(type='snmp_session', value='107.181.195.135', attributes={'cpe_mac': '9c:34:26:44:02:3b', 'dev_mac': '9c:34:26:44:02:3b', 'net_mac': '9c:34:26:44:02:3b', 'router_id': 'ap.TestNE'}),
                '107.181.195.136': NetworkMetadataReference(type='snmp_session', value='107.181.195.136', attributes={'cpe_mac': '38:70:0c:68:4f:cb', 'dev_mac': '38:70:0c:68:4f:cb', 'net_mac': '38:70:0c:68:4f:cb', 'router_id': 'ap.TestNE'}),
                '107.181.195.137': NetworkMetadataReference(type='snmp_session', value='107.181.195.137', attributes={'cpe_mac': '78:71:9c:90:f2:b4', 'dev_mac': '78:71:9c:90:f2:b4', 'net_mac': '78:71:9c:90:f2:b4', 'router_id': 'ap.TestNE'}),
                '107.181.195.139': NetworkMetadataReference(type='snmp_session', value='107.181.195.139', attributes={'cpe_mac': '5c:8f:e0:28:69:93', 'dev_mac': '5c:8f:e0:28:69:93', 'net_mac': '5c:8f:e0:28:69:93', 'router_id': 'ap.TestNE'}),
                '107.181.195.141': NetworkMetadataReference(type='snmp_session', value='107.181.195.141', attributes={'cpe_mac': '38:70:0c:8f:a6:0f', 'dev_mac': '38:70:0c:8f:a6:0f', 'net_mac': '38:70:0c:8f:a6:0f', 'router_id': 'ap.TestNE'}),
                '107.181.195.142': NetworkMetadataReference(type='snmp_session', value='107.181.195.142', attributes={'cpe_mac': '5c:8f:e0:1b:14:a3', 'dev_mac': '5c:8f:e0:1b:14:a3', 'net_mac': '5c:8f:e0:1b:14:a3', 'router_id': 'ap.TestNE'}),
                '107.181.195.144': NetworkMetadataReference(type='snmp_session', value='107.181.195.144', attributes={'cpe_mac': '3c:7a:8a:f2:92:13', 'dev_mac': '3c:7a:8a:f2:92:13', 'net_mac': '3c:7a:8a:f2:92:13', 'router_id': 'ap.TestNE'}),
                '107.181.195.147': NetworkMetadataReference(type='snmp_session', value='107.181.195.147', attributes={'cpe_mac': 'ac:b3:13:b5:5b:74', 'dev_mac': 'ac:b3:13:b5:5b:74', 'net_mac': 'ac:b3:13:b5:5b:74', 'router_id': 'ap.TestNE'}),
                '107.181.195.149': NetworkMetadataReference(type='snmp_session', value='107.181.195.149', attributes={'cpe_mac': '38:70:0c:92:91:80', 'dev_mac': '38:70:0c:92:91:80', 'net_mac': '38:70:0c:92:91:80', 'router_id': 'ap.TestNE'}),
                '107.181.195.153': NetworkMetadataReference(type='snmp_session', value='107.181.195.153', attributes={'cpe_mac': '5c:8f:e0:1a:e7:83', 'dev_mac': '5c:8f:e0:1a:e7:83', 'net_mac': '5c:8f:e0:1a:e7:83', 'router_id': 'ap.TestNE'}),
                '107.181.195.154': NetworkMetadataReference(type='snmp_session', value='107.181.195.154', attributes={'cpe_mac': '94:a6:7e:75:65:68', 'dev_mac': '94:a6:7e:75:65:68', 'net_mac': '94:a6:7e:75:65:68', 'router_id': 'ap.TestNE'}),
                '107.181.195.155': NetworkMetadataReference(type='snmp_session', value='107.181.195.155', attributes={'cpe_mac': '5c:8f:e0:28:74:9b', 'dev_mac': '5c:8f:e0:28:74:9b', 'net_mac': '5c:8f:e0:28:74:9b', 'router_id': 'ap.TestNE'}),
                '107.181.195.156': NetworkMetadataReference(type='snmp_session', value='107.181.195.156', attributes={'cpe_mac': 'cc:a4:62:26:71:94', 'dev_mac': 'cc:a4:62:26:71:94', 'net_mac': 'cc:a4:62:26:71:94', 'router_id': 'ap.TestNE'}),
                '107.181.195.158': NetworkMetadataReference(type='snmp_session', value='107.181.195.158', attributes={'cpe_mac': '2c:99:24:8e:59:2c', 'dev_mac': '2c:99:24:8e:59:2c', 'net_mac': '2c:99:24:8e:59:2c', 'router_id': 'ap.TestNE'}),
                '107.181.195.16': NetworkMetadataReference(type='snmp_session', value='107.181.195.16', attributes={'cpe_mac': '98:6b:3d:c6:d7:94', 'dev_mac': '98:6b:3d:c6:d7:94', 'net_mac': '98:6b:3d:c6:d7:94', 'router_id': 'ap.TestNE'}),
                '107.181.195.162': NetworkMetadataReference(type='snmp_session', value='107.181.195.162', attributes={'cpe_mac': '90:1a:ca:db:eb:f4', 'dev_mac': '90:1a:ca:db:eb:f4', 'net_mac': '90:1a:ca:db:eb:f4', 'router_id': 'ap.TestNE'}),
                '107.181.195.167': NetworkMetadataReference(type='snmp_session', value='107.181.195.167', attributes={'cpe_mac': '90:1a:ca:71:7f:54', 'dev_mac': '90:1a:ca:71:7f:54', 'net_mac': '90:1a:ca:71:7f:54', 'router_id': 'ap.TestNE'}),
                '107.181.195.169': NetworkMetadataReference(type='snmp_session', value='107.181.195.169', attributes={'cpe_mac': '9c:c9:eb:32:e0:f9', 'dev_mac': '9c:c9:eb:32:e0:f9', 'net_mac': '9c:c9:eb:32:e0:f9', 'router_id': 'ap.TestNE'}),
                '107.181.195.170': NetworkMetadataReference(type='snmp_session', value='107.181.195.170', attributes={'cpe_mac': '90:1a:ca:d9:7a:d4', 'dev_mac': '90:1a:ca:d9:7a:d4', 'net_mac': '90:1a:ca:d9:7a:d4', 'router_id': 'ap.TestNE'}),
                '107.181.195.173': NetworkMetadataReference(type='snmp_session', value='107.181.195.173', attributes={'cpe_mac': '5c:8f:e0:1b:a7:b3', 'dev_mac': '5c:8f:e0:1b:a7:b3', 'net_mac': '5c:8f:e0:1b:a7:b3', 'router_id': 'ap.TestNE'}),
                '107.181.195.176': NetworkMetadataReference(type='snmp_session', value='107.181.195.176', attributes={'cpe_mac': '5c:8f:e0:4f:ac:1b', 'dev_mac': '5c:8f:e0:4f:ac:1b', 'net_mac': '5c:8f:e0:4f:ac:1b', 'router_id': 'ap.TestNE'}),
                '107.181.195.178': NetworkMetadataReference(type='snmp_session', value='107.181.195.178', attributes={'cpe_mac': '60:32:b1:3e:0a:78', 'dev_mac': '60:32:b1:3e:0a:78', 'net_mac': '60:32:b1:3e:0a:78', 'router_id': 'ap.TestNE'}),
                '107.181.195.18': NetworkMetadataReference(type='snmp_session', value='107.181.195.18', attributes={'cpe_mac': 'ac:b3:13:9f:dc:c4', 'dev_mac': 'ac:b3:13:9f:dc:c4', 'net_mac': 'ac:b3:13:9f:dc:c4', 'router_id': 'ap.TestNE'}),
                '107.181.195.180': NetworkMetadataReference(type='snmp_session', value='107.181.195.180', attributes={'cpe_mac': '78:71:9c:a1:7d:c4', 'dev_mac': '78:71:9c:a1:7d:c4', 'net_mac': '78:71:9c:a1:7d:c4', 'router_id': 'ap.TestNE'}),
                '107.181.195.184': NetworkMetadataReference(type='snmp_session', value='107.181.195.184', attributes={'cpe_mac': '3c:7a:8a:7d:b7:ab', 'dev_mac': '3c:7a:8a:7d:b7:ab', 'net_mac': '3c:7a:8a:7d:b7:ab', 'router_id': 'ap.TestNE'}),
                '107.181.195.185': NetworkMetadataReference(type='snmp_session', value='107.181.195.185', attributes={'cpe_mac': '94:87:7c:0b:19:04', 'dev_mac': '94:87:7c:0b:19:04', 'net_mac': '94:87:7c:0b:19:04', 'router_id': 'ap.TestNE'}),
                '107.181.195.186': NetworkMetadataReference(type='snmp_session', value='107.181.195.186', attributes={'cpe_mac': 'c0:c5:22:e1:67:3b', 'dev_mac': 'c0:c5:22:e1:67:3b', 'net_mac': 'c0:c5:22:e1:67:3b', 'router_id': 'ap.TestNE'}),
                '107.181.195.190': NetworkMetadataReference(type='snmp_session', value='107.181.195.190', attributes={'cpe_mac': '2c:99:24:d8:34:7a', 'dev_mac': '2c:99:24:d8:34:7a', 'net_mac': '2c:99:24:d8:34:7a', 'router_id': 'ap.TestNE'}),
                '107.181.195.191': NetworkMetadataReference(type='snmp_session', value='107.181.195.191', attributes={'cpe_mac': '2c:99:24:d6:60:ca', 'dev_mac': '2c:99:24:d6:60:ca', 'net_mac': '2c:99:24:d6:60:ca', 'router_id': 'ap.TestNE'}),
                '107.181.195.194': NetworkMetadataReference(type='snmp_session', value='107.181.195.194', attributes={'cpe_mac': '38:70:0c:96:38:31', 'dev_mac': '38:70:0c:96:38:31', 'net_mac': '38:70:0c:96:38:31', 'router_id': 'ap.TestNE'}),
                '107.181.195.195': NetworkMetadataReference(type='snmp_session', value='107.181.195.195', attributes={'cpe_mac': '90:9a:4a:f2:7e:17', 'dev_mac': '90:9a:4a:f2:7e:17', 'net_mac': '90:9a:4a:f2:7e:17', 'router_id': 'ap.TestNE'}),
                '107.181.195.197': NetworkMetadataReference(type='snmp_session', value='107.181.195.197', attributes={'cpe_mac': '38:70:0c:8f:ac:91', 'dev_mac': '38:70:0c:8f:ac:91', 'net_mac': '38:70:0c:8f:ac:91', 'router_id': 'ap.TestNE'}),
                '107.181.195.199': NetworkMetadataReference(type='snmp_session', value='107.181.195.199', attributes={'cpe_mac': '54:65:de:6f:c4:d4', 'dev_mac': '54:65:de:6f:c4:d4', 'net_mac': '54:65:de:6f:c4:d4', 'router_id': 'ap.TestNE'}),
                '107.181.195.2': NetworkMetadataReference(type='snmp_session', value='107.181.195.2', attributes={'cpe_mac': '78:71:9c:9b:e1:14', 'dev_mac': '78:71:9c:9b:e1:14', 'net_mac': '78:71:9c:9b:e1:14', 'router_id': 'ap.TestNE'}),
                '107.181.195.20': NetworkMetadataReference(type='snmp_session', value='107.181.195.20', attributes={'cpe_mac': '38:70:0c:f2:84:3b', 'dev_mac': '38:70:0c:f2:84:3b', 'net_mac': '38:70:0c:f2:84:3b', 'router_id': 'ap.TestNE'}),
                '107.181.195.200': NetworkMetadataReference(type='snmp_session', value='107.181.195.200', attributes={'cpe_mac': '14:ab:f0:d3:47:d4', 'dev_mac': '14:ab:f0:d3:47:d4', 'net_mac': '14:ab:f0:d3:47:d4', 'router_id': 'ap.TestNE'}),
                '107.181.195.205': NetworkMetadataReference(type='snmp_session', value='107.181.195.205', attributes={'cpe_mac': '6c:cd:d6:ae:51:00', 'dev_mac': '6c:cd:d6:ae:51:00', 'net_mac': '6c:cd:d6:ae:51:00', 'router_id': 'ap.TestNE'}),
                '107.181.195.207': NetworkMetadataReference(type='snmp_session', value='107.181.195.207', attributes={'cpe_mac': '38:70:0c:63:a7:55', 'dev_mac': '38:70:0c:63:a7:55', 'net_mac': '38:70:0c:63:a7:55', 'router_id': 'ap.TestNE'}),
                '107.181.195.208': NetworkMetadataReference(type='snmp_session', value='107.181.195.208', attributes={'cpe_mac': '60:38:e0:33:04:34', 'dev_mac': '60:38:e0:33:04:34', 'net_mac': '60:38:e0:33:04:34', 'router_id': 'ap.TestNE'}),
                '107.181.195.209': NetworkMetadataReference(type='snmp_session', value='107.181.195.209', attributes={'cpe_mac': '38:70:0c:8f:5b:5b', 'dev_mac': '38:70:0c:8f:5b:5b', 'net_mac': '38:70:0c:8f:5b:5b', 'router_id': 'ap.TestNE'}),
                '107.181.195.21': NetworkMetadataReference(type='snmp_session', value='107.181.195.21', attributes={'cpe_mac': '0c:f8:93:de:7d:e4', 'dev_mac': '0c:f8:93:de:7d:e4', 'net_mac': '0c:f8:93:de:7d:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.210': NetworkMetadataReference(type='snmp_session', value='107.181.195.210', attributes={'cpe_mac': '38:70:0c:8f:6b:ed', 'dev_mac': '38:70:0c:8f:6b:ed', 'net_mac': '38:70:0c:8f:6b:ed', 'router_id': 'ap.TestNE'}),
                '107.181.195.211': NetworkMetadataReference(type='snmp_session', value='107.181.195.211', attributes={'cpe_mac': '98:6b:3d:dc:60:44', 'dev_mac': '98:6b:3d:dc:60:44', 'net_mac': '98:6b:3d:dc:60:44', 'router_id': 'ap.TestNE'}),
                '107.181.195.213': NetworkMetadataReference(type='snmp_session', value='107.181.195.213', attributes={'cpe_mac': '3c:7a:8a:ec:11:2b', 'dev_mac': '3c:7a:8a:ec:11:2b', 'net_mac': '3c:7a:8a:ec:11:2b', 'router_id': 'ap.TestNE'}),
                '107.181.195.214': NetworkMetadataReference(type='snmp_session', value='107.181.195.214', attributes={'cpe_mac': '9c:34:26:b0:f7:d3', 'dev_mac': '9c:34:26:b0:f7:d3', 'net_mac': '9c:34:26:b0:f7:d3', 'router_id': 'ap.TestNE'}),
                '107.181.195.215': NetworkMetadataReference(type='snmp_session', value='107.181.195.215', attributes={'cpe_mac': '9c:34:26:b0:be:6b', 'dev_mac': '9c:34:26:b0:be:6b', 'net_mac': '9c:34:26:b0:be:6b', 'router_id': 'ap.TestNE'}),
                '107.181.195.216': NetworkMetadataReference(type='snmp_session', value='107.181.195.216', attributes={'cpe_mac': '38:70:0c:6d:47:d3', 'dev_mac': '38:70:0c:6d:47:d3', 'net_mac': '38:70:0c:6d:47:d3', 'router_id': 'ap.TestNE'}),
                '107.181.195.217': NetworkMetadataReference(type='snmp_session', value='107.181.195.217', attributes={'cpe_mac': '38:70:0c:93:98:3f', 'dev_mac': '38:70:0c:93:98:3f', 'net_mac': '38:70:0c:93:98:3f', 'router_id': 'ap.TestNE'}),
                '107.181.195.218': NetworkMetadataReference(type='snmp_session', value='107.181.195.218', attributes={'cpe_mac': '44:10:91:66:03:e4', 'dev_mac': '44:10:91:66:03:e4', 'net_mac': '44:10:91:66:03:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.219': NetworkMetadataReference(type='snmp_session', value='107.181.195.219', attributes={'cpe_mac': '08:3e:0c:ec:43:04', 'dev_mac': '08:3e:0c:ec:43:04', 'net_mac': '08:3e:0c:ec:43:04', 'router_id': 'ap.TestNE'}),
                '107.181.195.22': NetworkMetadataReference(type='snmp_session', value='107.181.195.22', attributes={'cpe_mac': 'ac:b3:13:b6:cb:94', 'dev_mac': 'ac:b3:13:b6:cb:94', 'net_mac': 'ac:b3:13:b6:cb:94', 'router_id': 'ap.TestNE'}),
                '107.181.195.220': NetworkMetadataReference(type='snmp_session', value='107.181.195.220', attributes={'cpe_mac': '38:70:0c:6a:7e:c8', 'dev_mac': '38:70:0c:6a:7e:c8', 'net_mac': '38:70:0c:6a:7e:c8', 'router_id': 'ap.TestNE'}),
                '107.181.195.221': NetworkMetadataReference(type='snmp_session', value='107.181.195.221', attributes={'cpe_mac': '70:8b:cd:ac:dc:80', 'dev_mac': '70:8b:cd:ac:dc:80', 'net_mac': '70:8b:cd:ac:dc:80', 'router_id': 'ap.TestNE'}),
                '107.181.195.222': NetworkMetadataReference(type='snmp_session', value='107.181.195.222', attributes={'cpe_mac': '94:87:7c:10:62:24', 'dev_mac': '94:87:7c:10:62:24', 'net_mac': '94:87:7c:10:62:24', 'router_id': 'ap.TestNE'}),
                '107.181.195.24': NetworkMetadataReference(type='snmp_session', value='107.181.195.24', attributes={'cpe_mac': '90:1a:ca:5e:3a:84', 'dev_mac': '90:1a:ca:5e:3a:84', 'net_mac': '90:1a:ca:5e:3a:84', 'router_id': 'ap.TestNE'}),
                '107.181.195.25': NetworkMetadataReference(type='snmp_session', value='107.181.195.25', attributes={'cpe_mac': 'ac:b3:13:0c:b0:54', 'dev_mac': 'ac:b3:13:0c:b0:54', 'net_mac': 'ac:b3:13:0c:b0:54', 'router_id': 'ap.TestNE'}),
                '107.181.195.251': NetworkMetadataReference(type='snmp_session', value='107.181.195.251', attributes={'cpe_mac': 'f0:9f:c2:36:00:dc', 'dev_mac': 'f0:9f:c2:36:00:dc', 'net_mac': 'f0:9f:c2:36:00:dc', 'router_id': 'ap.TestNE'}),
                '107.181.195.253': NetworkMetadataReference(type='snmp_session', value='107.181.195.253', attributes={'cpe_mac': '38:70:0c:40:aa:0b', 'dev_mac': '38:70:0c:40:aa:0b', 'net_mac': '38:70:0c:40:aa:0b', 'router_id': 'ap.TestNE'}),
                '107.181.195.254': NetworkMetadataReference(type='snmp_session', value='107.181.195.254', attributes={'cpe_mac': 'ac:b3:13:8f:85:34', 'dev_mac': 'ac:b3:13:8f:85:34', 'net_mac': 'ac:b3:13:8f:85:34', 'router_id': 'ap.TestNE'}),
                '107.181.195.27': NetworkMetadataReference(type='snmp_session', value='107.181.195.27', attributes={'cpe_mac': '90:1a:ca:7b:69:94', 'dev_mac': '90:1a:ca:7b:69:94', 'net_mac': '90:1a:ca:7b:69:94', 'router_id': 'ap.TestNE'}),
                '107.181.195.28': NetworkMetadataReference(type='snmp_session', value='107.181.195.28', attributes={'cpe_mac': '98:6b:3d:dc:a2:84', 'dev_mac': '98:6b:3d:dc:a2:84', 'net_mac': '98:6b:3d:dc:a2:84', 'router_id': 'ap.TestNE'}),
                '107.181.195.3': NetworkMetadataReference(type='snmp_session', value='107.181.195.3', attributes={'cpe_mac': '90:1a:ca:5e:14:c4', 'dev_mac': '90:1a:ca:5e:14:c4', 'net_mac': '90:1a:ca:5e:14:c4', 'router_id': 'ap.TestNE'}),
                '107.181.195.30': NetworkMetadataReference(type='snmp_session', value='107.181.195.30', attributes={'cpe_mac': '98:6b:3d:e5:51:e4', 'dev_mac': '98:6b:3d:e5:51:e4', 'net_mac': '98:6b:3d:e5:51:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.31': NetworkMetadataReference(type='snmp_session', value='107.181.195.31', attributes={'cpe_mac': '6c:ae:f6:4f:03:20', 'dev_mac': '6c:ae:f6:4f:03:20', 'net_mac': '6c:ae:f6:4f:03:20', 'router_id': 'ap.TestNE'}),
                '107.181.195.33': NetworkMetadataReference(type='snmp_session', value='107.181.195.33', attributes={'cpe_mac': '94:87:7c:00:27:14', 'dev_mac': '94:87:7c:00:27:14', 'net_mac': '94:87:7c:00:27:14', 'router_id': 'ap.TestNE'}),
                '107.181.195.34': NetworkMetadataReference(type='snmp_session', value='107.181.195.34', attributes={'cpe_mac': '38:70:0c:8f:01:b2', 'dev_mac': '38:70:0c:8f:01:b2', 'net_mac': '38:70:0c:8f:01:b2', 'router_id': 'ap.TestNE'}),
                '107.181.195.38': NetworkMetadataReference(type='snmp_session', value='107.181.195.38', attributes={'cpe_mac': '08:3e:0c:ec:39:b4', 'dev_mac': '08:3e:0c:ec:39:b4', 'net_mac': '08:3e:0c:ec:39:b4', 'router_id': 'ap.TestNE'}),
                '107.181.195.39': NetworkMetadataReference(type='snmp_session', value='107.181.195.39', attributes={'cpe_mac': '08:3e:0c:e8:51:b4', 'dev_mac': '08:3e:0c:e8:51:b4', 'net_mac': '08:3e:0c:e8:51:b4', 'router_id': 'ap.TestNE'}),
                '107.181.195.4': NetworkMetadataReference(type='snmp_session', value='107.181.195.4', attributes={'cpe_mac': '2c:99:24:8e:5f:04', 'dev_mac': '2c:99:24:8e:5f:04', 'net_mac': '2c:99:24:8e:5f:04', 'router_id': 'ap.TestNE'}),
                '107.181.195.41': NetworkMetadataReference(type='snmp_session', value='107.181.195.41', attributes={'cpe_mac': '14:ab:f0:d0:22:e4', 'dev_mac': '14:ab:f0:d0:22:e4', 'net_mac': '14:ab:f0:d0:22:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.42': NetworkMetadataReference(type='snmp_session', value='107.181.195.42', attributes={'cpe_mac': '98:6b:3d:c5:50:44', 'dev_mac': '98:6b:3d:c5:50:44', 'net_mac': '98:6b:3d:c5:50:44', 'router_id': 'ap.TestNE'}),
                '107.181.195.43': NetworkMetadataReference(type='snmp_session', value='107.181.195.43', attributes={'cpe_mac': 'ac:b3:13:a2:05:14', 'dev_mac': 'ac:b3:13:a2:05:14', 'net_mac': 'ac:b3:13:a2:05:14', 'router_id': 'ap.TestNE'}),
                '107.181.195.45': NetworkMetadataReference(type='snmp_session', value='107.181.195.45', attributes={'cpe_mac': '40:70:09:94:7a:04', 'dev_mac': '40:70:09:94:7a:04', 'net_mac': '40:70:09:94:7a:04', 'router_id': 'ap.TestNE'}),
                '107.181.195.46': NetworkMetadataReference(type='snmp_session', value='107.181.195.46', attributes={'cpe_mac': '54:65:de:6f:6a:54', 'dev_mac': '54:65:de:6f:6a:54', 'net_mac': '54:65:de:6f:6a:54', 'router_id': 'ap.TestNE'}),
                '107.181.195.47': NetworkMetadataReference(type='snmp_session', value='107.181.195.47', attributes={'cpe_mac': '14:cf:e2:9c:ce:e4', 'dev_mac': '14:cf:e2:9c:ce:e4', 'net_mac': '14:cf:e2:9c:ce:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.48': NetworkMetadataReference(type='snmp_session', value='107.181.195.48', attributes={'cpe_mac': '40:70:09:aa:b2:e4', 'dev_mac': '40:70:09:aa:b2:e4', 'net_mac': '40:70:09:aa:b2:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.49': NetworkMetadataReference(type='snmp_session', value='107.181.195.49', attributes={'cpe_mac': '78:71:9c:8d:0f:84', 'dev_mac': '78:71:9c:8d:0f:84', 'net_mac': '78:71:9c:8d:0f:84', 'router_id': 'ap.TestNE'}),
                '107.181.195.5': NetworkMetadataReference(type='snmp_session', value='107.181.195.5', attributes={'cpe_mac': '54:65:de:fe:9d:24', 'dev_mac': '54:65:de:fe:9d:24', 'net_mac': '54:65:de:fe:9d:24', 'router_id': 'ap.TestNE'}),
                '107.181.195.50': NetworkMetadataReference(type='snmp_session', value='107.181.195.50', attributes={'cpe_mac': '14:cf:e2:a0:15:c4', 'dev_mac': '14:cf:e2:a0:15:c4', 'net_mac': '14:cf:e2:a0:15:c4', 'router_id': 'ap.TestNE'}),
                '107.181.195.51': NetworkMetadataReference(type='snmp_session', value='107.181.195.51', attributes={'cpe_mac': '00:ac:e0:b8:07:94', 'dev_mac': '00:ac:e0:b8:07:94', 'net_mac': '00:ac:e0:b8:07:94', 'router_id': 'ap.TestNE'}),
                '107.181.195.52': NetworkMetadataReference(type='snmp_session', value='107.181.195.52', attributes={'cpe_mac': '94:87:7c:0a:91:24', 'dev_mac': '94:87:7c:0a:91:24', 'net_mac': '94:87:7c:0a:91:24', 'router_id': 'ap.TestNE'}),
                '107.181.195.53': NetworkMetadataReference(type='snmp_session', value='107.181.195.53', attributes={'cpe_mac': '2c:99:24:8e:4d:24', 'dev_mac': '2c:99:24:8e:4d:24', 'net_mac': '2c:99:24:8e:4d:24', 'router_id': 'ap.TestNE'}),
                '107.181.195.54': NetworkMetadataReference(type='snmp_session', value='107.181.195.54', attributes={'cpe_mac': '2c:99:24:8e:56:14', 'dev_mac': '2c:99:24:8e:56:14', 'net_mac': '2c:99:24:8e:56:14', 'router_id': 'ap.TestNE'}),
                '107.181.195.55': NetworkMetadataReference(type='snmp_session', value='107.181.195.55', attributes={'cpe_mac': '94:87:7c:01:e3:a4', 'dev_mac': '94:87:7c:01:e3:a4', 'net_mac': '94:87:7c:01:e3:a4', 'router_id': 'ap.TestNE'}),
                '107.181.195.58': NetworkMetadataReference(type='snmp_session', value='107.181.195.58', attributes={'cpe_mac': '9c:34:26:4e:92:5b', 'dev_mac': '9c:34:26:4e:92:5b', 'net_mac': '9c:34:26:4e:92:5b', 'router_id': 'ap.TestNE'}),
                '107.181.195.59': NetworkMetadataReference(type='snmp_session', value='107.181.195.59', attributes={'cpe_mac': '98:6b:3d:d2:0d:b4', 'dev_mac': '98:6b:3d:d2:0d:b4', 'net_mac': '98:6b:3d:d2:0d:b4', 'router_id': 'ap.TestNE'}),
                '107.181.195.6': NetworkMetadataReference(type='snmp_session', value='107.181.195.6', attributes={'cpe_mac': '10:86:8c:16:1e:13', 'dev_mac': '10:86:8c:16:1e:13', 'net_mac': '10:86:8c:16:1e:13', 'router_id': 'ap.TestNE'}),
                '107.181.195.64': NetworkMetadataReference(type='snmp_session', value='107.181.195.64', attributes={'cpe_mac': '5c:8f:e0:1b:c1:5b', 'dev_mac': '5c:8f:e0:1b:c1:5b', 'net_mac': '5c:8f:e0:1b:c1:5b', 'router_id': 'ap.TestNE'}),
                '107.181.195.65': NetworkMetadataReference(type='snmp_session', value='107.181.195.65', attributes={'cpe_mac': '08:3e:0c:ec:ce:c4', 'dev_mac': '08:3e:0c:ec:ce:c4', 'net_mac': '08:3e:0c:ec:ce:c4', 'router_id': 'ap.TestNE'}),
                '107.181.195.66': NetworkMetadataReference(type='snmp_session', value='107.181.195.66', attributes={'cpe_mac': '00:ac:e0:bf:40:94', 'dev_mac': '00:ac:e0:bf:40:94', 'net_mac': '00:ac:e0:bf:40:94', 'router_id': 'ap.TestNE'}),
                '107.181.195.67': NetworkMetadataReference(type='snmp_session', value='107.181.195.67', attributes={'cpe_mac': '38:70:0c:6a:ea:4c', 'dev_mac': '38:70:0c:6a:ea:4c', 'net_mac': '38:70:0c:6a:ea:4c', 'router_id': 'ap.TestNE'}),
                '107.181.195.68': NetworkMetadataReference(type='snmp_session', value='107.181.195.68', attributes={'cpe_mac': '78:d2:94:b0:d4:ff', 'dev_mac': '78:d2:94:b0:d4:ff', 'net_mac': '78:d2:94:b0:d4:ff', 'router_id': 'ap.TestNE'}),
                '107.181.195.70': NetworkMetadataReference(type='snmp_session', value='107.181.195.70', attributes={'cpe_mac': '90:1a:ca:7c:2b:64', 'dev_mac': '90:1a:ca:7c:2b:64', 'net_mac': '90:1a:ca:7c:2b:64', 'router_id': 'ap.TestNE'}),
                '107.181.195.71': NetworkMetadataReference(type='snmp_session', value='107.181.195.71', attributes={'cpe_mac': 'ac:b3:13:a9:70:04', 'dev_mac': 'ac:b3:13:a9:70:04', 'net_mac': 'ac:b3:13:a9:70:04', 'router_id': 'ap.TestNE'}),
                '107.181.195.72': NetworkMetadataReference(type='snmp_session', value='107.181.195.72', attributes={'cpe_mac': '40:70:09:ac:13:84', 'dev_mac': '40:70:09:ac:13:84', 'net_mac': '40:70:09:ac:13:84', 'router_id': 'ap.TestNE'}),
                '107.181.195.73': NetworkMetadataReference(type='snmp_session', value='107.181.195.73', attributes={'cpe_mac': 'e8:89:2c:7e:83:e4', 'dev_mac': 'e8:89:2c:7e:83:e4', 'net_mac': 'e8:89:2c:7e:83:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.74': NetworkMetadataReference(type='snmp_session', value='107.181.195.74', attributes={'cpe_mac': '38:70:0c:ea:0e:fb', 'dev_mac': '38:70:0c:ea:0e:fb', 'net_mac': '38:70:0c:ea:0e:fb', 'router_id': 'ap.TestNE'}),
                '107.181.195.75': NetworkMetadataReference(type='snmp_session', value='107.181.195.75', attributes={'cpe_mac': '38:70:0c:85:c3:8b', 'dev_mac': '38:70:0c:85:c3:8b', 'net_mac': '38:70:0c:85:c3:8b', 'router_id': 'ap.TestNE'}),
                '107.181.195.77': NetworkMetadataReference(type='snmp_session', value='107.181.195.77', attributes={'cpe_mac': '38:70:0c:6a:bf:5e', 'dev_mac': '38:70:0c:6a:bf:5e', 'net_mac': '38:70:0c:6a:bf:5e', 'router_id': 'ap.TestNE'}),
                '107.181.195.79': NetworkMetadataReference(type='snmp_session', value='107.181.195.79', attributes={'cpe_mac': '3c:7a:8a:ec:11:53', 'dev_mac': '3c:7a:8a:ec:11:53', 'net_mac': '3c:7a:8a:ec:11:53', 'router_id': 'ap.TestNE'}),
                '107.181.195.80': NetworkMetadataReference(type='snmp_session', value='107.181.195.80', attributes={'cpe_mac': '3c:7a:8a:ef:02:0b', 'dev_mac': '3c:7a:8a:ef:02:0b', 'net_mac': '3c:7a:8a:ef:02:0b', 'router_id': 'ap.TestNE'}),
                '107.181.195.81': NetworkMetadataReference(type='snmp_session', value='107.181.195.81', attributes={'cpe_mac': 'ac:b3:13:77:26:e4', 'dev_mac': 'ac:b3:13:77:26:e4', 'net_mac': 'ac:b3:13:77:26:e4', 'router_id': 'ap.TestNE'}),
                '107.181.195.85': NetworkMetadataReference(type='snmp_session', value='107.181.195.85', attributes={'cpe_mac': 'b0:b9:8a:51:1f:1a', 'dev_mac': 'b0:b9:8a:51:1f:1a', 'net_mac': 'b0:b9:8a:51:1f:1a', 'router_id': 'ap.TestNE'}),
                '107.181.195.91': NetworkMetadataReference(type='snmp_session', value='107.181.195.91', attributes={'cpe_mac': 'ac:b3:13:80:8a:04', 'dev_mac': 'ac:b3:13:80:8a:04', 'net_mac': 'ac:b3:13:80:8a:04', 'router_id': 'ap.TestNE'})
        }
        cpe_mac_refs = {
                '00:ac:e0:b8:07:92': NetworkMetadataReference(type='cpe_mac', value='00:ac:e0:b8:07:92', attributes={'ap': 'TestNE', 'sm': '00:ac:e0:b8:07:92'}),
                '00:ac:e0:bf:40:92': NetworkMetadataReference(type='cpe_mac', value='00:ac:e0:bf:40:92', attributes={'ap': 'TestNE', 'sm': '00:ac:e0:bf:40:92'}),
                '08:3e:0c:10:67:72': NetworkMetadataReference(type='cpe_mac', value='08:3e:0c:10:67:72', attributes={'ap': 'TestNE', 'sm': '08:3e:0c:10:67:72'}),
                '08:3e:0c:e8:51:b2': NetworkMetadataReference(type='cpe_mac', value='08:3e:0c:e8:51:b2', attributes={'ap': 'TestNE', 'sm': '08:3e:0c:e8:51:b2'}),
                '08:3e:0c:ec:39:b2': NetworkMetadataReference(type='cpe_mac', value='08:3e:0c:ec:39:b2', attributes={'ap': 'TestNE', 'sm': '08:3e:0c:ec:39:b2'}),
                '08:3e:0c:ec:43:02': NetworkMetadataReference(type='cpe_mac', value='08:3e:0c:ec:43:02', attributes={'ap': 'TestNE', 'sm': '08:3e:0c:ec:43:02'}),
                '08:3e:0c:ec:ce:c2': NetworkMetadataReference(type='cpe_mac', value='08:3e:0c:ec:ce:c2', attributes={'ap': 'TestNE', 'sm': '08:3e:0c:ec:ce:c2'}),
                '0c:f8:93:de:7d:e2': NetworkMetadataReference(type='cpe_mac', value='0c:f8:93:de:7d:e2', attributes={'ap': 'TestNE', 'sm': '0c:f8:93:de:7d:e2'}),
                '10:86:8c:16:1e:12': NetworkMetadataReference(type='cpe_mac', value='10:86:8c:16:1e:12', attributes={'ap': 'TestNE', 'sm': '10:86:8c:16:1e:12'}),
                '14:91:82:29:0a:63': NetworkMetadataReference(type='cpe_mac', value='14:91:82:29:0a:63', attributes={'ap': 'TestNE', 'sm': 'f8:ed:a5:dc:5e:e2'}),
                '14:91:82:fd:e6:89': NetworkMetadataReference(type='cpe_mac', value='14:91:82:fd:e6:89', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:78:36:72'}),
                '14:ab:f0:d0:22:e2': NetworkMetadataReference(type='cpe_mac', value='14:ab:f0:d0:22:e2', attributes={'ap': 'TestNE', 'sm': '14:ab:f0:d0:22:e2'}),
                '14:ab:f0:d3:47:d2': NetworkMetadataReference(type='cpe_mac', value='14:ab:f0:d3:47:d2', attributes={'ap': 'TestNE', 'sm': '14:ab:f0:d3:47:d2'}),
                '14:cf:e2:9c:ce:e2': NetworkMetadataReference(type='cpe_mac', value='14:cf:e2:9c:ce:e2', attributes={'ap': 'TestNE', 'sm': '14:cf:e2:9c:ce:e2'}),
                '14:cf:e2:a0:15:c2': NetworkMetadataReference(type='cpe_mac', value='14:cf:e2:a0:15:c2', attributes={'ap': 'TestNE', 'sm': '14:cf:e2:a0:15:c2'}),
                '14:cf:e2:a0:26:12': NetworkMetadataReference(type='cpe_mac', value='14:cf:e2:a0:26:12', attributes={'ap': 'TestNE', 'sm': '14:cf:e2:a0:26:12'}),
                '14:cf:e2:a9:1f:12': NetworkMetadataReference(type='cpe_mac', value='14:cf:e2:a9:1f:12', attributes={'ap': 'TestNE', 'sm': '14:cf:e2:a9:1f:12'}),
                '24:94:cb:28:6f:1d': NetworkMetadataReference(type='cpe_mac', value='24:94:cb:28:6f:1d', attributes={'ap': 'TestNE', 'sm': '24:94:cb:28:6f:1d'}),
                '2c:99:24:52:71:e3': NetworkMetadataReference(type='cpe_mac', value='2c:99:24:52:71:e3', attributes={'ap': 'TestNE', 'sm': '2c:99:24:52:71:e3'}),
                '2c:99:24:8e:4d:23': NetworkMetadataReference(type='cpe_mac', value='2c:99:24:8e:4d:23', attributes={'ap': 'TestNE', 'sm': '2c:99:24:8e:4d:23'}),
                '2c:99:24:8e:56:13': NetworkMetadataReference(type='cpe_mac', value='2c:99:24:8e:56:13', attributes={'ap': 'TestNE', 'sm': '2c:99:24:8e:56:13'}),
                '2c:99:24:8e:59:2b': NetworkMetadataReference(type='cpe_mac', value='2c:99:24:8e:59:2b', attributes={'ap': 'TestNE', 'sm': '2c:99:24:8e:59:2b'}),
                '2c:99:24:8e:5f:03': NetworkMetadataReference(type='cpe_mac', value='2c:99:24:8e:5f:03', attributes={'ap': 'TestNE', 'sm': '2c:99:24:8e:5f:03'}),
                '2c:99:24:d6:60:c9': NetworkMetadataReference(type='cpe_mac', value='2c:99:24:d6:60:c9', attributes={'ap': 'TestNE', 'sm': '2c:99:24:d6:60:c9'}),
                '2c:99:24:d8:34:79': NetworkMetadataReference(type='cpe_mac', value='2c:99:24:d8:34:79', attributes={'ap': 'TestNE', 'sm': '2c:99:24:d8:34:79'}),
                '38:70:0c:02:e8:b8': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:02:e8:b8', attributes={'ap': 'TestNE', 'sm': '38:70:0c:02:e8:b8'}),
                '38:70:0c:3e:cc:3a': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:3e:cc:3a', attributes={'ap': 'TestNE', 'sm': '38:70:0c:3e:cc:3a'}),
                '38:70:0c:40:aa:0a': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:40:aa:0a', attributes={'ap': 'TestNE', 'sm': '38:70:0c:40:aa:0a'}),
                '38:70:0c:63:a7:54': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:63:a7:54', attributes={'ap': 'TestNE', 'sm': '38:70:0c:63:a7:54'}),
                '38:70:0c:68:4f:ca': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:68:4f:ca', attributes={'ap': 'TestNE', 'sm': '38:70:0c:68:4f:ca'}),
                '38:70:0c:6a:57:e5': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:6a:57:e5', attributes={'ap': 'TestNE', 'sm': '38:70:0c:6a:57:e5'}),
                '38:70:0c:6a:7e:c7': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:6a:7e:c7', attributes={'ap': 'TestNE', 'sm': '38:70:0c:6a:7e:c7'}),
                '38:70:0c:6a:bf:5d': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:6a:bf:5d', attributes={'ap': 'TestNE', 'sm': '38:70:0c:6a:bf:5d'}),
                '38:70:0c:6a:ea:4b': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:6a:ea:4b', attributes={'ap': 'TestNE', 'sm': '38:70:0c:6a:ea:4b'}),
                '38:70:0c:6d:47:d2': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:6d:47:d2', attributes={'ap': 'TestNE', 'sm': '38:70:0c:6d:47:d2'}),
                '38:70:0c:85:c3:8a': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:85:c3:8a', attributes={'ap': 'TestNE', 'sm': '38:70:0c:85:c3:8a'}),
                '38:70:0c:8f:01:b1': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:8f:01:b1', attributes={'ap': 'TestNE', 'sm': '38:70:0c:8f:01:b1'}),
                '38:70:0c:8f:5b:5a': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:8f:5b:5a', attributes={'ap': 'TestNE', 'sm': '38:70:0c:8f:5b:5a'}),
                '38:70:0c:8f:6b:ec': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:8f:6b:ec', attributes={'ap': 'TestNE', 'sm': '38:70:0c:8f:6b:ec'}),
                '38:70:0c:8f:85:7d': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:8f:85:7d', attributes={'ap': 'TestNE', 'sm': '38:70:0c:8f:85:7d'}),
                '38:70:0c:8f:a6:0e': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:8f:a6:0e', attributes={'ap': 'TestNE', 'sm': '38:70:0c:8f:a6:0e'}),
                '38:70:0c:8f:ac:90': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:8f:ac:90', attributes={'ap': 'TestNE', 'sm': '38:70:0c:8f:ac:90'}),
                '38:70:0c:8f:bc:a4': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:8f:bc:a4', attributes={'ap': 'TestNE', 'sm': '38:70:0c:8f:bc:a4'}),
                '38:70:0c:8f:c2:70': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:8f:c2:70', attributes={'ap': 'TestNE', 'sm': '38:70:0c:8f:c2:70'}),
                '38:70:0c:92:91:7f': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:92:91:7f', attributes={'ap': 'TestNE', 'sm': '38:70:0c:92:91:7f'}),
                '38:70:0c:93:98:3e': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:93:98:3e', attributes={'ap': 'TestNE', 'sm': '38:70:0c:93:98:3e'}),
                '38:70:0c:96:38:30': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:96:38:30', attributes={'ap': 'TestNE', 'sm': '38:70:0c:96:38:30'}),
                '38:70:0c:ea:0e:fa': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:ea:0e:fa', attributes={'ap': 'TestNE', 'sm': '38:70:0c:ea:0e:fa'}),
                '38:70:0c:f2:84:3a': NetworkMetadataReference(type='cpe_mac', value='38:70:0c:f2:84:3a', attributes={'ap': 'TestNE', 'sm': '38:70:0c:f2:84:3a'}),
                '3c:7a:8a:75:a7:42': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:75:a7:42', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:75:a7:42'}),
                '3c:7a:8a:76:94:4a': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:76:94:4a', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:76:94:4a'}),
                '3c:7a:8a:78:0d:3a': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:78:0d:3a', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:78:0d:3a'}),
                '3c:7a:8a:7a:f6:e2': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:7a:f6:e2', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:7a:f6:e2'}),
                '3c:7a:8a:7d:b7:aa': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:7d:b7:aa', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:7d:b7:aa'}),
                '3c:7a:8a:7e:91:fa': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:7e:91:fa', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:7e:91:fa'}),
                '3c:7a:8a:7e:c0:32': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:7e:c0:32', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:7e:c0:32'}),
                '3c:7a:8a:ec:11:2a': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:ec:11:2a', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:ec:11:2a'}),
                '3c:7a:8a:ec:11:52': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:ec:11:52', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:ec:11:52'}),
                '3c:7a:8a:ef:02:0a': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:ef:02:0a', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:ef:02:0a'}),
                '3c:7a:8a:f2:92:12': NetworkMetadataReference(type='cpe_mac', value='3c:7a:8a:f2:92:12', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:f2:92:12'}),
                '40:70:09:71:53:5c': NetworkMetadataReference(type='cpe_mac', value='40:70:09:71:53:5c', attributes={'ap': 'TestNE', 'sm': '40:70:09:71:53:5c'}),
                '40:70:09:71:a2:32': NetworkMetadataReference(type='cpe_mac', value='40:70:09:71:a2:32', attributes={'ap': 'TestNE', 'sm': '40:70:09:71:a2:32'}),
                '40:70:09:86:0b:62': NetworkMetadataReference(type='cpe_mac', value='40:70:09:86:0b:62', attributes={'ap': 'TestNE', 'sm': '40:70:09:86:0b:62'}),
                '40:70:09:8a:27:92': NetworkMetadataReference(type='cpe_mac', value='40:70:09:8a:27:92', attributes={'ap': 'TestNE', 'sm': '40:70:09:8a:27:92'}),
                '40:70:09:8c:07:a2': NetworkMetadataReference(type='cpe_mac', value='40:70:09:8c:07:a2', attributes={'ap': 'TestNE', 'sm': '40:70:09:8c:07:a2'}),
                '40:70:09:91:e4:c2': NetworkMetadataReference(type='cpe_mac', value='40:70:09:91:e4:c2', attributes={'ap': 'TestNE', 'sm': '40:70:09:91:e4:c2'}),
                '40:70:09:94:7a:02': NetworkMetadataReference(type='cpe_mac', value='40:70:09:94:7a:02', attributes={'ap': 'TestNE', 'sm': '40:70:09:94:7a:02'}),
                '40:70:09:aa:b2:e2': NetworkMetadataReference(type='cpe_mac', value='40:70:09:aa:b2:e2', attributes={'ap': 'TestNE', 'sm': '40:70:09:aa:b2:e2'}),
                '40:70:09:ac:13:82': NetworkMetadataReference(type='cpe_mac', value='40:70:09:ac:13:82', attributes={'ap': 'TestNE', 'sm': '40:70:09:ac:13:82'}),
                '44:10:91:66:03:e4': NetworkMetadataReference(type='cpe_mac', value='44:10:91:66:03:e4', attributes={'ap': 'TestNE', 'sm': '24:94:cb:28:6f:1d'}),
                '44:a5:6e:45:d0:d5': NetworkMetadataReference(type='cpe_mac', value='44:a5:6e:45:d0:d5', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:7e:91:fa'}),
                '54:65:de:6f:6a:52': NetworkMetadataReference(type='cpe_mac', value='54:65:de:6f:6a:52', attributes={'ap': 'TestNE', 'sm': '54:65:de:6f:6a:52'}),
                '54:65:de:6f:c4:d2': NetworkMetadataReference(type='cpe_mac', value='54:65:de:6f:c4:d2', attributes={'ap': 'TestNE', 'sm': '54:65:de:6f:c4:d2'}),
                '54:65:de:bd:4e:62': NetworkMetadataReference(type='cpe_mac', value='54:65:de:bd:4e:62', attributes={'ap': 'TestNE', 'sm': '54:65:de:bd:4e:62'}),
                '54:65:de:fe:9d:22': NetworkMetadataReference(type='cpe_mac', value='54:65:de:fe:9d:22', attributes={'ap': 'TestNE', 'sm': '54:65:de:fe:9d:22'}),
                '5c:8f:e0:1a:e7:82': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:1a:e7:82', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:1a:e7:82'}),
                '5c:8f:e0:1b:06:7a': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:1b:06:7a', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:1b:06:7a'}),
                '5c:8f:e0:1b:14:a2': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:1b:14:a2', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:1b:14:a2'}),
                '5c:8f:e0:1b:a7:b2': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:1b:a7:b2', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:1b:a7:b2'}),
                '5c:8f:e0:1b:c1:5a': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:1b:c1:5a', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:1b:c1:5a'}),
                '5c:8f:e0:27:ed:12': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:27:ed:12', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:27:ed:12'}),
                '5c:8f:e0:28:69:92': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:28:69:92', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:28:69:92'}),
                '5c:8f:e0:28:74:9a': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:28:74:9a', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:28:74:9a'}),
                '5c:8f:e0:4f:ac:1a': NetworkMetadataReference(type='cpe_mac', value='5c:8f:e0:4f:ac:1a', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:4f:ac:1a'}),
                '60:19:71:ee:44:72': NetworkMetadataReference(type='cpe_mac', value='60:19:71:ee:44:72', attributes={'ap': 'TestNE', 'sm': '60:19:71:ee:44:72'}),
                '60:32:b1:3e:0a:78': NetworkMetadataReference(type='cpe_mac', value='60:32:b1:3e:0a:78', attributes={'ap': 'TestNE', 'sm': '78:71:9c:8b:f9:b2'}),
                '60:38:e0:33:04:34': NetworkMetadataReference(type='cpe_mac', value='60:38:e0:33:04:34', attributes={'ap': 'TestNE', 'sm': '8c:09:f4:04:06:62'}),
                '6c:ae:f6:4f:03:20': NetworkMetadataReference(type='cpe_mac', value='6c:ae:f6:4f:03:20', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:ac:c4:c2'}),
                '6c:cd:d6:ae:51:00': NetworkMetadataReference(type='cpe_mac', value='6c:cd:d6:ae:51:00', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:76:94:4a'}),
                '70:8b:cd:ac:dc:80': NetworkMetadataReference(type='cpe_mac', value='70:8b:cd:ac:dc:80', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:0e:bb:02'}),
                '78:71:9c:8b:f9:b2': NetworkMetadataReference(type='cpe_mac', value='78:71:9c:8b:f9:b2', attributes={'ap': 'TestNE', 'sm': '78:71:9c:8b:f9:b2'}),
                '78:71:9c:8d:0f:82': NetworkMetadataReference(type='cpe_mac', value='78:71:9c:8d:0f:82', attributes={'ap': 'TestNE', 'sm': '78:71:9c:8d:0f:82'}),
                '78:71:9c:90:f2:b2': NetworkMetadataReference(type='cpe_mac', value='78:71:9c:90:f2:b2', attributes={'ap': 'TestNE', 'sm': '78:71:9c:90:f2:b2'}),
                '78:71:9c:94:20:72': NetworkMetadataReference(type='cpe_mac', value='78:71:9c:94:20:72', attributes={'ap': 'TestNE', 'sm': '78:71:9c:94:20:72'}),
                '78:71:9c:9b:e1:12': NetworkMetadataReference(type='cpe_mac', value='78:71:9c:9b:e1:12', attributes={'ap': 'TestNE', 'sm': '78:71:9c:9b:e1:12'}),
                '78:71:9c:a1:7d:c2': NetworkMetadataReference(type='cpe_mac', value='78:71:9c:a1:7d:c2', attributes={'ap': 'TestNE', 'sm': '78:71:9c:a1:7d:c2'}),
                '78:d2:94:b0:d4:ff': NetworkMetadataReference(type='cpe_mac', value='78:d2:94:b0:d4:ff', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:a1:a9:e2'}),
                '8c:09:f4:04:06:62': NetworkMetadataReference(type='cpe_mac', value='8c:09:f4:04:06:62', attributes={'ap': 'TestNE', 'sm': '8c:09:f4:04:06:62'}),
                '8c:09:f4:10:7b:82': NetworkMetadataReference(type='cpe_mac', value='8c:09:f4:10:7b:82', attributes={'ap': 'TestNE', 'sm': '8c:09:f4:10:7b:82'}),
                '90:1a:ca:5e:14:c2': NetworkMetadataReference(type='cpe_mac', value='90:1a:ca:5e:14:c2', attributes={'ap': 'TestNE', 'sm': '90:1a:ca:5e:14:c2'}),
                '90:1a:ca:5e:3a:82': NetworkMetadataReference(type='cpe_mac', value='90:1a:ca:5e:3a:82', attributes={'ap': 'TestNE', 'sm': '90:1a:ca:5e:3a:82'}),
                '90:1a:ca:61:ca:52': NetworkMetadataReference(type='cpe_mac', value='90:1a:ca:61:ca:52', attributes={'ap': 'TestNE', 'sm': '90:1a:ca:61:ca:52'}),
                '90:1a:ca:71:7f:52': NetworkMetadataReference(type='cpe_mac', value='90:1a:ca:71:7f:52', attributes={'ap': 'TestNE', 'sm': '90:1a:ca:71:7f:52'}),
                '90:1a:ca:7b:69:92': NetworkMetadataReference(type='cpe_mac', value='90:1a:ca:7b:69:92', attributes={'ap': 'TestNE', 'sm': '90:1a:ca:7b:69:92'}),
                '90:1a:ca:7c:2b:62': NetworkMetadataReference(type='cpe_mac', value='90:1a:ca:7c:2b:62', attributes={'ap': 'TestNE', 'sm': '90:1a:ca:7c:2b:62'}),
                '90:1a:ca:d9:7a:d2': NetworkMetadataReference(type='cpe_mac', value='90:1a:ca:d9:7a:d2', attributes={'ap': 'TestNE', 'sm': '90:1a:ca:d9:7a:d2'}),
                '90:1a:ca:db:eb:f2': NetworkMetadataReference(type='cpe_mac', value='90:1a:ca:db:eb:f2', attributes={'ap': 'TestNE', 'sm': '90:1a:ca:db:eb:f2'}),
                '90:9a:4a:f2:7e:17': NetworkMetadataReference(type='cpe_mac', value='90:9a:4a:f2:7e:17', attributes={'ap': 'TestNE', 'sm': '3c:7a:8a:78:0d:3a'}),
                '94:87:7c:00:27:12': NetworkMetadataReference(type='cpe_mac', value='94:87:7c:00:27:12', attributes={'ap': 'TestNE', 'sm': '94:87:7c:00:27:12'}),
                '94:87:7c:01:e3:a2': NetworkMetadataReference(type='cpe_mac', value='94:87:7c:01:e3:a2', attributes={'ap': 'TestNE', 'sm': '94:87:7c:01:e3:a2'}),
                '94:87:7c:06:b4:02': NetworkMetadataReference(type='cpe_mac', value='94:87:7c:06:b4:02', attributes={'ap': 'TestNE', 'sm': '94:87:7c:06:b4:02'}),
                '94:87:7c:0a:91:22': NetworkMetadataReference(type='cpe_mac', value='94:87:7c:0a:91:22', attributes={'ap': 'TestNE', 'sm': '94:87:7c:0a:91:22'}),
                '94:87:7c:0b:19:02': NetworkMetadataReference(type='cpe_mac', value='94:87:7c:0b:19:02', attributes={'ap': 'TestNE', 'sm': '94:87:7c:0b:19:02'}),
                '94:87:7c:10:62:22': NetworkMetadataReference(type='cpe_mac', value='94:87:7c:10:62:22', attributes={'ap': 'TestNE', 'sm': '94:87:7c:10:62:22'}),
                '94:a6:7e:75:65:68': NetworkMetadataReference(type='cpe_mac', value='94:a6:7e:75:65:68', attributes={'ap': 'TestNE', 'sm': '5c:8f:e0:1b:06:7a'}),
                '98:6b:3d:c3:17:e2': NetworkMetadataReference(type='cpe_mac', value='98:6b:3d:c3:17:e2', attributes={'ap': 'TestNE', 'sm': '98:6b:3d:c3:17:e2'}),
                '98:6b:3d:c5:50:42': NetworkMetadataReference(type='cpe_mac', value='98:6b:3d:c5:50:42', attributes={'ap': 'TestNE', 'sm': '98:6b:3d:c5:50:42'}),
                '98:6b:3d:c6:d7:92': NetworkMetadataReference(type='cpe_mac', value='98:6b:3d:c6:d7:92', attributes={'ap': 'TestNE', 'sm': '98:6b:3d:c6:d7:92'}),
                '98:6b:3d:c7:56:62': NetworkMetadataReference(type='cpe_mac', value='98:6b:3d:c7:56:62', attributes={'ap': 'TestNE', 'sm': '98:6b:3d:c7:56:62'}),
                '98:6b:3d:d2:0d:b2': NetworkMetadataReference(type='cpe_mac', value='98:6b:3d:d2:0d:b2', attributes={'ap': 'TestNE', 'sm': '98:6b:3d:d2:0d:b2'}),
                '98:6b:3d:dc:60:42': NetworkMetadataReference(type='cpe_mac', value='98:6b:3d:dc:60:42', attributes={'ap': 'TestNE', 'sm': '98:6b:3d:dc:60:42'}),
                '98:6b:3d:dc:a2:82': NetworkMetadataReference(type='cpe_mac', value='98:6b:3d:dc:a2:82', attributes={'ap': 'TestNE', 'sm': '98:6b:3d:dc:a2:82'}),
                '98:6b:3d:e5:51:e2': NetworkMetadataReference(type='cpe_mac', value='98:6b:3d:e5:51:e2', attributes={'ap': 'TestNE', 'sm': '98:6b:3d:e5:51:e2'}),
                '9c:34:26:44:02:3a': NetworkMetadataReference(type='cpe_mac', value='9c:34:26:44:02:3a', attributes={'ap': 'TestNE', 'sm': '9c:34:26:44:02:3a'}),
                '9c:34:26:4e:92:5a': NetworkMetadataReference(type='cpe_mac', value='9c:34:26:4e:92:5a', attributes={'ap': 'TestNE', 'sm': '9c:34:26:4e:92:5a'}),
                '9c:34:26:ac:47:5a': NetworkMetadataReference(type='cpe_mac', value='9c:34:26:ac:47:5a', attributes={'ap': 'TestNE', 'sm': '9c:34:26:ac:47:5a'}),
                '9c:34:26:b0:be:6a': NetworkMetadataReference(type='cpe_mac', value='9c:34:26:b0:be:6a', attributes={'ap': 'TestNE', 'sm': '9c:34:26:b0:be:6a'}),
                '9c:34:26:b0:f7:d2': NetworkMetadataReference(type='cpe_mac', value='9c:34:26:b0:f7:d2', attributes={'ap': 'TestNE', 'sm': '9c:34:26:b0:f7:d2'}),
                '9c:c9:eb:32:e0:f9': NetworkMetadataReference(type='cpe_mac', value='9c:c9:eb:32:e0:f9', attributes={'ap': 'TestNE', 'sm': '38:70:0c:3e:cc:3a'}),
                'ac:b3:13:04:d1:12': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:04:d1:12', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:04:d1:12'}),
                'ac:b3:13:0c:b0:52': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:0c:b0:52', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:0c:b0:52'}),
                'ac:b3:13:0e:bb:02': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:0e:bb:02', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:0e:bb:02'}),
                'ac:b3:13:77:26:e2': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:77:26:e2', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:77:26:e2'}),
                'ac:b3:13:78:36:72': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:78:36:72', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:78:36:72'}),
                'ac:b3:13:80:8a:02': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:80:8a:02', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:80:8a:02'}),
                'ac:b3:13:85:c2:72': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:85:c2:72', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:85:c2:72'}),
                'ac:b3:13:8f:85:32': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:8f:85:32', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:8f:85:32'}),
                'ac:b3:13:9f:dc:c2': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:9f:dc:c2', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:9f:dc:c2'}),
                'ac:b3:13:a1:a9:e2': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:a1:a9:e2', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:a1:a9:e2'}),
                'ac:b3:13:a2:05:12': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:a2:05:12', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:a2:05:12'}),
                'ac:b3:13:a4:98:02': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:a4:98:02', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:a4:98:02'}),
                'ac:b3:13:a9:70:02': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:a9:70:02', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:a9:70:02'}),
                'ac:b3:13:ac:c4:c2': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:ac:c4:c2', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:ac:c4:c2'}),
                'ac:b3:13:b2:b4:52': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:b2:b4:52', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:b2:b4:52'}),
                'ac:b3:13:b5:5b:72': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:b5:5b:72', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:b5:5b:72'}),
                'ac:b3:13:b6:cb:92': NetworkMetadataReference(type='cpe_mac', value='ac:b3:13:b6:cb:92', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:b6:cb:92'}),
                'b0:39:56:6b:c8:09': NetworkMetadataReference(type='cpe_mac', value='b0:39:56:6b:c8:09', attributes={'ap': 'TestNE', 'sm': '40:70:09:71:a2:32'}),
                'b0:b9:8a:51:1f:1a': NetworkMetadataReference(type='cpe_mac', value='b0:b9:8a:51:1f:1a', attributes={'ap': 'TestNE', 'sm': 'ac:b3:13:85:c2:72'}),
                'c0:c1:c0:fb:af:a1': NetworkMetadataReference(type='cpe_mac', value='c0:c1:c0:fb:af:a1', attributes={'ap': 'TestNE', 'sm': '40:70:09:8a:27:92'}),
                'c0:c5:22:e1:67:3a': NetworkMetadataReference(type='cpe_mac', value='c0:c5:22:e1:67:3a', attributes={'ap': 'TestNE', 'sm': 'c0:c5:22:e1:67:3a'}),
                'c0:c5:22:f9:75:a4': NetworkMetadataReference(type='cpe_mac', value='c0:c5:22:f9:75:a4', attributes={'ap': 'TestNE', 'sm': 'c0:c5:22:f9:75:a4'}),
                'cc:a4:62:26:71:92': NetworkMetadataReference(type='cpe_mac', value='cc:a4:62:26:71:92', attributes={'ap': 'TestNE', 'sm': 'cc:a4:62:26:71:92'}),
                'cc:a4:62:37:0c:12': NetworkMetadataReference(type='cpe_mac', value='cc:a4:62:37:0c:12', attributes={'ap': 'TestNE', 'sm': 'cc:a4:62:37:0c:12'}),
                'dc:ef:09:17:05:e9': NetworkMetadataReference(type='cpe_mac', value='dc:ef:09:17:05:e9', attributes={'ap': 'TestNE', 'sm': 'cc:a4:62:37:0c:12'}),
                'e4:90:7e:f6:83:a0': NetworkMetadataReference(type='cpe_mac', value='e4:90:7e:f6:83:a0', attributes={'ap': 'TestNE', 'sm': '38:70:0c:02:e8:b8'}),
                'e8:89:2c:7e:83:e2': NetworkMetadataReference(type='cpe_mac', value='e8:89:2c:7e:83:e2', attributes={'ap': 'TestNE', 'sm': 'e8:89:2c:7e:83:e2'}),
                'f0:9f:c2:36:00:dc': NetworkMetadataReference(type='cpe_mac', value='f0:9f:c2:36:00:dc', attributes={'ap': 'TestNE', 'sm': '40:70:09:8c:07:a2'}),
                'f8:ed:a5:db:b3:f2': NetworkMetadataReference(type='cpe_mac', value='f8:ed:a5:db:b3:f2', attributes={'ap': 'TestNE', 'sm': 'f8:ed:a5:db:b3:f2'}),
                'f8:ed:a5:dc:5e:e2': NetworkMetadataReference(type='cpe_mac', value='f8:ed:a5:dc:5e:e2', attributes={'ap': 'TestNE', 'sm': 'f8:ed:a5:dc:5e:e2'}),
                'f8:ed:a5:df:27:12': NetworkMetadataReference(type='cpe_mac', value='f8:ed:a5:df:27:12', attributes={'ap': 'TestNE', 'sm': 'f8:ed:a5:df:27:12'}),
                'f8:ed:a5:e8:1a:d2': NetworkMetadataReference(type='cpe_mac', value='f8:ed:a5:e8:1a:d2', attributes={'ap': 'TestNE', 'sm': 'f8:ed:a5:e8:1a:d2'}),
                'f8:ed:a5:f2:aa:b2': NetworkMetadataReference(type='cpe_mac', value='f8:ed:a5:f2:aa:b2', attributes={'ap': 'TestNE', 'sm': 'f8:ed:a5:f2:aa:b2'})
        }
        self.assertEqual(snmp_session_refs, self.ctx.netmeta_model.refs.get('snmp_session'))
        self.assertEqual(cpe_mac_refs, self.ctx.netmeta_model.refs.get('cpe_mac'))

    def test_poll_stations_01(self):
        """Test that poll_station is called with a http client that is looked
           up from the MAC address."""
        polled_station = False
        def poll_station(sta, snmp_client, http_client):
            nonlocal polled_station
            polled_station = True
            self.assertEqual(sta.mac_address, '01:00:00:00:00:01')
            self.assertIsNotNone(http_client)
            self.assertEqual(http_client._url, 'https://192.0.2.100')
            return sta
        ap1 = Ap(self.ctx, 'TestAP1', 'TestAP1', 'MySite', FakeClient('MyHost1'), None, 'ap', None)
        Sta = namedtuple('Sta', ('mac_address',))
        ap1.module = FakeModule()
        self._await(self.ctx.netmeta_model.set_ref(NetworkMetadataReference('snmp_session', '192.0.2.100', {'dev_mac': Reference('01:00:00:00:00:01')})))
        ap1.module.station_poller = poll_station
        ap1.module.stations = [Sta('01:00:00:00:00:01')]
        self._await(ap1.load())
        self.assertTrue(polled_station)

    def test_linkrate_calc1(self):
        """Test the creation of network_element_update_msg and correctness of radio level linkrate calculations in case of pre-existing SM level link_rates (tx direction only exist in stations) """
        df = self.datafiles.get(('cambium-canopy.450.PMP 450i.14.3.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        ap.in_out_bytes = {}
        aggr_rx_linkrate = 0
        aggr_tx_linkrate = 0

        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        ne_u.data.uptime += 60
        for n in range(len(ne_u.data.interfaces[1].links)):
            ne_u.data.interfaces[1].links[n].in_octets += (n+1)*2000000
            ne_u.data.interfaces[1].links[n].out_octets += (n+1)*1000000

        self._await(ap.log_ne_update())
        # set radio link_rates to None
        ne_u.data.interfaces[1].radios[0].rx_link_rate = None
        ne_u.data.interfaces[1].radios[0].tx_link_rate = None

        ap.calc_radio_link_rate(ne_u)
        links_len = len(ne_u.data.interfaces[1].radios[0].streams[0].links)
        total_in_octets_interval = 12000000
        total_out_octets_interval = 6000000

        for n in range(links_len):
            sm_rx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].rx_link_rate
            sm_tx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].tx_link_rate
            sm_in_octets_interval = (n+1)*2000000
            sm_out_octets_interval = (n+1)*1000000
            self.assertIsNone(sm_rx_linkrate) #rx linkrate in links should be None in this snmp dump
            self.assertIsNotNone(sm_tx_linkrate)
            aggr_rx_linkrate += sm_rx_linkrate * sm_in_octets_interval/total_in_octets_interval if sm_rx_linkrate and sm_in_octets_interval and total_in_octets_interval else 0
            aggr_tx_linkrate += sm_tx_linkrate * sm_out_octets_interval/total_out_octets_interval if sm_tx_linkrate and sm_out_octets_interval and total_out_octets_interval else 0

        self.assertIsNone(ne_u.data.interfaces[1].radios[0].rx_link_rate)
        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].tx_link_rate, 5),  round(int(aggr_tx_linkrate),5))

    def test_linkrate_calc2(self):
        """Test the creation of network_element_update_msg and correctness of radio level linkrate calculations in case of pre-existing SM level link_rates (both directions exist in stations) """
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP Force 180.4.4.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        ap.in_out_bytes = {}
        aggr_rx_linkrate = 0
        aggr_tx_linkrate = 0

        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        ne_u.data.uptime += 60
        for n in range(len(ne_u.data.interfaces[1].links)):
            ne_u.data.interfaces[1].links[n].in_octets += (n+1)*2000000
            ne_u.data.interfaces[1].links[n].out_octets += (n+1)*1000000

        self._await(ap.log_ne_update())
        # set radio link_rates to None
        ne_u.data.interfaces[1].radios[0].rx_link_rate = None
        ne_u.data.interfaces[1].radios[0].tx_link_rate = None

        ap.calc_radio_link_rate(ne_u)
        links_len = len(ne_u.data.interfaces[1].radios[0].streams[0].links)
        total_in_octets_interval = 6000000
        total_out_octets_interval = 3000000

        for n in range(links_len):
            sm_rx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].rx_link_rate
            sm_tx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].tx_link_rate
            sm_in_octets_interval = (n+1)*2000000
            sm_out_octets_interval = (n+1)*1000000
            self.assertIsNotNone(sm_rx_linkrate)
            self.assertIsNotNone(sm_tx_linkrate)
            aggr_rx_linkrate += sm_rx_linkrate * sm_in_octets_interval/total_in_octets_interval if sm_rx_linkrate and sm_in_octets_interval and total_in_octets_interval else 0
            aggr_tx_linkrate += sm_tx_linkrate * sm_out_octets_interval/total_out_octets_interval if sm_tx_linkrate and sm_out_octets_interval and total_out_octets_interval else 0

        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].rx_link_rate, 5), round(int(aggr_rx_linkrate),5))
        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].tx_link_rate, 5),  round(int(aggr_tx_linkrate),5))

    def test_linkrate_calc3(self):
        """Test the creation of network_element_update_msg and correctness of regular average radio level linkrate calculations for both ways in case of pre-existing SM level link_rates (both directions exist in stations) """
        df = self.datafiles.get(('mikrotik.mikrotik-ap.RB SXT r2 5nD.6.43.8.185_140_95_130', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        ap.in_out_bytes = {}
        aggr_rx_linkrate = 0
        aggr_tx_linkrate = 0

        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        ne_u.data.uptime += 60

        self._await(ap.log_ne_update())
        # set radio link_rates to None
        ne_u.data.interfaces[1].radios[0].rx_link_rate = None
        ne_u.data.interfaces[1].radios[0].tx_link_rate = None

        ap.calc_radio_link_rate(ne_u)
        links_len = len(ne_u.data.interfaces[1].radios[0].streams[0].links)

        for n in range(links_len):
            sm_rx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].rx_link_rate
            sm_tx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].tx_link_rate
            self.assertIsNotNone(sm_rx_linkrate)
            self.assertIsNotNone(sm_tx_linkrate)
            aggr_rx_linkrate += sm_rx_linkrate
            aggr_tx_linkrate += sm_tx_linkrate
        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].rx_link_rate, 5), round(int(aggr_rx_linkrate/links_len),5))
        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].tx_link_rate, 5),  round(int(aggr_tx_linkrate/links_len),5))

    def test_linkrate_calc4(self):
        """Test the creation of network_element_update_msg and correctness of aggregated weighted (tx_linkrate) radio level and regular average (rx_linkrate) calculations. In case of pre-existing SM level link_rates (both directions exist in stations) """
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP Force 180.4.4.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        ap.in_out_bytes = {}
        aggr_rx_linkrate = 0
        aggr_tx_linkrate = 0

        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        ne_u.data.uptime += 60
        for n in range(len(ne_u.data.interfaces[1].links)):
            ne_u.data.interfaces[1].links[n].in_octet = 0
            ne_u.data.interfaces[1].links[n].out_octets += (n+1)*1000000

        self._await(ap.log_ne_update())
        # set radio link_rates to None
        ne_u.data.interfaces[1].radios[0].rx_link_rate = None
        ne_u.data.interfaces[1].radios[0].tx_link_rate = None

        ap.calc_radio_link_rate(ne_u)
        links_len = len(ne_u.data.interfaces[1].radios[0].streams[0].links)
        total_out_octets_interval = 3000000

        for n in range(links_len):
            sm_rx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].rx_link_rate
            sm_tx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].tx_link_rate
            sm_out_octets_interval = (n+1)*1000000
            self.assertIsNotNone(sm_rx_linkrate)
            self.assertIsNotNone(sm_tx_linkrate)
            aggr_rx_linkrate += sm_rx_linkrate
            aggr_tx_linkrate += sm_tx_linkrate * sm_out_octets_interval/total_out_octets_interval if sm_tx_linkrate and sm_out_octets_interval and total_out_octets_interval else 0

        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].rx_link_rate, 5), round(int(aggr_rx_linkrate/links_len),5))
        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].tx_link_rate, 5),  round(int(aggr_tx_linkrate),5))

    def test_linkrate_calc5(self):
        """Test the creation of network_element_update_msg and correctness of aggregated weighted (rx_linkrate) radio level and regular average (tx_linkrate) calculations. In case of pre-existing SM level link_rates (both directions exist in stations) """
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP Force 180.4.4.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        ap.in_out_bytes = {}
        aggr_rx_linkrate = 0
        aggr_tx_linkrate = 0

        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        ne_u.data.uptime += 60
        for n in range(len(ne_u.data.interfaces[1].links)):
            ne_u.data.interfaces[1].links[n].in_octets += (n+1)*2000000
            ne_u.data.interfaces[1].links[n].out_octets = 0

        self._await(ap.log_ne_update())
        # set radio link_rates to None
        ne_u.data.interfaces[1].radios[0].rx_link_rate = None
        ne_u.data.interfaces[1].radios[0].tx_link_rate = None

        ap.calc_radio_link_rate(ne_u)
        links_len = len(ne_u.data.interfaces[1].radios[0].streams[0].links)
        total_in_octets_interval = 6000000

        for n in range(links_len):
            sm_rx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].rx_link_rate
            sm_tx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].tx_link_rate
            sm_in_octets_interval = (n+1)*2000000
            self.assertIsNotNone(sm_rx_linkrate)
            self.assertIsNotNone(sm_tx_linkrate)
            aggr_rx_linkrate += sm_rx_linkrate * sm_in_octets_interval/total_in_octets_interval if sm_rx_linkrate and sm_in_octets_interval and total_in_octets_interval else 0
            aggr_tx_linkrate += sm_tx_linkrate

        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].rx_link_rate, 5), round(int(aggr_rx_linkrate),5))
        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].tx_link_rate, 5),  round(int(aggr_tx_linkrate/links_len),5))

    def test_set_element_status1(self):
        """Test the creation of network_element_update_msg message Status online/disconnected/offline/bad data range and propogated module errors"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.5.6.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        ap.elem_uuid = UUID('491c9831-7597-4a2e-8464-000000000000')
        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        self.assertEqual(ne_u.instance,  'unittest')
        self.assertEqual(ne_u.data.management_ip, ap.host)
        self.assertEqual(ne_u.active, ap.active)

        # test online status, no errors
        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ONLINE)
        self.assertEqual(len(ne_u.data.errors), 0)

        # test dissconnected leads to snmp_connect Error
        self._await(ap.poll())
        self.wait_for_ap_poll()
        ap.snmp_error = 'disconnected'
        self._await(ap.log_ne_update())
        ne_u = ap.offline_disconnected_update_msg
        expected_err = network_poller_pb2.Error()
        expected_err.snmp_connect.CopyFrom(network_poller_pb2.SnmpConnectError())

        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ERROR)
        lasterr = ne_u.data.errors[-1].pb
        self.assertEqual(expected_err, lasterr)

        # test offline status, no errors
        self._await(ap.poll())
        self.wait_for_ap_poll()
        ap.snmp_error = 'offline'
        self._await(ap.log_ne_update())
        ne_u= ap.offline_disconnected_update_msg
        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_OFFLINE)
        self.assertEqual(len(ne_u.data.errors), 0)

        # test data range status and error
        self._await(ap.poll())
        self.wait_for_ap_poll()
        ne_u= ap.module.network_element_update_msg
        errctx =ErrorContext()
        value = 128628618197
        errctx.add(network_poller_pb2.PollError.ERROR_BAD_DATA_RANGE,
                        'Invalid value for NetworkElement.Link.in_ucast_pkts',
                        value=value, metric_name='NetworkElement.Link.in_ucast_pkts')
        ap.set_element_status(ne_u.data, errctx)
        expected_err = network_poller_pb2.Error()
        expected_err.bad_data_range.metric_name = "NetworkElement.Link.in_ucast_pkts"
        expected_err.bad_data_range.metric_value = "128628618197"
        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ERROR)
        lasterr = ne_u.data.errors[-1].pb
        self.assertEqual(expected_err, lasterr)

        # test all the propogated errors are stayed and status is set to Error
        self._await(ap.poll())
        self.wait_for_ap_poll()
        ne_u= ap.module.network_element_update_msg
        err1 = network_poller_pb2.Error()
        err1.snmp_connect.CopyFrom(network_poller_pb2.SnmpConnectError())
        ne_u.data.errors.append(err1)
        err2 = network_poller_pb2.Error()
        err2.http_connect.CopyFrom(network_poller_pb2.HttpConnectError())
        ne_u.data.errors.append(err2)
        errctx =ErrorContext()
        ap.set_element_status(ne_u.data, errctx)

        expected_err1 = network_poller_pb2.Error()
        expected_err1.snmp_connect.CopyFrom(network_poller_pb2.SnmpConnectError())
        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ERROR)
        resulted_err1 = ne_u.data.errors[0].pb
        self.assertEqual(expected_err1, resulted_err1)

        expected_err2 = network_poller_pb2.Error()
        expected_err2.http_connect.CopyFrom(network_poller_pb2.HttpConnectError())
        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ERROR)
        resulted_err2 = ne_u.data.errors[1].pb
        self.assertEqual(expected_err2, resulted_err2)

    def test_snmp_reset1(self):
        """Test the snmp v1 reset works for snmpv1 unspupported element"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP 2000.4.5.6.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        self.assertEqual(ne.snmp_ne.device.snmp_version, 2)
        ne.snmp_ne.device.snmp_version = 1
        self._await(ap.poll())
        self.wait_for_ap_poll()
        self.assertEqual(ne.snmp_ne.device.snmp_version, 2)

    def test_snmp_reset2(self):
        """Test the snmp v1 reset doesn't work for snmpv1 supported element"""
        df = self.datafiles.get(('cambium-canopy.fsk.PMP 100 FSK.13.1.3.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(ap.poll())
        self.wait_for_ap_poll()
        self.assertEqual(ne.snmp_ne.device.snmp_version, 2)
        ne.snmp_ne.device.snmp_version = 1
        self._await(ap.poll())
        self.wait_for_ap_poll()
        self.assertEqual(ne.snmp_ne.device.snmp_version, 1)


    def test_cpe_name_mapping_1(self):
        """Test CPE name mapping."""
        self.ctx.bridge_mapping_enabled = True
        self.ctx.cpe_name_mapping_enabled = True
        ap = Ap(self.ctx, 'TestAP1', 'TestAP1', 'MySite', FakeClient('MyHost1'), None, 'ap', None)
        ap.module = FakeModule()
        Sta = namedtuple('Sta', ('mac_address', 'name', 'dev_macs'))
        ap.module.stations = [Sta('01:00:00:00:00:01', 'Subscriber 1', ['02:00:00:00:00:01', '03:00:00:00:00:01']), Sta('a1:00:00:00:00:01', 'Subscriber 2', [])]
        self._await(ap.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_mac')
        self.assertEqual(len(refs), 4)
        self.assertEqual(refs['01:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='01:00:00:00:00:01', attributes={'ap': 'TestAP1', 'sm': '01:00:00:00:00:01', 'service': 'Subscriber 1'}))
        self.assertEqual(refs['a1:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='a1:00:00:00:00:01', attributes={'ap': 'TestAP1', 'sm': 'a1:00:00:00:00:01', 'service': 'Subscriber 2'}))
        self.assertEqual(refs['02:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='02:00:00:00:00:01', attributes={'ap': 'TestAP1', 'sm': '01:00:00:00:00:01', 'service': 'Subscriber 1'}))
        self.assertEqual(refs['03:00:00:00:00:01'], NetworkMetadataReference(type='cpe_mac', value='03:00:00:00:00:01', attributes={'ap': 'TestAP1', 'sm': '01:00:00:00:00:01', 'service': 'Subscriber 1'}))
        self.assertIsInstance(refs['01:00:00:00:00:01'].attributes['service'], Reference)


    def test_disable_offline_status(self):
        """simulate various conditions for disable_offline_status config"""
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        ap.errctx = getattr(ap.module, 'errctx', ErrorContext())
        ap.network_element_update_msg = getattr(ap.module, 'network_element_update_msg', NetworkElementUpdate(errctx=ap.errctx))

        # simulate no effect if disable_offline_status flag doesn't exist; the offline message is saved
        ap.snmp_error = 'offline'
        self._await(ap.log_ne_update())
        ne_u_offline_disconnected= ap.offline_disconnected_update_msg

        self.assertEqual(ne_u_offline_disconnected.data.status, network_poller_pb2.ELEMENT_STATUS_OFFLINE)
        self.assertEqual(len(ne_u_offline_disconnected.data.errors), 0)

        # simulate no effect if disable_offline_status flag doesn't exist; the disconnected message is saved
        ap.snmp_error = 'disconnected'
        self._await(ap.log_ne_update())

        ne_u_offline_disconnected = ap.offline_disconnected_update_msg

        expected_err = network_poller_pb2.Error()
        expected_err.snmp_connect.CopyFrom(network_poller_pb2.SnmpConnectError())
        self.assertEqual(ne_u_offline_disconnected.data.status, network_poller_pb2.ELEMENT_STATUS_ERROR)
        self.assertEqual(len(ne_u_offline_disconnected.data.errors), 1)
        lasterr = ne_u_offline_disconnected.data.errors[-1].pb
        self.assertEqual(expected_err, lasterr)

        #back to no errors
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        ap.errctx = getattr(ap.module, 'errctx', ErrorContext())
        ap.network_element_update_msg = getattr(ap.module, 'network_element_update_msg', NetworkElementUpdate(errctx=ap.errctx))
        ap.snmp_error = ''
        ne_u_offline_disconnected = ap.offline_disconnected_update_msg
        self._await(ap.log_ne_update())

        #simulate no update message is sent in case of enabled flag and Element is offline
        self.ctx.disable_offline_status = True
        ap.snmp_error = 'offline'
        self._await(ap.log_ne_update())
        self.assertIsNone(ne_u_offline_disconnected)


        # simulate no effect if disable_offline_status flag is enabled; the disconnected message is saved
        ap.snmp_error = 'disconnected'
        self._await(ap.log_ne_update())

        ne_u_offline_disconnected = ap.offline_disconnected_update_msg

        expected_err = network_poller_pb2.Error()
        expected_err.snmp_connect.CopyFrom(network_poller_pb2.SnmpConnectError())
        self.assertEqual(ne_u_offline_disconnected.data.status, network_poller_pb2.ELEMENT_STATUS_ERROR)
        self.assertEqual(len(ne_u_offline_disconnected.data.errors), 1)
        lasterr = ne_u_offline_disconnected.data.errors[-1].pb
        self.assertEqual(expected_err, lasterr)

    def test_offline_neu_msg(self):
        """Verify that empty NEU msg is generated when AP starts in offline state."""
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        self._await(ap.log_ne_update())
        self.assertIsNotNone(ap.network_element_update_msg)
        self.assertIsInstance(ap.network_element_update_msg, NetworkElementUpdate)

    def test_fwd_ap_errors(self):
        """Verify that AP Errors are indicated on CPEs with a special error: PeerElementInError"""
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        ap.elem_uuid = UUID('491c9831-7597-4a2e-8464-000000000000')
        ap.errctx = getattr(ap.module, 'errctx', ErrorContext())
        ap.network_element_update_msg = getattr(ap.module, 'network_element_update_msg', NetworkElementUpdate(errctx=ap.errctx))
        ap.network_element_update_msg.data.management_ip = '192.0.2.1'
        ap.network_element_update_msg.data.name = 'abra-RP- 60'
        ap.network_element_update_msg.data.manufacturer = 'Ubiquiti'
        ap.network_element_update_msg.data.model = 'Rocket Prism 5AC Gen2'
        ap.network_element_update_msg.data.poller_hash = b'\x01'
        err = network_poller_pb2.Error()
        err.http_connect.CopyFrom(network_poller_pb2.HttpConnectError())
        ap.network_element_update_msg.data.errors.append(err)
        peer = ap.network_element_update_msg.peers.add()
        peer.management_ip = '10.3.3.57'
        peer.manufacturer = 'Ubiquiti'
        peer.name = '19358-3003-UPB5AC SM'
        peer.status = network_poller_pb2.ELEMENT_STATUS_ONLINE
        peer.system_mac_address = 'f4:92:bf:bc:48:de'
        ap.set_element_status(ap.network_element_update_msg.data, errctx=ap.errctx)
        r_peer = ap.network_element_update_msg.peers[0]
        self.assertEqual(r_peer.status, network_poller_pb2.ELEMENT_STATUS_ERROR)
        r_err = r_peer.errors[0]
        self.assertEqual(r_err.pb.WhichOneof('kind'), 'peer_in_error')
        self.assertEqual(r_err.pb.peer_in_error.source_uuid, ap.elem_uuid.bytes)

    def test_serial_mapping(self):
        """Test that cpe_serial gets mapped from a CPE with serial number."""
        ap = Ap(self.ctx, 'TestAP1', 'TestAP1', 'MySite', None, None, 'ap',
                None)
        ap.module = FakeModule()
        Sta = namedtuple('Sta', ('serial', ))
        stations = [Sta('S1234567890'), Sta('S4444333322')]
        ap.module.stations = stations
        self._await(ap.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_serial') or {}
        self.assertEqual(len(refs), 2)
        ref = refs.get('S1234567890')
        self.assertEqual(
            ref,
            NetworkMetadataReference('cpe_serial', 'S1234567890',
                                     {'ap': 'TestAP1'}))
        ref = refs.get('S4444333322')
        self.assertEqual(
            ref,
            NetworkMetadataReference('cpe_serial', 'S4444333322',
                                     {'ap': 'TestAP1'}))
        # station disconnected from AP, mapping removed
        ap.module.stations = [Sta('S4444333322')]
        self._await(ap.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_serial') or {}
        self.assertEqual(len(refs), 1)
        # last module disconnected
        ap.module.stations = []
        self._await(ap.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_serial') or {}
        self.assertEqual(len(refs), 0)
        # station reconnected, mapping added
        ap.module.stations = [Sta('S1234567890')]
        self._await(ap.load())
        refs = self.ctx.netmeta_model.refs.get('cpe_serial') or {}
        self.assertEqual(len(refs), 1)
        # ap goes offline, mapping removed
        ap.offline = True
        self._await(ap.remove_ref_offline())
        refs = self.ctx.netmeta_model.refs.get('cpe_serial') or {}
        self.assertEqual(len(refs), 0)

    def test_ssl_module(self):
        """Test matching based on SSL certificate data."""
        snmp_ne = FakeDevice(None)
        async def fake_ssl_probe(host, port):
            return SslProbeResult(error=None, org='Tarana Wireless', cn='S134F2215100421.radio.taranawireless.com')
        self.ctx.ssl_probe = fake_ssl_probe
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', snmp_ne, None, 'ap', None)
        try:
            self._await(ap.poll())
        except:
            pass  # the poll won't work but we are just checking the match here
        self.assertIsNotNone(ap.module)
        self.assertEqual(type(ap.module).__name__, 'TaranaAP')


    def test_peer_errors(self):
        """Test that STATUS and errors remained unchanged in peers."""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        msg = ap.module.network_element_update_msg = network_element_update.NetworkElementUpdate()
        msg.poller_hash = b'serAP1'
        msg.data.poller_hash = b'serAP'
        msg.data.status = network_poller_pb2.ELEMENT_STATUS_ONLINE

        intf = msg.data.interfaces.add()
        intf.poller_hash = b'ser1wl1'
        radio = intf.radios.add()
        radio.id = '1'
        stream = radio.streams.add()

        peer = msg.peers.add()
        peer.poller_hash = b'serCPE1'
        # interface with poller_hash
        peer_intf = peer.interfaces.add()
        peer_intf.id = 'wl1'
        peer_intf.poller_hash = b'serCPE1wl1'
        peer.status = network_poller_pb2.ELEMENT_STATUS_ONLINE

        p1link = peer_intf.links.add()
        p1link.poller_hash = intf.poller_hash
        p1r = peer_intf.radios.add()
        p1s = p1r.streams.add()
        p1wlink = p1s.links.add()
        p1wlink.poller_hash = intf.poller_hash

        peer2 = msg.peers.add()
        peer2.poller_hash = b'serCPE2'
        # interface with poller_hash
        peer2_intf = peer2.interfaces.add()
        peer2_intf.id = 'wl1'
        peer2_intf.poller_hash =  b'serCPE2wl1'
        peer2.status = network_poller_pb2.ELEMENT_STATUS_ONLINE
        p2link = peer2_intf.links.add()
        p2link.poller_hash = intf.poller_hash
        p2r = peer2_intf.radios.add()
        p2s = p2r.streams.add()
        p2wlink = p2s.links.add()
        p2wlink.poller_hash = intf.poller_hash


        link1 = intf.links.add()
        wlink1 = stream.links.add()
        link1.poller_hash = peer_intf.poller_hash
        wlink1.poller_hash = peer_intf.poller_hash

        link2 = intf.links.add()
        wlink2 = stream.links.add()
        link2.poller_hash = peer2_intf.poller_hash
        wlink2.poller_hash = peer2_intf.poller_hash


        # some other peer try to set dict with None key (which is actually impossible since we're preventing None keys entry now)
        other_peer_err = network_poller_pb2.Error()
        other_peer_err.snmp_connect.CopyFrom(network_poller_pb2.SnmpConnectError())
        ap.peer_errors[None] = other_peer_err

        self._await(ap.load())
        ne_u = ap.module.network_element_update_msg


        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ONLINE)
        self.assertEqual(len(ne_u.data.errors), 0)

        self.assertEqual(ne_u.peers[0].status, network_poller_pb2.ELEMENT_STATUS_ONLINE)
        self.assertEqual(len(ne_u.peers[0].errors), 0)

        self.assertEqual(ne_u.peers[1].status, network_poller_pb2.ELEMENT_STATUS_ONLINE)
        self.assertEqual(len(ne_u.peers[1].errors), 0)

    def test_set_element_status_ap_peer_in_error_GE1001_no_popogation(self):
        """Test the creation of network_element_update_msg message and GE1001 error is not popogated"""
        df = self.datafiles.get(('cambium-canopy.450.PMP 450.15.1.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE1', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(asyncio.sleep(2))
        ap.in_out_bytes = {}
        ap.elem_uuid = UUID('491c9831-7597-4a2e-8464-000000000002')
        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        self.assertEqual(ne_u.instance,  'unittest')
        self.assertEqual(ne_u.data.management_ip, ap.host)

        # check status of AP is ERROR because some/all peers have error (GE1001)
        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ONLINE)

        peers_phs = []

        for p in ne_u.peers:
            peers_phs.append(p.pb.poller_hash)
            # verify all peers has GE1001 type. (pre-set in module)
            self.assertEqual(p.pb.errors[0].error.code, 'GE1001')
            # verify no more errors
            self.assertEqual(len(p.pb.errors), 1)

    def test_set_element_status_ap_peer_in_error_GE1001_and_peer_in_error(self):
        """Test the creation of network_element_update_msg message Status error from propogated CPE's GE1001 error"""
        df = self.datafiles.get(('cambium-canopy.450.PMP 450i.CANOPY 15.0.3 AP-None.192_168_56_2', 'snmp'))
        ne = self._await(self.reg.set('TestNE2', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        self._await(asyncio.sleep(2))
        ap.in_out_bytes = {}
        ap.elem_uuid = UUID('491c9831-7597-4a2e-8464-000000000001')
        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        self.assertEqual(ne_u.instance,  'unittest')
        self.assertEqual(ne_u.data.management_ip, ap.host)

        # check status of AP is ERROR because some/all peers have error (GE1001)
        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ERROR)

        peers_phs = []

        for p in ne_u.peers:
            peers_phs.append(p.pb.poller_hash)
            # verify all peers has GE1001 type. (pre-set in module)
            self.assertEqual(p.pb.errors[0].error.code, 'GE1001')
            # check all peers get AP's peer_in_error type appended as last error.
            self.assertEqual(p.pb.errors[-1].WhichOneof('kind'), 'peer_in_error')
            # check all peers peer_in_error have Element' UUID set as source_uuid .
            self.assertEqual(p.pb.errors[-1].peer_in_error.source_uuid, ap.elem_uuid.bytes)


        # check all errors in AP errors are peer_in_error type with poller_hash set from peer's poller_hash
        for i, err in enumerate(ne_u.data.errors):
            self.assertEqual(err.pb.WhichOneof('kind'), 'peer_in_error')
            self.assertEqual(err.pb.peer_in_error.poller_hash, peers_phs[i])


    def test_linkrate_calc6(self):
        """ test discarding link level link_rate value"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP Force 180.4.4.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        ap.in_out_bytes = {}
        aggr_rx_linkrate = 0
        aggr_tx_linkrate = 0

        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        ne_u.data.uptime += 60
        for n in range(len(ne_u.data.interfaces[1].links)):
            ne_u.data.interfaces[1].links[n].in_octets += (n+1)*2000000
            ne_u.data.interfaces[1].links[n].out_octets += (n+1)*1000000
        links_len = len(ne_u.data.interfaces[1].radios[0].streams[0].links)
        # set one of link-rates high that ratio link_rate/chan_width will be more than 101
        for n in range(links_len):
            if n==0:
                ne_u.data.interfaces[1].radios[0].streams[0].links[n].rx_link_rate = 1010000000
                ne_u.peers[n].interfaces[0].radios[0].streams[0].links[0].tx_link_rate = 1010000000
            if n==1:
                ne_u.data.interfaces[1].radios[0].streams[0].links[n].tx_link_rate = 1020000000
                ne_u.peers[n].interfaces[0].radios[0].streams[0].links[0].rx_link_rate = 1020000000

        self._await(ap.log_ne_update())
        # set radio link_rates to None
        ne_u.data.interfaces[1].radios[0].rx_link_rate = None
        ne_u.data.interfaces[1].radios[0].tx_link_rate = None

        ap.calc_radio_link_rate(ne_u)
        total_in_octets_interval = 6000000
        total_out_octets_interval = 3000000

        for n in range(links_len):
            sm_rx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].rx_link_rate
            sm_tx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].tx_link_rate
            peer_rx_linkrate = ne_u.peers[n].interfaces[0].radios[0].streams[0].links[0].rx_link_rate
            peer_tx_linkrate = ne_u.peers[n].interfaces[0].radios[0].streams[0].links[0].tx_link_rate

            sm_in_octets_interval = (n+1)*2000000
            sm_out_octets_interval = (n+1)*1000000
            r_err = ne_u.peers[n].errors[0]
            self.assertEqual(r_err.pb.WhichOneof('kind'), 'bad_data_range')
            self.assertEqual(ne_u.peers[n].status, network_poller_pb2.ELEMENT_STATUS_ERROR)
            if n==0:
                self.assertIsNone(sm_rx_linkrate)
                self.assertIsNone(peer_tx_linkrate)
                self.assertIsNotNone(peer_rx_linkrate)
                self.assertEqual(r_err.pb.bad_data_range.metric_name, "NetworkElement.WirelessLink.tx_link_rate")
                self.assertEqual(r_err.pb.bad_data_range.metric_value, "1010000000")

            else:
                self.assertIsNotNone(sm_rx_linkrate)
            if n==1:
                self.assertIsNone(sm_tx_linkrate)
                self.assertIsNone(peer_rx_linkrate)
                self.assertIsNotNone(peer_tx_linkrate)
                self.assertEqual(r_err.pb.bad_data_range.metric_name, "NetworkElement.WirelessLink.rx_link_rate")
                self.assertEqual(r_err.pb.bad_data_range.metric_value, "1020000000")
            else:
                self.assertIsNotNone(sm_tx_linkrate)
            aggr_rx_linkrate += sm_rx_linkrate * sm_in_octets_interval/total_in_octets_interval if sm_rx_linkrate and sm_in_octets_interval and total_in_octets_interval else 0
            aggr_tx_linkrate += sm_tx_linkrate * sm_out_octets_interval/total_out_octets_interval if sm_tx_linkrate and sm_out_octets_interval and total_out_octets_interval else 0

        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].rx_link_rate, 5), round(int(aggr_rx_linkrate),5))
        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].tx_link_rate, 5),  round(int(aggr_tx_linkrate),5))

    def test_linkrate_calc7(self):
        """ test discarding radio level link_rate value"""
        df = self.datafiles.get(('cambium-epmp.epmp.ePMP Force 180.4.4.2.01', 'snmp'))
        ne = self._await(self.reg.set('TestNE', {'name': 'Test Element', 'site': 'My Site', 'path': df.path}))
        ap = Ap(self.ctx, ne.id, ne.name, ne.site, ne.snmp_ne, ne.snmp_ne_v3, ne.ne_type, ne.holdoff)
        ap.in_out_bytes = {}
        aggr_tx_linkrate = 0

        self._await(ap.poll())
        self.wait_for_ap_poll()
        self._await(ap.log_ne_update())
        ne_u= ap.module.network_element_update_msg
        links_len = len(ne_u.data.interfaces[1].radios[0].streams[0].links)

        # set one of radio level link-rates above limit (so average calculation done only for TX while RX is discarded)
        ne_u.data.interfaces[1].radios[0].rx_link_rate = 1010000000
        ne_u.data.interfaces[1].radios[0].tx_link_rate = None

        ap.calc_radio_link_rate(ne_u)
        total_out_octets_interval = 3000000
        ap_err = ne_u.data.errors[0]
        self.assertEqual(ap_err.pb.WhichOneof('kind'), 'bad_data_range')
        self.assertEqual(ne_u.data.status, network_poller_pb2.ELEMENT_STATUS_ERROR)
        self.assertEqual(ap_err.pb.bad_data_range.metric_name, "NetworkElement.Radio.rx_link_rate")
        self.assertEqual(ap_err.pb.bad_data_range.metric_value, "1010000000")

        for n in range(links_len):
            sm_tx_linkrate = ne_u.data.interfaces[1].radios[0].streams[0].links[n].tx_link_rate
            sm_out_octets_interval = (n+1)*1000000
            self.assertIsNotNone(sm_tx_linkrate)
            aggr_tx_linkrate += sm_tx_linkrate * sm_out_octets_interval/total_out_octets_interval if sm_tx_linkrate and sm_out_octets_interval and total_out_octets_interval else 0
        self.assertIsNone(ne_u.data.interfaces[1].radios[0].rx_link_rate)
        self.assertEqual(round(ne_u.data.interfaces[1].radios[0].tx_link_rate, 5),  round(int(aggr_tx_linkrate),5))

    def test_service_port(self):
        """Test service port is created correctly"""
        self.ctx.company_uuid = UUID('4a24ad99-d502-3846-a8de-6c202c665a37')
        ap = Ap(self.ctx, 'TestNE', 'Test Element', 'site', None, None, 'ap', None)
        ap.module = FakeModule()
        msg = ap.module.network_element_update_msg = network_element_update.NetworkElementUpdate(
        )
        msg.data.system_mac_address = '01:02:03:04:05:0a'
        # msg.data.poller_hash = b'\x01'

        lan_intf = msg.data.interfaces.add()
        lan_intf.id = "1"
        lan_intf.name = "LAN"
        lan_intf.mac_address = '01:02:03:04:05:0a'

        radio_intf = msg.data.interfaces.add()
        radio_intf.id = "2"
        radio_intf.name = "WLAN"
        radio_intf.mac_address = '01:02:03:04:05:0b'
        radio = radio_intf.radios.add()
        radio.id = '1'
        stream = radio.streams.add()

        peer = msg.peers.add()
        # peer.poller_hash = b'\x02'
        peer.system_mac_address = '01:02:03:04:05:01'
        peer_radio_intf = peer.interfaces.add()
        peer_radio_intf.id = '1'
        peer_radio_intf.mac_address = b'01:02:03:04:05:02'
        p1link = peer_radio_intf.links.add()
        p1link.mac_address = radio_intf.mac_address
        p1r = peer_radio_intf.radios.add()
        p1s = p1r.streams.add()
        p1wlink = p1s.links.add()
        p1wlink.mac_address = radio_intf.mac_address

        peer2 = msg.peers.add()
        # peer2.poller_hash = b'\x03'
        peer2.system_mac_address = '11:02:03:04:05:01'
        peer2_radio_intf = peer2.interfaces.add()
        peer2_radio_intf.id = '2'
        peer2_radio_intf.mac_address = b'11:02:03:04:05:02'
        p2link = peer2_radio_intf.links.add()
        p2link.mac_address = radio_intf.mac_address
        p2r = peer2_radio_intf.radios.add()
        p2s = p2r.streams.add()
        p2wlink = p2s.links.add()
        p2wlink.mac_address = radio_intf.mac_address

        peer2_eth = peer2.interfaces.add()
        peer2_eth.id = '1'
        peer2_eth.name = 'Ethernet'
        peer2_eth.mac_address = '01:02:03:04:05:01'

        link1 = radio_intf.links.add()
        wlink1 = stream.links.add()
        link1.mac_address = peer_radio_intf.mac_address
        wlink1.mac_address = peer_radio_intf.mac_address

        link2 = radio_intf.links.add()
        wlink2 = stream.links.add()
        link2.mac_address = peer2_radio_intf.mac_address
        wlink2.mac_address = peer2_radio_intf.mac_address

        self._await(ap.load())

        for peer in msg.peers:
            if peer.system_mac_address == '01:02:03:04:05:01':
                has_service = True
            elif peer.system_mac_address == '11:02:03:04:05:01':
                has_service = False
            
            for p_intf in peer.interfaces:
                if p_intf.id == "service":
                    assert (has_service)
                    break
            else:
                assert (not has_service)
